----------------------------------------------------------------------------------
-- Company:        ETH Zurich, Institute for Particle Physics
-- Engineer:       Patrick Vogler
-- 
-- Create Date:    14 February 2010
-- Design Name:    
-- Module Name:    FTM Clock conditioner Interface
-- Project Name: 
-- Target Devices: 
-- Tool versions: 
-- Description:    Interface to the LMK03000 Clock conditioner
--
-- Dependencies: 
--
-- Revision: 
-- Revision 0.01 - File Created
-- Additional Comments: 
--
--
-- modifications:  February 21  2011 by Patrick Vogler
----------------------------------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

---- Uncomment the following library declaration if instantiating
---- any Xilinx primitives in this code.
--library UNISIM;
--use UNISIM.VComponents.all;

library ftm_definitions;
USE ftm_definitions.ftm_array_types.all;
USE ftm_definitions.ftm_constants.all;



entity Clock_cond_interface is
  port(

    
-- Clock
-------------------------------------------------------------------------------
   clk   : IN  STD_LOGIC;               -- 50 MHz system clock

   
-- Clock conditioner LMK03000
-------------------------------------------------------------------------------
   CLK_Clk_Cond  : out STD_LOGIC;  -- clock conditioner MICROWIRE interface clock
   LE_Clk_Cond   : out STD_LOGIC;  -- clock conditioner MICROWIRE interface latch enable   
   DATA_Clk_Cond : out STD_LOGIC;  -- clock conditioner MICROWIRE interface data
   
   SYNC_Clk_Cond : out STD_LOGIC;  -- clock conditioner global clock synchronization
   LD_Clk_Cond   : in STD_LOGIC;   -- clock conditioner lock detect                  

   
-- Time Marker
-------------------------------------------------------------------------------
   TIM_Sel    : out STD_LOGIC;   -- Time Marker selector 
                                 -- 1 = time marker from Clock conditioner
                                 --     for DRS timing calibration
                                 --
                                 -- 0 = time marker from FPGA for normal
                                 --     operation / physics run

   
   
-- FPGA intern clock conditioner configuration data
-------------------------------------------------------------------------------
   cc_R0             : in std_logic_vector (31 downto 0) := (others => '0');
   cc_R1             : in std_logic_vector (31 downto 0) := (others => '0');
   cc_R8             : in std_logic_vector (31 downto 0) := (others => '0');
   cc_R9             : in std_logic_vector (31 downto 0) := (others => '0');
   cc_R11            : in std_logic_vector (31 downto 0) := (others => '0');
   cc_R13            : in std_logic_vector (31 downto 0) := (others => '0');
   cc_R14            : in std_logic_vector (31 downto 0) := (others => '0');
   cc_R15            : in std_logic_vector (31 downto 0) := (others => '0');

  
 
   
-- FPGA intern control signals
-------------------------------------------------------------------------------
   start_config : in STD_LOGIC;        -- load new configuration into the clock
                                       -- conditioner
   
   config_started : out STD_LOGIC;     -- indicates that the new configuration
                                       -- is currently loaded into the clock conditioner

   config_done : out STD_LOGIC;        -- indicates that the configuration has
                                       -- been loaded and the clock conditioners
                                       -- PLL is locked

   timemarker_select: in STD_LOGIC     -- selects time marker source
                                       --
                                       -- 1 = time marker from Clock conditioner
                                       --     for DRS timing calibration
                                       --
                                       -- 0 = time marker from FPGA for normal
                                       --     operation / physics run
   
  );
end Clock_cond_interface;




architecture Behavioral of Clock_cond_interface is
  
 
component microwire_interface IS
   PORT(
      clk               : IN     std_logic;
      clk_uwire         : OUT    std_logic;  --- IN or OUT ?         
      data_uwire        : OUT    std_logic;       
      le_uwire          : OUT    std_logic;
      clk_cond_array    : IN     clk_cond_array_type;    
      config_start      : IN     std_logic;
      config_ready      : OUT    std_logic; 
      config_started    : OUT    std_logic      
   );
end component;



  
  signal clk_50M_sig : STD_LOGIC;       -- system clock
--  signal start_config_sig : STD_LOGIC;  

  signal config_ready_sig : STD_LOGIC;
  signal clk_uwire_sig : STD_LOGIC;

  signal config_started_sig : STD_LOGIC;
 
  signal clk_cond_array_sig : clk_cond_array_type;  

  signal cc_R0_sig             : std_logic_vector (31 downto 0); 
  signal cc_R1_sig             : std_logic_vector (31 downto 0); 
  signal cc_R8_sig             : std_logic_vector (31 downto 0); 
  signal cc_R9_sig             : std_logic_vector (31 downto 0); 
  signal cc_R11_sig            : std_logic_vector (31 downto 0); 
  signal cc_R13_sig            : std_logic_vector (31 downto 0); 
  signal cc_R14_sig            : std_logic_vector (31 downto 0); 
  signal cc_R15_sig            : std_logic_vector (31 downto 0); 
  
begin
  
  Inst_microwire_interface:microwire_interface
    port map (
        clk                 => clk_50M_sig,         
        clk_uwire           => clk_uwire_sig,  
        data_uwire          => DATA_Clk_Cond,
        le_uwire            => LE_Clk_Cond,        
        clk_cond_array      => clk_cond_array_sig,  
        
		  config_start        =>  start_config,
   --     config_start        => start_config_sig,  
	--		config_start        <= start_config_sig,
        
        config_ready        => config_ready_sig,
        config_started      => config_started_sig
       );

      
  config_done <= (config_ready_sig AND LD_Clk_Cond);  -- indicates that the configuration 
                                                      -- has been loaded and
                                                      -- the PLL is locked again
 
  TIM_Sel <= timemarker_select;                                   
  CLK_Clk_Cond <= clk_uwire_sig;

  clk_50M_sig <= clk;
 -- start_config_sig <= start_config;

 --  start_config <= start_config_sig;      


  config_started <= config_started_sig;

  cc_R0_sig  <= cc_R0;             
  cc_R1_sig  <= cc_R1;                 
  cc_R8_sig  <= cc_R8;       
  cc_R9_sig  <= cc_R9;      
  cc_R11_sig <= cc_R11;       
  cc_R13_sig <= cc_R13;          
  cc_R14_sig <= cc_R14;           
  cc_R15_sig <= cc_R15;

  clk_cond_array_sig(0) <= LMK03000_Reset;      -- reset LKM03000 by setting
                                                -- bit 31 of register 0
  clk_cond_array_sig(1) <= cc_R0_sig;
  clk_cond_array_sig(2) <= cc_R1_sig;  
  clk_cond_array_sig(3) <= cc_R8_sig; 
  clk_cond_array_sig(4) <= cc_R9_sig; 
  clk_cond_array_sig(5) <= cc_R11_sig;
  clk_cond_array_sig(6) <= cc_R13_sig;
  clk_cond_array_sig(7) <= cc_R14_sig;
  clk_cond_array_sig(8) <= cc_R15_sig;


end Behavioral;


