
/* ============================================================ 

Edd - Evidence Data Display

Qt-based graphical user interface for the Evidence contron system

EddLineDisplay changes its background colour in case it display
a DIM status service

April 2010, Oliver Grimm

============================================================ */

#include "GUI.h"

class EddDim *Handler;

//
// History chooser function (opens plot for numeric data, TextHist for all other)
//
QWidget *OpenHistory(char *Service, int Index) {

  QString Format;
  DimBrowser Browser;
  class EvidenceHistory *Hist = Handler->GetHistory(Service);

  // Check if history service available
  if (Hist == NULL || Hist->GetFormat() == NULL) {
	QMessageBox::warning(NULL, "Edd Message", QString("Could not retrieve history for service ") + Service ,QMessageBox::Ok);

	// If service currently available, take its format
	char *Name, *Fmt;

	Browser.getServices(Service);
	if (Browser.getNextService(Name, Fmt) != 0) Format = QString(Fmt);
	else {
	  Handler->DropHistory(Service);
	  return NULL;
	}
  }

  if (Format.isEmpty()) Format = Hist->GetFormat();
  Handler->DropHistory(Service);
  
  if (Format.size() == 1 && Format[0] != 'C') return new EddPlot(Service, Index);
  else return new EddText(Service);
}

//
// Set status tip (returns true if service was available)
//
bool SetStatus(QWidget *W, QString Name, int Time, QString Format, int Index) {

  QString Status;

  if (Index != -1) Name = Name + "(" + QString::number(Index) + ")";

  if (Time == -1) Status = QString("%1:  unavailable").arg(Name);
  else Status = QString("%1:  Last update %2   Format '%3'").arg(Name).arg(QDateTime::fromTime_t(Time).toString()).arg(Format); 

  W->setStatusTip(Status);
  
  return(Time != -1);
}


//////////////////////////////////////////
// Text display for arbitary DIM service//
//////////////////////////////////////////

EddLineDisplay::EddLineDisplay(QString Name, int Index, QWidget *P):
	QLineEdit(P), ServiceName(Name), Index(Index) {

  LastHist = NULL;
 
  // Widget properties
  setReadOnly(true);
  setMaximumWidth(100);
  ShowAsTime = false;
  setFrame(false);
  setAttribute(Qt::WA_DeleteOnClose);
  
  // Connect to DIM handler
  if (connect(Handler, SIGNAL(YEP(QString, int, QByteArray, QString, QString)), SLOT(Update(QString, int, QByteArray, QString, QString))) == false) {
    printf("Failed connection for %s\n", Name.toAscii().data());
  }

  // Context menu
  Menu = new QMenu(this);
  Menu->addAction("Open new history", this, SLOT(MenuOpenHistory()));
  Menu->addAction("Copy service", this, SLOT(MenuCopyService()));
  Menu->addAction("Copy data", this, SLOT(MenuCopyData()));

  // Subscribe to service
  Handler->Subscribe(Name);
}

// Destructor
EddLineDisplay::~EddLineDisplay() {

  Handler->Unsubscribe(ServiceName);
}

// Update widget
void EddLineDisplay::Update(QString Name, int Time, QByteArray, QString Format, QString Text) {

  if (ServiceName != Name) return;

  // Check if service available
  QPalette Pal = palette();  
  if (!SetStatus(this, Name, Time, Format, Index)) {
    setText("n/a");
    Pal.setColor(QPalette::Base, Qt::lightGray);
	setPalette(Pal);
	return;
  }
  else Pal.setColor(QPalette::Base, Qt::white);

  // Message service backgound colour determined by severity 
  if (Name.endsWith("/Message")) {
    switch (Text.section(' ', 0, 0).toInt()) {
      case 0:  Pal.setColor(QPalette::Base, Qt::white); break;
      case 1:  Pal.setColor(QPalette::Base, Qt::yellow); break;
      case 2:  Pal.setColor(QPalette::Base, Qt::red); break;
      case 3:  Pal.setColor(QPalette::Base, Qt::red); break;
      default: break;
    }
	Text = Text.section(' ', 1);
  }
  else if (Format[0].toUpper() != 'C' && Format != "I:1;C") Text = Text.section(' ', Index, Index);

  if (!ShowAsTime) setText(Text);
  else setText(QDateTime::fromTime_t(Text.toInt()).toString());

  setCursorPosition(0);  
  setPalette(Pal);
}

// Open plot if mouse release within widget
void EddLineDisplay::mouseReleaseEvent(QMouseEvent *Event) {

  if (Event->button()!=Qt::LeftButton || !contentsRect().contains(Event->pos())) return;

  // Check if last history plot still open, then raise
  foreach (QWidget *Widget, QApplication::allWidgets()) {
    if (Widget == LastHist) {
      Widget->activateWindow();
      Widget->raise();
      return;
    }
  }

  // If not, open new plot
  EddLineDisplay::MenuOpenHistory();
}

// Handling of mouse press event: Register start position for drag
void EddLineDisplay::mousePressEvent(QMouseEvent *Event) {

  if (Event->button() == Qt::LeftButton) dragStart = Event->pos();
}

// Handling of dragging (Drag and MimeData will be deleted by Qt)
void EddLineDisplay::mouseMoveEvent(QMouseEvent *Event) {

  if ((Event->buttons() & Qt::LeftButton) == 0) return;
  if ((Event->pos()-dragStart).manhattanLength() < QApplication::startDragDistance()) return;

  QDrag *Drag = new QDrag(this);
  QMimeData *MimeData = new QMimeData;
  QByteArray Data;
  MimeData->setData("Edd/Service", Data.append(ServiceName + " " + QString::number(Index)));
  Drag->setMimeData(MimeData);
  Drag->exec();
}

//
// Opening context menu
//
void EddLineDisplay::contextMenuEvent(QContextMenuEvent *Event) {

  Menu->exec(Event->globalPos());
}

// Menu: Open history plot
void EddLineDisplay::MenuOpenHistory() {
  
  LastHist = OpenHistory(ServiceName.toAscii().data(), Index);
  if (LastHist != NULL) LastHist->show();
}

// Menu: Copy service name
void EddLineDisplay::MenuCopyService() {
  
  QMimeData *MimeData = new QMimeData;
  QByteArray Data;
  MimeData->setData("Edd/Service", Data.append(ServiceName + " " + QString::number(Index)));
  QApplication::clipboard()->setMimeData(MimeData);
}

// Menu: Copy data
void EddLineDisplay::MenuCopyData() {

  QApplication::clipboard()->setText(text());
}


//////////////////////////////////////////
// Sending string command to DIM server //
//////////////////////////////////////////

EddCommand::EddCommand(QString Name, QWidget *P): QLineEdit(P), Name(Name) {

  setToolTip("Send command "+Name);  
  connect(this, SIGNAL(returnPressed()), SLOT(SendCommand()));
}

// Send command
void EddCommand::SendCommand() {

  DimClient::sendCommand(Name.toAscii().data(), text().toAscii().data());
  clear();
}


//////////////////////////////////
// History plot for DIM service //
//////////////////////////////////

EddPlot::EddPlot(QString Service, int Index, QWidget *P): EddBasePlot(P) {

  // Widget properties
  setAcceptDrops(true);
  setAxisScaleDraw(QwtPlot::xBottom, new EddTimeScale);
	
  // Update time range on plot when axis change	(update() results in paintEvent())
  connect(axisWidget(QwtPlot::xBottom), SIGNAL(scaleDivChanged()), SLOT(update()));

  legend()->setItemMode(QwtLegend::ClickableItem);
  connect(this, SIGNAL(legendClicked (QwtPlotItem *)), SLOT(LegendClicked(QwtPlotItem *)));
  
  // Connect to DIM handler
  if (connect(Handler, SIGNAL(YEP(QString, int, QByteArray, QString, QString)), SLOT(Update(QString, int, QByteArray, QString, QString))) == false) {
    printf("Failed connection for %s\n", Service.toAscii().data());
  }

  // Additonal context menu items
  QAction* Action = Menu->addAction("Paste service", this, SLOT(MenuPasteService()));
  Menu->removeAction(Action);
  Menu->insertAction(Menu->actions().value(1), Action);
  
  // Maximum number if points in curve (will be increased in Update())
  SizeLimit = 0;

  // DIM client
  if (!Service.isEmpty()) AddService(Service, Index);
}

// Destructor (items with parent widget are automatically deleted)
EddPlot::~EddPlot() {

  while (!List.isEmpty()) DeleteCurve(List.last().Signal);
}

// Add history service to plot
void EddPlot::AddService(QString Name, int Index) {

  // Check if curve already present on plot
  for (int i=0; i<List.size(); i++) {
    if (Name == List[i].Name && Index == List[i].Index) {
      QMessageBox::warning(this, "Edd Message",Name+" ("+QString::number(Index)+") already present",QMessageBox::Ok);
      return;
    }
  }  
  
  // Generate new curve and subscribe to service
  struct ItemDetails N;

  N.Name = Name;
  N.Signal = NewCurve(Name+"("+QString::number(Index)+")");
  N.Index = Index;
  List.append(N);

  Handler->Subscribe(Name);
}

// Update widget (must happen in GUI thread)
void EddPlot::Update(QString Name, int Time, QByteArray, QString Format, QString Text) {

  bool DoUpdate = false;

  // Determine which plot item this call belongs to
  int ItemNo;
  for (ItemNo=0; ItemNo<List.size(); ItemNo++) if (List[ItemNo].Name == Name) {

	DoUpdate = true;

	// If size limit reached, clear buffer
    if (List[ItemNo].Signal->dataSize() > SizeLimit) List[ItemNo].Signal->setData(QPolygonF());

	// If buffer empty, request new history buffer
    if (List[ItemNo].Signal->dataSize() == 0) {
	  int Count=0;
	  const struct EvidenceHistory::Item *R;
	  class EvidenceHistory *Hist;

	  if ((Hist = Handler->GetHistory(List[ItemNo].Name)) != NULL) {
		double Number=0;
		while ((R=Hist->Next()) != NULL) {
		  switch (*(Hist->GetFormat())) {
    		case 'I':
			case 'L':  Number = *((int *) R->Data + List[ItemNo].Index);   break;
    		case 'S':  Number = *((short *) R->Data + List[ItemNo].Index);   break;
    		case 'F':  Number = *((float *) R->Data + List[ItemNo].Index);   break;
    		case 'D':  Number = *((double *) R->Data + List[ItemNo].Index);   break;
    		case 'X':  Number = *((long long *) R->Data + List[ItemNo].Index);   break;
    		default: break;
		  }
		  AddPoint(ItemNo, R->Time, Number);
		  Count++;
		}

		// Local buffer at least twice as large as longest history
		if (SizeLimit < 2*Count) SizeLimit = 2*Count;
	  }
	  Handler->DropHistory(List[ItemNo].Name);
	}

 	// Append data only if service available
	if (SetStatus(this, Name, Time, Format)) {
	  QString Txt = Text;
	  Txt = Txt.section(' ', List[ItemNo].Index, List[ItemNo].Index);
      AddPoint(ItemNo, Time, atof(Txt.toAscii().data()));
	}
  }

  if (DoUpdate) UpdatePlot();
}


// Add text indicating time range to plot
void EddPlot::paintEvent(QPaintEvent *) {

  QString Text;
  QFont Font;
  QPainter Painter(this);
  
  Text = QDateTime::fromTime_t((int) axisScaleDiv(QwtPlot::xBottom)->lowerBound()).toString("d-MMM-yyyy hh:mm:ss") + " to " + QDateTime::fromTime_t((int) axisScaleDiv(QwtPlot::xBottom)->upperBound()).toString("d-MMM-yyyy hh:mm:ss");

  Font.setPointSize(6);  
  Painter.setFont(Font);
  Painter.drawText(0, height(), Text);
}

// Drag and drop methods
void EddPlot::dragEnterEvent(QDragEnterEvent *Event) {
    
  if (Event->mimeData()->hasFormat("Edd/Service")) Event->acceptProposedAction();
}

void EddPlot::dropEvent(QDropEvent *Event) {

  QByteArray D(Event->mimeData()->data("Edd/Service"));
  AddService(D.left(D.lastIndexOf(' ')), D.right(D.size()-D.lastIndexOf(' ')).toInt());
}

void EddPlot::LegendClicked(QwtPlotItem *Item) {

  QString D(Item->title().text());
  D.replace('(',' ').chop(1);
  
  QDrag *Drag = new QDrag(this);
  QMimeData *MimeData = new QMimeData;
  QByteArray Data;
  MimeData->setData("Edd/Service", Data.append(D));
  Drag->setMimeData(MimeData);
  Drag->exec();
}

// Add new service by pasting name
void EddPlot::MenuPasteService() {

  const QMimeData *D = QApplication::clipboard()->mimeData();
  if (!D->hasFormat("Edd/Service")) return;
  
  QByteArray E(D->data("Edd/Service"));
  AddService(E.left(E.lastIndexOf(' ')), E.right(E.size()-E.lastIndexOf(' ')).toInt());
}

// Remove list entry
void EddPlot::DeleteCurve(QwtPlotCurve *Curve) {

  for (int i=0; i<List.size(); i++) if (List[i].Signal == Curve) {
    Handler->Unsubscribe(List[i].Name);
    List.removeAt(i);
  }
}


//////////////////
// General plot //
//////////////////

// Constructor (all slots called in GUI thread, therefore no mutex necessary)
EddBasePlot::EddBasePlot(QWidget *P): QwtPlot(P) {

  // Widget properties
  setAttribute(Qt::WA_DeleteOnClose);
  setAutoReplot(false);
  setCanvasBackground(EddPlotBackgroundColor);
  setMargin(15);

  // Plot navigation
  Zoomer = new QwtPlotZoomer(QwtPlot::xBottom,QwtPlot::yLeft,canvas());
  connect(Zoomer, SIGNAL(zoomed(const QwtDoubleRect &)), this, SLOT(HandleZoom(const QwtDoubleRect &)));
  connect(Zoomer, SIGNAL(zoomed(const QwtDoubleRect &)), this, SLOT(ReDoStats()));

  Magnifier = new QwtPlotMagnifier(canvas());
  Magnifier->setMouseButton(Qt::NoButton,Qt::NoButton);
  Magnifier->setZoomInKey(Qt::Key_M, Qt::NoModifier);
  Magnifier->setZoomOutKey(Qt::Key_M, Qt::ShiftModifier);

  Panner = new QwtPlotPanner(canvas());
  Panner->setMouseButton(Qt::LeftButton, Qt::ShiftModifier);
  connect(Panner, SIGNAL(panned(int, int)), this, SLOT(ReDoStats()));

  Picker = new QwtPicker(QwtPicker::CornerToCorner|QwtPicker::RectSelection, QwtPicker::RectRubberBand, QwtPicker::AlwaysOff, this);
  connect(Picker, SIGNAL(selected(const QwtPolygon &)), SLOT(MouseSelection(const QwtPolygon &)));

  // Grid and legend
  Grid = new QwtPlotGrid;
  Grid->setMajPen(QPen(Qt::gray, 0, Qt::DotLine));
  Grid->attach(this);
  
  insertLegend(new QwtLegend(), QwtPlot::TopLegend, 0.3);

  // Marker for statistics text
  Stats = new QwtPlotMarker();
  Stats->setLineStyle(QwtPlotMarker::NoLine);
  Stats->setLabelAlignment(Qt::AlignLeft | Qt::AlignBottom);
  Stats->hide();
  Stats->attach(this);

  // Context menu
  Menu = new QMenu(this);
  StripAction = Menu->addAction("Stripchart", this, SLOT(UpdatePlot()));
  StripAction->setCheckable(true);
  Menu->addSeparator();
  YLogAction = Menu->addAction("y scale log", this, SLOT(UpdatePlot()));
  YLogAction->setCheckable(true);
  NormAction = Menu->addAction("Normalize", this, SLOT(UpdatePlot()));
  NormAction->setCheckable(true);
  StyleAction = Menu->addAction("Draw dots", this, SLOT(UpdatePlot()));
  StyleAction->setCheckable(true);
  StatisticsAction = Menu->addAction("Statistics", this, SLOT(ReDoStats()));
  StatisticsAction->setCheckable(true);
  Menu->addAction("Zoom out", this, SLOT(MenuZoomOut()));
  Menu->addAction("Single trace", this, SLOT(MenuSingleTrace()));
  Menu->addSeparator();
  Menu->addAction("Save as ASCII", this, SLOT(MenuSaveASCII()));
  Menu->addAction("Save plot", this, SLOT(MenuSave()));
  Menu->addAction("Print plot", this, SLOT(MenuPrint()));
  Menu->addAction("Plot help", this, SLOT(MenuPlotHelp()));
}

// Destructor (items with parent widget are automatically deleted)
EddBasePlot::~EddBasePlot() {

  for (int i=0; i<Items.size(); i++) delete Items[i].Signal;
  delete Grid;
}

// Print statistics to plot if requested
void EddBasePlot::ReDoStats() {

  QString Text;
  double Mean, Sigma;
  unsigned long Count = 0;

  // Set visibility of statistics box
  Stats->setVisible(StatisticsAction->isChecked());

  for (int i=0; i<Items.size(); i++) {
	Mean = 0;
	Sigma = 0;
	Count = 0;

    // Calculate mean and sigma for data points currently visible	
	for (int j=0; j<Items[i].y.size(); j++) {
	  if (!axisScaleDiv(QwtPlot::xBottom)->contains(Items[i].x[j])) continue;
	  Mean += Items[i].y[j];
	  Sigma += Items[i].y[j]*Items[i].y[j];
	  Count++;
	}
	
	if (Count == 0) Mean = 0;
	else Mean /= Count;
	if (Count < 2) Sigma = 0;
	else Sigma = sqrt(Count/(Count-1)*(Sigma/Count-Mean*Mean));
  
	// Prepare string
	Text += Items[i].Signal->title().text() + ": m " + QString::number(Mean, 'f', 2) + ", s " + QString::number(Sigma, 'f', 2) + "\n";
  }

  // Replot once to get axis correct
  replot();

  // Print string to plot
  QwtText text(Text);
  text.setFont(QFont("Helvetica", 8));
  Stats->setValue(axisScaleDiv(QwtPlot::xBottom)->upperBound(), axisScaleDiv(QwtPlot::yLeft)->upperBound());
  Stats->setLabel(text);

  // Replot again to update text
  replot();
}

// Update all curves in plot
void EddBasePlot::UpdatePlot() {

  double Lower = axisScaleDiv(QwtPlot::xBottom)->lowerBound();
  double Upper = axisScaleDiv(QwtPlot::xBottom)->upperBound();
  double MaxTime = DBL_MIN;
  static QwtSymbol Symbol, Sym1;
  Symbol.setStyle(QwtSymbol::Ellipse);
  Symbol.setSize(4);

  // Select engine for linear or logarithmic scale
  if (!YLogAction->isChecked()) {
    setAxisScaleEngine(QwtPlot::yLeft, new QwtLinearScaleEngine);
  }
  else setAxisScaleEngine(QwtPlot::yLeft, new QwtLog10ScaleEngine);

  for (int ItemNo=0; ItemNo<Items.size(); ItemNo++) {
	// Determine current maximum value for strip chart plotting
	if (Items[ItemNo].Signal->boundingRect().right() > MaxTime) {
	  MaxTime = Items[ItemNo].Signal->boundingRect().right();
	}

	// Set symbol if requested
    if (StyleAction->isChecked()) Items[ItemNo].Signal->setSymbol(Symbol);
    else Items[ItemNo].Signal->setSymbol(Sym1);

	// Determine number of data points
    int DataPoints = Items[ItemNo].x.size();	
	if (DataPoints == 0) continue;

    // Normalize y scale if requested
    double *y = new double [DataPoints];
    for (int i=0; i<DataPoints; i++) {
      y[i] = Items[ItemNo].y[i];

      if (NormAction->isChecked()) {
	    if (Items[ItemNo].Smallest != Items[ItemNo].Largest) {
		  y[i] = (y[i] - Items[ItemNo].Smallest)/(Items[ItemNo].Largest-Items[ItemNo].Smallest);
		}
	    else y[i] = 1;
      }  
    }
 
    // Plot data
    Items[ItemNo].Signal->setData(Items[ItemNo].x.data(), y, DataPoints);
    Items[ItemNo].Signal->show();
	delete[] y;

	// Generate bounding box of all curves for zoomer
	BBox = BBox.unite(Items[ItemNo].Signal->boundingRect());
  }

  // Reset zoom base to include all data
  Zoomer->setZoomBase(BBox);

  // If plot is strip char, only move axis but keep range
  if (StripAction->isChecked()) {
    setCanvasBackground(EddPlotBackgroundColor.lighter(90));
    setAxisScale(QwtPlot::xBottom, Lower+ BBox.right() - MaxTime, Upper + BBox.right() - MaxTime);
  }
  else setCanvasBackground(EddPlotBackgroundColor);

  ReDoStats();
}

// Append curve to plot
QwtPlotCurve *EddBasePlot::NewCurve(QwtText Title) {

  static Qt::GlobalColor LineColors[] = {Qt::black, Qt::blue, Qt::red, Qt::green, Qt::white,
   	Qt::darkRed, Qt::darkGreen, Qt::darkBlue, Qt::cyan, Qt::darkCyan, Qt::magenta, Qt::darkMagenta,
	Qt::gray, Qt::darkGray, Qt::lightGray};
  struct PlotItem N;

  N.Signal = new QwtPlotCurve;
  N.Signal->attach(this);
  N.Signal->setTitle(Title);
  N.Signal->setPen(QColor(LineColors[Items.size() % (sizeof(LineColors)/sizeof(Qt::GlobalColor))]));
  N.Largest = DBL_MIN;
  N.Smallest = DBL_MAX;  
  Items.append(N);
  
  return N.Signal;
}

// Clear curve data
void EddBasePlot::ClearCurve(unsigned int Item) {

  if (Item >= (unsigned int) Items.size()) return;
  
  Items[Item].x.clear();
  Items[Item].y.clear();
}

// Append data point
void EddBasePlot::AddPoint(unsigned int Item, double x, double y) {

  if (Item >= (unsigned int) Items.size()) return;
  
  Items[Item].x.append(x);
  Items[Item].y.append(y);

  if (y > Items[Item].Largest) Items[Item].Largest = y;
  if (y < Items[Item].Smallest) Items[Item].Smallest = y;    
}

// Rescale plot in case selection has been made outside the canvas
void EddBasePlot::MouseSelection(const QPolygon &P) {
 
  QwtDoubleInterval xPlot, xMouse, yPlot, yMouse; 

  // Shift selected rectangle so that upper left corner is 0/0 on canvas
  QRect R = P.boundingRect().translated(-plotLayout()->canvasRect().topLeft());

  // Current axis intervals
  xPlot = axisScaleDiv(QwtPlot::xBottom)->interval();
  yPlot = axisScaleDiv(QwtPlot::yLeft)->interval();
  
  // Selected axis intervals
  xMouse = QwtDoubleInterval(invTransform(QwtPlot::xBottom, R.left()),
  		invTransform(QwtPlot::xBottom, R.right()));
  yMouse = QwtDoubleInterval(invTransform(QwtPlot::yLeft, R.bottom()),
		invTransform(QwtPlot::yLeft, R.top()));

  // Selection region outside all axis?
  if (R.right() < 0 && R.top() > plotLayout()->canvasRect().height()) return;
  
  // Rescale both axis if selected rectangle encompasses canvas completely
  if (P.boundingRect().contains(plotLayout()->canvasRect())) {
	yMouse.setMaxValue(yMouse.maxValue() + yPlot.width());
	yMouse.setMinValue(yMouse.minValue() - yPlot.width());
	xMouse.setMinValue(xMouse.minValue() - xPlot.width());
	xMouse.setMaxValue(xMouse.maxValue() + xPlot.width());

	setAxisScale(QwtPlot::xBottom, xMouse.minValue(), xMouse.maxValue());
	setAxisScale(QwtPlot::yLeft, yMouse.minValue(), yMouse.maxValue());
  }

  // Rescale y axis (increase range if selected rectangle larger than axis area)
  if (R.right() < 0) {
	if (yMouse.maxValue() > axisScaleDiv(QwtPlot::yLeft)->upperBound()) {
	  yMouse.setMaxValue(yMouse.maxValue() + yPlot.width());
	}
	if (yMouse.minValue() < axisScaleDiv(QwtPlot::yLeft)->lowerBound()) {
	  yMouse.setMinValue(yMouse.minValue() - yPlot.width());
	}
	
	setAxisScale(QwtPlot::yLeft, yMouse.minValue(), yMouse.maxValue());
  }

  // Rescale x axis (increase range if selected rectangle larger than axis area)
  if (R.top() > plotLayout()->canvasRect().height()) {
	if (xMouse.maxValue() > axisScaleDiv(QwtPlot::xBottom)->upperBound()) {
	  xMouse.setMaxValue(xMouse.maxValue() + xPlot.width());
	}
	if (xMouse.minValue() < axisScaleDiv(QwtPlot::xBottom)->lowerBound()) {
	  xMouse.setMinValue(xMouse.minValue() - xPlot.width());
	}

	setAxisScale(QwtPlot::xBottom, xMouse.minValue(), xMouse.maxValue());
  }

  // Draw new scales
  replot();
} 

// Reset graph axes to autoscale when fully unzoomed
void EddBasePlot::HandleZoom(const QwtDoubleRect &) {

  if(Zoomer->zoomRectIndex() == 0) {
    setAxisAutoScale(QwtPlot::xBottom);
    setAxisAutoScale(QwtPlot::yLeft);
  }
  
  UpdatePlot();
}
    
// Opening context menu
void EddBasePlot::contextMenuEvent(QContextMenuEvent *Event) {

  Menu->exec(Event->globalPos());
}

// Zoom completely out
void EddBasePlot::MenuZoomOut() {

  Zoomer->zoom(0);
  UpdatePlot();
}

// Remove all items except one
void EddBasePlot::MenuSingleTrace() {

  while (Items.size() > 1) {  
	DeleteCurve(Items.last().Signal);
    delete Items.last().Signal;
    Items.takeLast();
  }
  UpdatePlot();
}

// Save data of plot as test 
void EddBasePlot::MenuSaveASCII() {
  QString Filename = QFileDialog::getSaveFileName(this,
     "Filename", ".", "Text files (*.txt *.ascii *.asc);;All files (*)");
  if (Filename.length() <= 0) return;
  
  QFile File(Filename);
  if (!File.open(QFile::WriteOnly | QIODevice::Text | QFile::Truncate)) {
    QMessageBox::warning(this, "Edd Message","Could not open file for writing.",QMessageBox::Ok);
    return;
  }

   // Write x and y data for all signals to file
  QTextStream Stream(&File);

  for (int ItemNo=0; ItemNo<Items.size(); ItemNo++) {
    Stream << QString("# ") + Items[ItemNo].Signal->title().text() + ".hist" << endl;
    for (int i=0; i<Items[ItemNo].Signal->dataSize(); i++) {
      Stream << Items[ItemNo].x.at(i) << " " << Items[ItemNo].Signal->y(i) << endl;
    }
  }
}

// Print plot
void EddBasePlot::MenuPrint() {

  QPrinter *Printer = new QPrinter;
  QPrintDialog *PrintDialog = new QPrintDialog(Printer, this);
  if (PrintDialog->exec() == QDialog::Accepted) {
    QPainter Painter(Printer);
    QPixmap Pixmap = QPixmap::grabWidget(this);
    Painter.drawPixmap(0, 0, Pixmap);
  }
  delete Printer;	delete PrintDialog;
}

// Save plot as image
void EddBasePlot::MenuSave() {

  QString Filename = QFileDialog::getSaveFileName(this,
     "Filename of image", "/home/ogrimm/ddd", "Image files (*.bmp *.jpg *.png *.ppm *.tiff *.xbm *.xpm);;All files (*)");
  if (Filename.length()>0) {
    QPixmap Pixmap = QPixmap::grabWidget(this);
    if(!Pixmap.save(Filename)) {
      QMessageBox::warning(this, "Edd Message","Could not write image file.",QMessageBox::Ok);
      remove(Filename.toAscii().data());
    }
  }
}

// Help text
void EddBasePlot::MenuPlotHelp() {

  QMessageBox::about(this, "Edd - Plot navigation",
    "Zoom\tMouse wheel\n"
	"\tKeys m and shift-m\n"
	"\tSelecting region with left mouse button\n"
	"\tMiddle button zooms out one level\n"
	"\tSelecting a range on an axis\n"
	"\tSelecting whole canvas\n\n"
    "Pan\tShift and left mouse button\n\n"
	"ESC cancels selection\n"
    "Cursor keys move mouse\n\n"
	"Statistics are calculated over the current x axis extend");
}

//////////////////////////////////////
// History text box for DIM service //
//////////////////////////////////////

// Constructor
EddText::EddText(QString Name, bool Pure, QWidget *P):
	QTextEdit(P), Name(Name), Pure(Pure) {
  
  // Widget properties
  setReadOnly(true);
  setAttribute(Qt::WA_DeleteOnClose);
  setAutoFillBackground(true);
  document()->setMaximumBlockCount(1000);
  Accumulate = true;
  
  // Connect to DIM handler
  if (connect(Handler, SIGNAL(YEP(QString, int, QByteArray, QString, QString)), SLOT(Update(QString, int, QByteArray, QString, QString))) == false) {
    printf("Failed connection for %s\n", Name.toAscii().data());
  }

  if (!Pure) {
	// Get history for this service
	const struct EvidenceHistory::Item *R;
	class EvidenceHistory *Hist;

	if ((Hist = Handler->GetHistory(Name)) != NULL) {
	  while ((R=Hist->Next()) != NULL) {
		moveCursor (QTextCursor::Start);
		insertPlainText(QString("(")+QDateTime::fromTime_t(R->Time).toString()+") " +
		  QString::fromStdString(EvidenceServer::ToString(Hist->GetFormat(), (void *) R->Data, R->Size)) + "\n"); 
	  }	
	}
	Handler->DropHistory(Name);
  }

  // DIM client
  Handler->Subscribe(Name);
}

// Destructor
EddText::~EddText() {

  Handler->Unsubscribe(Name);
}


// Update widget (must happen in GUI thread)
void EddText::Update(QString Name, int Time, QByteArray, QString Format, QString Text) {

  if (this->Name != Name) return;
  QPalette Pal = palette();  

  // Check if service available
  if (!SetStatus(this, Name, Time, Format)) {
    Pal.setColor(QPalette::Base, Qt::lightGray);
	setPalette(Pal);
	return;
  }

  Pal.setColor(QPalette::Base, Qt::white);
  setPalette(Pal);
  QDateTime Timex = QDateTime::fromTime_t(Time); 

  // Clear display in case text should not accumulate
  if (Accumulate == false) clear();

  if (!Pure) {
    moveCursor(QTextCursor::Start);
    insertPlainText(QString("(")+Timex.toString()+QString(") "));	  
    insertPlainText(Text + "\n");	  
  }
  else 	if (Format == "C") insertPlainText(Text);
}


/////////////////////////////
// Interface to Dim system //
/////////////////////////////
EddDim::EddDim() {

  Mutex = new QMutex(QMutex::Recursive);
  Volume = 0;

  // Timer to calculate data rates
  QTimer *Timer = new QTimer(this);
  Timer->connect(Timer, SIGNAL(timeout()), this, SLOT(UpdateStatistics()));
  Timer->start(10000);

  // Connect to DIM handler
  if (connect(this, SIGNAL(INT(QString, int, QByteArray, QString, QString)), SLOT(Update(QString, int, QByteArray, QString, QString))) == false) {
    printf("Failed connection in EddDim()\n");
  }
}

// Destructor
EddDim::~EddDim() {

  QList<QString> L = HistoryList.keys();
  for(int i=0; i<L.size(); i++) delete HistoryList[L[i]].HistClass;

  delete Mutex;
}

// Subscribe to DIM service
void EddDim::Subscribe(QString Name) {

  // Lock before accessing list
  QMutexLocker Locker(Mutex);

  // If already subscribed to service, increase usage count and reemit data for new subscriber
  if (ServiceList.contains(Name)) {
	ServiceList[Name].Count++;
	YEP(Name, ServiceList[Name].TimeStamp, ServiceList[Name].ByteArray, ServiceList[Name].Format, ServiceList[Name].Text);
	return;
  }

  // Create new entry in service list
  ServiceList[Name].ByteArray = QByteArray();
  ServiceList[Name].TimeStamp = -1;
  ServiceList[Name].Count = 1;
  ServiceList[Name].DIMService = new DimStampedInfo(Name.toAscii().data(), INT_MAX, NO_LINK, this);
  return;
}

// Unsubsribe from DIM service
void EddDim::Unsubscribe(QString Name) {

  // Lock before accessing list
  QMutexLocker Locker(Mutex);

  if (ServiceList.contains(Name)) ServiceList[Name].Count--;

  if (ServiceList[Name].Count == 0) {
	delete ServiceList[Name].DIMService;
	ServiceList.remove(Name);
	return;
  }
}

// Get history buffer
class EvidenceHistory *EddDim::GetHistory(QString Name) {

  // History already available (only request again if too old)
  if (HistoryList.contains(Name)) {
	HistoryList[Name].Count++;

	if (time(NULL)-HistoryList[Name].LastUpdate < 5) {
	  HistoryList[Name].HistClass->Rewind();
	  return HistoryList[Name].HistClass;
	}
	HistoryList[Name].LastUpdate = time(NULL);
	if (HistoryList[Name].HistClass->GetHistory()) return HistoryList[Name].HistClass;
	else return NULL;
  }

  // Create new history class
  HistoryList[Name].HistClass = new EvidenceHistory(Name.toStdString());
  HistoryList[Name].Count = 1;
  HistoryList[Name].LastUpdate = time(NULL);

  if (HistoryList[Name].HistClass->GetHistory()) return HistoryList[Name].HistClass;
  else return NULL;
}

// Reduce history usage counter
void EddDim::DropHistory(QString Name) {

  if (HistoryList.contains(Name)) HistoryList[Name].Count--;
}

// Update throughput statistics and clear up history memory
void EddDim::UpdateStatistics() {

  // Lock before accessing internal variables
  QMutexLocker Locker(Mutex);

  // Remove unused histories after not less than 5 seconds
  QList<QString> L = HistoryList.keys();
  for(int i=0; i<L.size(); i++) {
	if ((HistoryList[L[i]].Count <= 0) && (time(NULL)-HistoryList[L[i]].LastUpdate) > 5) { 
	  delete HistoryList[L[i]].HistClass;
	  HistoryList.remove(L[i]);
	}
  }
  
  float Rate = Volume/1024.0/10;
  Volume = 0;
  YEP("Edd/Rate_kBSec", time(NULL), QByteArray::number(Rate), "F", QString::number(Rate));
}

// Store service information for usage by Subscribe(), update statistics and emit signal to widgets
void EddDim::Update(QString Name, int Time, QByteArray Data, QString Format, QString Text) {

  // Lock before accessing list
  QMutexLocker Locker(Mutex);

  // Store service data
  if (ServiceList.contains(Name)) {
	  ServiceList[Name].TimeStamp = Time;
	  ServiceList[Name].ByteArray = Data;
	  ServiceList[Name].Format = Format;
	  ServiceList[Name].Text = Text;
  }

  // Update statistics only for actual Dim services
  if (!Name.startsWith("Edd/")) Volume += Data.size();
  
  // Emit signal to all widgets
  YEP(Name, Time, Data, Format, Text);
}

// Handling of DIM service update
// No locking allowed. Signal triggers only EddDim::Update() when the main event loop is idle.
void EddDim::infoHandler() {

  if (!EvidenceServer::ServiceOK(getInfo())) INT(getInfo()->getName(), -1);
  else {
	INT(getInfo()->getName(), getInfo()->getTimestamp(), QByteArray((char *) getInfo()->getData(), 
		getInfo()->getSize()), getInfo()->getFormat(), QString::fromStdString(EvidenceServer::ToString(getInfo()->getFormat(), getInfo()->getData(), getInfo()->getSize())));
  }
}


/////////////////////
// Open new window //
/////////////////////

// Constructor
EddWindow::EddWindow(QString ButtonName, QString WindowName): QPushButton(ButtonName) {
  
  M = new QMainWindow;
  M->setCentralWidget(new QWidget);
  M->setStatusBar(new QStatusBar(M));
  M->setWindowTitle(WindowName);

  L = new QGridLayout(M->centralWidget());

  connect(this, SIGNAL(pressed()), SLOT(Toggle()));
}

// Show or hide window
void EddWindow::Toggle() {

  if (M->isVisible()) M->hide();
  else M->show();
}

// Return layout
QGridLayout *EddWindow::Layout() {

  return L;
}

// Return window
QMainWindow *EddWindow::Window() {

  return M;
}
