#ifndef GUI_H_SEEN
#define GUI_H_SEEN

#include <QtGui>
 
#include <qwt_plot.h>
#include <qwt_plot_curve.h>
#include <qwt_plot_grid.h>
#include <qwt_plot_zoomer.h>
#include <qwt_plot_magnifier.h>
#include <qwt_plot_panner.h>
#include <qwt_scale_engine.h>
#include <qwt_analog_clock.h>
#include <qwt_scale_widget.h>
#include <qwt_plot_layout.h>
#include <qwt_legend.h>
#include <qwt_legend_item.h>
#include <qwt_symbol.h>
#include <qwt_plot_marker.h>
#include <qwt_data.h>
#include <qwt_color_map.h>

#include <limits.h>
#include <float.h>

#include "dic.hxx"
#include "Evidence.h"

const QColor EddPlotBackgroundColor(Qt::yellow);

QWidget *OpenHistory(char *, int);
bool SetStatus(QWidget *, QString, int, QString, int = -1);


// Base class for Edd plot
// DeleteCurve() is pure virtual and needs to be implemented iin the application class
class EddBasePlot: public QwtPlot {
  Q_OBJECT

  protected:    
    QMenu *Menu;
    QAction *StripAction;

  private:
    struct PlotItem {
      QwtPlotCurve *Signal;
	  QVector<double> x;
	  QVector<double> y;
      double Smallest;
      double Largest;
	  double Mean;
	  double Sigma;
    };
    QList<struct PlotItem> Items;

    QAction *YLogAction;
    QAction *NormAction;
    QAction *StyleAction;
    QAction *StatisticsAction;

    QwtPlotPanner *Panner;
    QwtPlotGrid *Grid;
    QwtPlotZoomer *Zoomer;
	QwtPlotMagnifier *Magnifier;
    QwtPicker *Picker;
	QwtDoubleRect BBox;
	QwtPlotMarker *Stats;
	 	
  public:
    EddBasePlot(QWidget * = NULL);
    ~EddBasePlot();

	QwtPlotCurve *NewCurve(QwtText);
	void ClearCurve(unsigned int);
	void AddPoint(unsigned int, double, double);
	virtual void DeleteCurve(QwtPlotCurve *) = 0;

  protected slots:
    void UpdatePlot();

  private slots:
	void ReDoStats();
    void HandleZoom(const QwtDoubleRect &);
	void MouseSelection(const QwtPolygon &);
	void contextMenuEvent(QContextMenuEvent *);
    void MenuSingleTrace();        
    void MenuZoomOut();
    void MenuSaveASCII();
    void MenuSave();
    void MenuPrint();
	void MenuPlotHelp();
};
		  
// General indicator for DIM service
class EddLineDisplay: public QLineEdit {
    Q_OBJECT

    QMenu *Menu;
    QPoint dragStart;
    QWidget *LastHist;
	
	QString ServiceName;
	int Index;
	
    void mousePressEvent(QMouseEvent *); 
    void mouseReleaseEvent(QMouseEvent *); 
    void mouseMoveEvent(QMouseEvent *); 
	
  public:
    EddLineDisplay(QString, int=0, QWidget * = NULL);
    ~EddLineDisplay();

	bool ShowAsTime;
	
  private slots:
	void Update(QString, int, QByteArray, QString, QString);
    void contextMenuEvent(QContextMenuEvent *);    
    void MenuOpenHistory();
    void MenuCopyService();
    void MenuCopyData();
};

// Sending command to DIM server
class EddCommand: public QLineEdit {
    Q_OBJECT

	QString Name;
	
  public:
    EddCommand(QString, QWidget * = NULL);

  private slots:
	void SendCommand();	
};

// Graph class for history display 
class EddPlot: public EddBasePlot {
    Q_OBJECT

	// Time scale for axis
	class EddTimeScale: public QwtScaleDraw {

	  public:
		EddTimeScale() {}

		virtual QwtText label(double v) const {
		  // Adapt text format to time span 
		  QString Format;
		  if (scaleDiv().range() < 60*60) Format = "hh' h\n'mm:ss";
		  else if (scaleDiv().range() < 24*60*60) Format = "hh:mm";
		  else if (scaleDiv().range() < 30*24*60*60) Format = "h' h\n'd-MMM";
		  else Format = "d-MMM'\n'yyyy";

		  // Generate text
		  QwtText Text = QDateTime::fromTime_t((int) v).toString(Format);
		  QFont Font = Text.font();
		  Font.setPointSize(7);
		  Text.setFont(Font);

    	  return Text;
		}
	};

    struct ItemDetails {
      QString Name;
	  int Index;
      QwtPlotCurve *Signal;	  
    };
    QList<struct ItemDetails> List;

  private:
	QwtLegend *Legend;
	int SizeLimit;

    void dragEnterEvent(QDragEnterEvent *);
    void dropEvent(QDropEvent *);
	void paintEvent(QPaintEvent *);
	 	
  public:
    EddPlot(QString = QString(), int = 0, QWidget * = NULL);
    ~EddPlot();
    void AddService(QString, int = 0);
	void DeleteCurve(QwtPlotCurve *);

  private slots:
	void Update(QString, int, QByteArray, QString, QString);
	void LegendClicked(QwtPlotItem *);
    void MenuPasteService();
};


// Text history and output class
class EddText: public QTextEdit {
  Q_OBJECT

  private:
	QString Name;
	bool Pure;
	
  public:
    EddText(QString, bool = false, QWidget * = NULL);
    ~EddText();

	bool Accumulate;

  private slots:
	void Update(QString, int, QByteArray, QString, QString);
};


// Interface to DIM system
class EddDim: public QObject, public DimInfo {
  Q_OBJECT

  private:
	struct Item {
	  DimStampedInfo *DIMService;
	  int Count;
	  int TimeStamp;
	  QByteArray ByteArray;
	  QString Format;
	  QString Text;
	};
    QMap<QString, struct Item> ServiceList;
    QMutex *Mutex;

	struct HistItem {
	  int Count;
	  int LastUpdate;
	  class EvidenceHistory *HistClass;
	};
    QMap<QString, struct HistItem> HistoryList;

	long long Volume;

	void infoHandler();

  private slots:
	void Update(QString, int, QByteArray, QString, QString);
	void UpdateStatistics();

  public:
    EddDim();
    ~EddDim();

	void Subscribe(QString);
	void Unsubscribe (QString);
	class EvidenceHistory *GetHistory(QString);
	void DropHistory(QString);

  signals:
    void YEP(QString, int, QByteArray = QByteArray(), QString = QString(), QString = QString());
    void INT(QString, int, QByteArray = QByteArray(), QString = QString(), QString = QString());
};

// Open new window
class EddWindow: public QPushButton {
  Q_OBJECT

  QMainWindow *M;
  QGridLayout *L;

  public:
    EddWindow(QString, QString);
	QGridLayout *Layout();
	QMainWindow *Window();
  private slots:
	void Toggle();
};

#endif
