#ifndef FACT_Converter
#define FACT_Converter

#include <vector>
#include <ostream>

#include <boost/any.hpp>

class Converter
{
public:
    typedef std::pair<const std::type_info *, int> Type;
    typedef std::pair<int, int>                    Offset;
    typedef std::pair<Type, Offset>                Format;
    typedef std::vector<Format>                    FormatList;

    struct O { };
    struct W { };

    static std::string Clean(std::string s);

private:
    std::ostream &wout;        /// ostream to which output is redirected

    const std::string fFormat; /// Original format string
    const FormatList  fList;   /// Compiled format description

    template <class T>
        T Get(std::stringstream &line) const;

    bool        GetBool(std::stringstream &line) const;
    std::string GetString(std::stringstream &line) const;
    std::string GetStringEol(std::stringstream &line) const;

    template<class T>
        void GetBinImp(std::vector<char> &v, const T &val) const;
    template<class T>
        void GetBinImp(std::vector<boost::any> &v, const T &val) const;

    void GetBinString(std::vector<char> &v, const std::string &val) const;
    void GetBinString(std::vector<boost::any> &v, const std::string &val) const;

    template<class T>
        std::string GetString(const char *&data) const;

    template<class T>
        static Type GetType();
    template<class T>
        static Type GetVoid();

    template <class T>
        std::vector<T> Get(const std::string &str) const;
    template <class T>
        T Get(const void *d, size_t size) const;



    template<class T>
        void Add(std::string &str, const char* &ptr) const;
    void AddString(std::string &str, const char* &ptr) const;
    template<class T>
        void Add(std::vector<boost::any> &vec, const char* &ptr) const;
    void AddString(std::vector<boost::any> &vec, const char* &ptr) const;


public:
    Converter(std::ostream &out, const std::string &fmt, bool strict=true);
    Converter(const std::string &fmt, bool strict=true);

    /// @returns whether the interpreted format was valid but empty ("")
    bool empty() const { return fList.size()==1 && fList.back().first.second==0; }

    /// @returns whether the compilation was successfull
    bool valid() const { return !fList.empty() && fList.back().first.second==0; }

    /// @returns true if the compilation failed
    bool operator!() const { return !valid(); }

    const FormatList &GetList() const { return fList; }

    static FormatList Compile(std::ostream &out, const std::string &fmt, bool strict=false);
    static FormatList Compile(const std::string &fmt, bool strict=false);

    std::string             GetString(const void *d, size_t size) const;
    std::vector<char>       GetVector(const void *d, size_t size) const;
    std::vector<boost::any> GetAny(const void *d, size_t size) const;

    std::vector<boost::any> GetAny(const std::string &str) const;
    std::vector<char>       GetVector(const std::string &str) const;

    static std::string GetHex(const void *d, size_t size, size_t chunk=1);
    
    void ToFits(void* dest, const void* src, size_t size);

/*
    // Returns the number of entries in Converter:vec
    int N() const { return vec.size(); }

    /// Checks if an entry with the given class type is available at index i in Converter::vec
    template <class T>
        bool Has(unsigned int i) const { return i<vec.size() && vec[i].type()==typeid(T); }

    /// return the entry with the given class type at index i from Converter::vec (exception safe)
    template <class T>
        T Get(int i) const { return !Has<T>(i) ? T() : boost::any_cast<T>(vec[i]); }

    /// return the entry with the given class type at index i from Converter::vec (throws exceptions)
    template <class T>
        T At(int i) const { return boost::any_cast<T>(vec[i]); }
*/

    static std::vector<std::string> Regex(const std::string &expr, const std::string &line);
};

#endif
