/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
// MReadMarsFile                                                           //
//                                                                         //
// This tasks opens all branches in a specified tree and creates the       //
// corresponding parameter containers if not already existing in the       //
// parameter list.                                                         //
//                                                                         //
// The Process function reads one events from the tree. To go through the  //
// events of one tree make sure that the event number is increased from    //
// outside. It makes also possible to go back by decreasing the number.    //
//                                                                         //
// If you don't want to start reading the first event you have to call     //
// MReadMarsFile::SetEventNum after instantiating your                     //
// MReadMarsFile-object.                                                   //
//                                                                         //
// To make reading much faster (up to a factor of 10 to 20) you can        //
// ensure that only the data you are really processing is enabled by       //
// calling MReadMarsFile::UseLeaf.                                         //
//                                                                         //
// FIXME: An automatic enabeling scheme would be nice.                     //
//        Can we use TBranch::SetAutoDelete?                               //
//                                                                         //
// Later we'll use TChain::SetNotify to notify MReadMarsFile if the TChain //
// starts to read a new file.                                              //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MReadMarsFile.h"

#include "MLog.h"

ClassImp(MReadMarsFile);

// --------------------------------------------------------------------------
//
//  Default constructor. It creates an TChain instance which represents the
//  the Tree you want to read and adds the given file (if you gave one).
//  More files can be added using MReadMarsFile::AddFile.
//  Also an empty veto list is created. This list is used if you want to
//  veto (disable or "don't enable") a branch in the tree.
//
MReadMarsFile::MReadMarsFile(const char *tname, const char *fname,
                             const char *name, const char *title)
    : MReadTree(tname, fname)
{
    fName  = name  ? name  : "MReadMarsFile";
    fTitle = title ? title : "Task to loop over all events in a tree of a Mars file.";

    //
    // open the input stream
    //
    fRun = new MReadTree("RunHeaders", fname);
    fRun->DisableAutoScheme();
}

// --------------------------------------------------------------------------
//
// Destructor. It deletes the TChain and veto list object
//
MReadMarsFile::~MReadMarsFile()
{
    delete fRun;
}

// --------------------------------------------------------------------------
//
//  If you want to read the given tree over several files you must add
//  the files here before PreProcess is called. Be careful: If the tree
//  doesn't have the same contents (branches) it may confuse your
//  program (trees which are are not existing in later files are not read
//  anymore, tree wich are not existing in the first file are never read)
//
//  Name may use the wildcarding notation, eg "xxx*.root" means all files
//  starting with xxx in the current file system directory.
//
Int_t MReadMarsFile::AddFile(const char *fname)
{
    //
    // FIXME! A check is missing whether the file already exists or not.
    //
    //
    // returns the number of file which were added
    //
    Int_t n1 = fRun->AddFile(fname);
    Int_t n2 = MReadTree::AddFile(fname);

    return n1 != n2 ? -1 : n1;
}

Bool_t MReadMarsFile::Notify()
{
    if (GetEventNum() == 0)
        return kTRUE;

    *fLog << "MReadMarsFile: Switching to next file '" << GetFileName() << "' ";
    *fLog << "(before Event #" << GetEventNum()-1 << ")" << endl;
    fRun->Process();

    MReadTree::Notify();

    return kTRUE;
}

// --------------------------------------------------------------------------
//
//  The PreProcess loops (till now) over the branches in the given tree.
//  It checks if the corresponding containers (containers with the same
//  name than the branch name) are existing in the Parameter Container List.
//  If not, a container of objec type 'branch-name' is created (everything
//  after the last semicolon in the branch name is stripped). Only
//  branches which don't have a veto (see VetoBranch) are enabled If the
//  object isn't found in the root dictionary (a list of classes known by the
//  root environment) the branch is skipped and an error message is printed
//  out.
//
Bool_t MReadMarsFile::PreProcess(MParList *pList)
{
    if (!fRun->PreProcess(pList))
        return kFALSE;

    return MReadTree::PreProcess(pList);
}

