----------------------------------------------------------------------------------
-- Company:        ETH Zurich, Institute for Particle Physics
-- Engineer:       Q. Weitzel
-- 
-- Create Date:    February 2011 
-- Design Name:    
-- Module Name:    ftm_definitions 
-- Project Name: 
-- Target Devices: 
-- Tool versions: 
-- Description:    library file for FTM design
--
-- Dependencies: 
--
-- Revision: 
-- Revision 0.01 - File Created
-- Additional Comments: 
--
-- modified:   Patrick Vogler, February 17 2011
-- merged with library file from Dortmund, Q. Weitzel, February 24, 2011
--
-- kw 25.02.:  changes for HDL-Designer (use FACT_FTM.lib.ftm_...),
-- DD_BLOCK_SIZE added (set to 0x008 for first tests), SD_ADDR_coin_win_[p,c] defined
--
-- modified:   Quirin Weitzel, March 14 2011
-- second merger with library file from dortmund (changes below)
-- kw 01.03.: added array sd_block_default_ftu_active_list (type and defaults)
-- kw 02.03.: added DD_BLOCK_SIZE_GENERAL (on-time counter + temperatures), changed DD_BLOCK_SIZE to 0x010 for testing
-- kw 03.03.: added FTM_HEADER_LENGTH
--
-- modified:   Quirin Weitzel, March 14 2011
-- third merger with library file from dortmund (changes below)
-- kw 22.03.: added FTU_ERROR_LENGTH
-- kw 30.03.: added CMD_AUTOSEND, PAR_AUTOSEND_EA, PAR_AUTOSEND_DA
--
---kw 11.04.: added SD_ADDR_ftu_prescaling_0
--
----------------------------------------------------------------------------------


library IEEE;
use IEEE.STD_LOGIC_1164.all;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
use IEEE.NUMERIC_STD.ALL;

package ftm_array_types is

  -- !!! some arrays are also defined in the ftm_constants package !!!
  
  -- data arrays for a single FTU
  type FTU_enable_array_type is array (0 to 3) of std_logic_vector(15 downto 0);
  type FTU_dac_array_type    is array (0 to 4) of std_logic_vector(15 downto 0);
  type FTU_rate_array_type   is array (0 to 4) of std_logic_vector(31 downto 0);
  type active_FTU_array_type is array (0 to 3) of std_logic_vector(15 downto 0);
  type FTU_answer_array_type is array (0 to 3) of integer range 0 to 10;

  -- data array for clock conditioner interface
  type clk_cond_array_type is array (0 to 14) of std_logic_vector (31 downto 0);

  -- network array types
  type ip_type  is array (0 to 3) of integer;
  type mac_type is array (0 to 2) of std_logic_vector (15 downto 0);

  -- Temperature Sensor interface
  type sensor_array_type is array (0 to 3) of integer range 0 to 2**16 - 1;
  
end ftm_array_types;


library IEEE;
use IEEE.STD_LOGIC_1164.all;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
-- for HDL-Designer
-- LIBRARY FACT_FTM_lib;
-- use FACT_FTM_lib.ftm_array_types.all;
library ftm_definitions;
use ftm_definitions.ftm_array_types.all;
use IEEE.NUMERIC_STD.ALL;
use ieee.math_real.all;

package ftm_constants is

  -- !!! many constants depend on each other or are defined 2x with different types !!!
  
  constant NO_OF_CRATES         : integer :=  4;
  constant NO_OF_FTUS_PER_CRATE : integer := 10;
  constant NO_OF_FTU_ENABLE_REG : integer :=  4;
  constant NO_OF_FTU_DAC_REG    : integer :=  5;
  
  --internal FPGA clock frequencies
  constant INT_CLK_FREQUENCY_1 : integer :=  50000000;  --  50MHz
  constant INT_CLK_FREQUENCY_2 : integer := 250000000;  -- 250MHz
  constant LOW_FREQUENCY       : integer :=   1000000;  -- has to be smaller than INT_CLK_FREQUENCY_1
  --constant SCALER_FREQ_DIVIDER : integer :=     10000;  -- for simulation, should normally be 1
  constant SCALER_FREQ_DIVIDER : integer :=         1;
  
  --FTM address and firmware ID
  constant FTM_ADDRESS : std_logic_vector(7 downto 0) := "11000000";  -- 192
  constant FIRMWARE_ID : std_logic_vector(7 downto 0) := "00000001";  -- firmware version
  
  --communication with FTUs
  constant FTU_RS485_BAUD_RATE   : integer := 250000;  -- bits / sec in our case
  constant FTU_RS485_TIMEOUT     : integer := (INT_CLK_FREQUENCY_1 * 2) / 1000;  -- 2ms @ 50MHz (100000 clk periods)
  --constant FTU_RS485_BAUD_RATE   : integer := 10000000;  -- for simulation
  --constant FTU_RS485_TIMEOUT     : integer := (INT_CLK_FREQUENCY_1 * 2) / 40000;  -- for simulation
  constant FTU_RS485_NO_OF_RETRY : integer range 0 to 2 := 2;  -- in case of timeout, !!! HAS TO BE < 3 !!!
  constant FTU_RS485_BLOCK_WIDTH : integer := 224;  -- 28 byte protocol
  constant FTU_RS485_START_DELIM : std_logic_vector(7 downto 0) := "01000000";  -- start delimiter "@"
  
  --CRC setup
  constant CRC_POLYNOMIAL : std_logic_vector(7 downto 0) := "00000111";  -- 8-CCITT
  constant CRC_INIT_VALUE : std_logic_vector(7 downto 0) := "11111111";
  
  --DNA identifier for simulation
  constant DNA_FOR_SIM : bit_vector := X"01710000E0000501";

  -- Clock conditioner (LMK03000, National semiconductor) interface 
  constant MICROWIRE_CLK_FREQUENCY : integer := 2000000;  -- 2 MHz
  -- value to be written to R0 of the LMK03000 to perform a reset, see LMK03000 datasheet
  constant LMK03000_Reset          : std_logic_vector (31 downto 0) :=  x"80000000";
  constant LMK03000_REGISTER_WIDTH : integer := 32;
  constant LMK03000_REGISTER_COUNT : integer := 15;  -- number of registers to be programmed in the LMK03000 including reset
  constant cc_R2_const              : std_logic_vector := X"00000102";  -- unused                                                                        
  constant cc_R3_const              : std_logic_vector := X"00000103";  -- channels
  constant cc_R4_const              : std_logic_vector := X"00000104";
  constant cc_R5_const              : std_logic_vector := X"00000105";
  constant cc_R6_const              : std_logic_vector := X"00000106";
  constant cc_R7_const              : std_logic_vector := X"00000107";
  
  -- network settings Dortmund
  -- constant MAC_ADDRESS : mac_type := (X"0011", X"9561", X"95A0");
  -- constant NETMASK : ip_type := (255, 255, 255, 0);
  -- constant IP_ADDRESS : ip_type := (129, 217, 160, 118);
  -- constant GATEWAY : ip_type := (129, 217, 160, 1);
  -- constant FIRST_PORT : integer := 5000;

  -- network settings Zuerich
  constant MAC_ADDRESS : mac_type := (X"FAC7", X"0FAD", X"1101");
  constant NETMASK     : ip_type  := (255, 255, 248, 0);
  constant IP_ADDRESS  : ip_type  := (192, 33, 99, 246);
  constant GATEWAY     : ip_type  := (192, 33, 96, 1);
  constant FIRST_PORT  : integer  := 5000;

  -- W5300 settings
  constant W5300_S_INC : std_logic_vector(6 downto 0) := "1000000"; -- socket address offset
  -- W5300 Registers
  constant W5300_BASE_ADR    : std_logic_vector (9 downto 0) := (others => '0');
  constant W5300_MR          : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"0";
  constant W5300_IR          : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"2";
  constant W5300_IMR         : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"4";
  constant W5300_SHAR        : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"8";
  constant W5300_GAR         : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"10";
  constant W5300_SUBR        : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"14";
  constant W5300_SIPR        : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"18";
  constant W5300_RTR         : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"1C";
  constant W5300_RCR         : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"1E";
  constant W5300_TMS01R      : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"20";
  constant W5300_TMS23R      : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"22";
  constant W5300_TMS45R      : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"24";
  constant W5300_TMS67R      : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"26";
  constant W5300_RMS01R      : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"28";
  constant W5300_RMS23R      : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"2A";
  constant W5300_RMS45R      : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"2C";
  constant W5300_RMS67R      : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"2E";
  constant W5300_MTYPER      : std_logic_vector (9 downto 0) := W5300_BASE_ADR + X"30";
  constant W5300_S0_MR       : std_logic_vector (9 downto 0) := W5300_BASE_ADR + "1000000000" + X"0";
  constant W5300_S0_CR       : std_logic_vector (9 downto 0) := W5300_BASE_ADR + "1000000000" + X"2";
  constant W5300_S0_IMR      : std_logic_vector (9 downto 0) := W5300_BASE_ADR + "1000000000" + X"4";
  constant W5300_S0_IR       : std_logic_vector (9 downto 0) := W5300_BASE_ADR + "1000000000" + X"6";
  constant W5300_S0_SSR      : std_logic_vector (9 downto 0) := W5300_BASE_ADR + "1000000000" + X"8";
  constant W5300_S0_PORTR    : std_logic_vector (9 downto 0) := W5300_BASE_ADR + "1000000000" + X"A";
  constant W5300_S0_DPORTR   : std_logic_vector (9 downto 0) := W5300_BASE_ADR + "1000000000" + X"12";
  constant W5300_S0_DIPR     : std_logic_vector (9 downto 0) := W5300_BASE_ADR + "1000000000" + X"14";
  constant W5300_S0_TX_WRSR  : std_logic_vector (9 downto 0) := W5300_BASE_ADR + "1000000000" + X"20";
  constant W5300_S0_TX_FSR   : std_logic_vector (9 downto 0) := W5300_BASE_ADR + "1000000000" + X"24";
  constant W5300_S0_RX_RSR   : std_logic_vector (9 downto 0) := W5300_BASE_ADR + "1000000000" + X"28";
  constant W5300_S0_TX_FIFOR : std_logic_vector (9 downto 0) := W5300_BASE_ADR + "1000000000" + X"2E";
  constant W5300_S0_RX_FIFOR : std_logic_vector (9 downto 0) := W5300_BASE_ADR + "1000000000" + X"30";
  -- End W5300 registers
  constant W5300_TX_FIFO_SIZE_8B : integer := 65536; -- Socket TX FIFO-Size in Bytes
  constant W5300_TX_FIFO_SIZE    : integer := (W5300_TX_FIFO_SIZE_8B / 2); -- Socket TX FIFO-Size in 16 Bit Words
  constant W5300_LAST_SOCKET     : integer := 0;

  -- Commands
  constant CMD_START_DELIMITER : std_logic_vector := X"0040"; -- "@"
  constant CMD_TLED            : std_logic_vector := X"C000"; -- only a test
  constant CMD_READ            : std_logic_vector := X"0001";
  constant PAR_READ_SD         : std_logic_vector := X"0001"; -- read static data block
  constant PAR_READ_DD         : std_logic_vector := X"0002"; -- read dynamic data block
  -- only for debugging: data_block (0) = ADDR
  constant PAR_READ_SD_ADDR : std_logic_vector := X"0004"; -- read from address in static data block
  constant PAR_READ_DD_ADDR : std_logic_vector := X"0008"; -- read from address in dynamic data block
  constant CMD_WRITE    : std_logic_vector     := X"0002";
  constant PAR_WRITE_SD : std_logic_vector     := X"0001"; -- write static data block
  -- only for debugging: data_block (0) = ADDR, data_block (1) = DATA
  constant PAR_WRITE_SD_ADDR : std_logic_vector := X"0002"; -- write to address in static data ram
  -- ping all FTUs
  constant CMD_PING : std_logic_vector := X"0010"; -- ping all FTUs
  -- turn automatic sending of dd-block and ftu-error-list on or off
  constant CMD_AUTOSEND : std_logic_vector := X"0020";
  constant PAR_AUTOSEND_EA : std_logic_vector := X"0001"; -- enable automatic sending
  constant PAR_AUTOSEND_DA : std_logic_vector := X"0000"; -- disable automatic sending

  -- header length of data packages
  constant FTM_HEADER_LENGTH : std_logic_vector (7 DOWNTO 0) := X"0B";
  
  -- FTU error message
  constant FTU_ERROR_LENGTH       : std_logic_vector (11 downto 0) := X"01D"; --(number of unsuccessful calls) + (28 * data) = 29
  
  -- FTU-list parameters
  constant FL_BLOCK_SIZE           : std_logic_vector := X"0F9"; -- FTU-list size -- 9 + (40 * 6) = 249
  constant NO_OF_FTU_LIST_REG      : integer := 6;
  constant FTU_LIST_RAM_OFFSET     : integer := 16#009#;
  constant FTU_LIST_RAM_ADDR_WIDTH : integer := 12;
  
  -- Static data block
  constant SD_BLOCK_SIZE_GENERAL     : integer := 32;               -- X"20" -- static data block size without FTU data
  constant SD_FTU_BASE_ADDR          : std_logic_vector := X"020";  -- beginning of FTU data
  constant STATIC_RAM_CFG_FTU_OFFSET : integer := 16#020#;
  constant STATIC_RAM_ADDR_WIDTH     : integer := 12;
  constant SD_FTU_DATA_SIZE          : integer := 10;               -- X"00A" -- size of one FTU data block
  constant SD_FTU_NUM                : integer := 40;               -- number of FTUs
  constant SD_FTU_ACTIVE_BASE_ADDR   : std_logic_vector := X"1B0";  -- beginning of active FTU lists
  constant SD_FTU_ACTIVE_NUM         : integer := 4;                -- number of active FTU lists (cr0 to cr3)
  constant SD_BLOCK_SIZE             : std_logic_vector (11 downto 0) := X"1B4";  -- total size of static data block

  -- dynamic data block
  --constant DD_BLOCK_SIZE         : std_logic_vector (11 downto 0) := X"010"; -- 7 + (40 * 12) = 0x1E7 --total size of dynamic data block
  constant DD_BLOCK_SIZE         : std_logic_vector (11 downto 0) := X"1E7"; -- 7 + (40 * 12) = 0x1E7 --total size of dynamic data block
  constant DD_BLOCK_SIZE_GENERAL : integer := 7; -- dynamic block size without FTU data
  
  -- addresses in static data block
  constant SD_ADDR_general_settings : std_logic_vector := X"000";
  constant SD_ADDR_led              : std_logic_vector := X"001";
  constant SD_ADDR_lp_pt_freq       : std_logic_vector := X"002";
  constant SD_ADDR_lp_pt_ratio      : std_logic_vector := X"003";
  constant SD_ADDR_lp1_amplitude    : std_logic_vector := X"004";
  constant SD_ADDR_lp2_amplitude    : std_logic_vector := X"005";
  constant SD_ADDR_lp1_delay        : std_logic_vector := X"006";
  constant SD_ADDR_lp2_delay        : std_logic_vector := X"007";
  constant SD_ADDR_coin_n_p         : std_logic_vector := X"008";
  constant SD_ADDR_coin_n_c         : std_logic_vector := X"009";
  constant SD_ADDR_trigger_delay    : std_logic_vector := X"00A";
  constant SD_ADDR_timemarker_delay : std_logic_vector := X"00B";
  constant SD_ADDR_dead_time        : std_logic_vector := X"00C";
  constant SD_ADDR_cc_R0_HI         : std_logic_vector := X"00D";
  constant SD_ADDR_cc_R0_LO         : std_logic_vector := X"00E";
  constant SD_ADDR_cc_R1_HI         : std_logic_vector := X"00F";
  constant SD_ADDR_cc_R1_LO         : std_logic_vector := X"010";
  constant SD_ADDR_cc_R8_HI         : std_logic_vector := X"011";
  constant SD_ADDR_cc_R8_LO         : std_logic_vector := X"012";
  constant SD_ADDR_cc_R9_HI         : std_logic_vector := X"013";
  constant SD_ADDR_cc_R9_LO         : std_logic_vector := X"014";
  constant SD_ADDR_cc_R11_HI        : std_logic_vector := X"015";
  constant SD_ADDR_cc_R11_LO        : std_logic_vector := X"016";
  constant SD_ADDR_cc_R13_HI        : std_logic_vector := X"017";
  constant SD_ADDR_cc_R13_LO        : std_logic_vector := X"018";
  constant SD_ADDR_cc_R14_HI        : std_logic_vector := X"019";
  constant SD_ADDR_cc_R14_LO        : std_logic_vector := X"01A";
  constant SD_ADDR_cc_R15_HI        : std_logic_vector := X"01B";
  constant SD_ADDR_cc_R15_LO        : std_logic_vector := X"01C";
  constant SD_ADDR_coin_win_p	    : std_logic_vector := X"01D";
  constant SD_ADDR_coin_win_c	    : std_logic_vector := X"01E";
  constant SD_ADDR_ftu_prescaling_0 : std_logic_vector := X"029";
  constant SD_ADDR_ftu_active_cr0   : std_logic_vector := X"1B0";
  constant SD_ADDR_ftu_active_cr1   : std_logic_vector := X"1B1";
  constant SD_ADDR_ftu_active_cr2   : std_logic_vector := X"1B2";
  constant SD_ADDR_ftu_active_cr3   : std_logic_vector := X"1B3";
  constant STATIC_RAM_ACT_FTU_OFFSET : integer := 16#1B0#;


  -- arrays for default values
  type sd_block_default_array_type is array (0 to (SD_BLOCK_SIZE_GENERAL - 1)) of std_logic_vector (15 downto 0);
  type sd_block_ftu_default_array_type is array (0 to (SD_FTU_DATA_SIZE - 1)) of std_logic_vector (15 downto 0);
  type sd_block_default_ftu_active_list_type is array (0 to (SD_FTU_ACTIVE_NUM - 1)) of std_logic_vector (15 downto 0);
  
  -- general default values
  -- !!! to be defined !!!
  constant sd_block_default_array : sd_block_default_array_type := (
      --X"0080", -- SD_ADDR_general_settings  -- general settings
      X"0060", -- SD_ADDR_general_settings  -- general settings
      X"0000", -- SD_ADDR_led               -- on-board status LEDs
      --X"0400", -- SD_ADDR_lp_pt_freq        -- light pulser and pedestal trigger frequency
      X"0001", -- SD_ADDR_lp_pt_freq        -- light pulser and pedestal trigger frequency
      --X"0001", -- SD_ADDR_lp_pt_ratio...    -- ratio between LP1, LP2 and pedestal triggers
      X"0420", -- SD_ADDR_lp_pt_ratio...    -- ratio between LP1, LP2 and pedestal triggers
      X"0004", -- SD_ADDR_lp1_amplitude     -- light pulser 1 amplitude
      X"0005", -- SD_ADDR_lp2_amplitude     -- light pulser 2 amplitude
      X"0006", -- SD_ADDR_lp1_delay         -- light pulser 1 delay
      X"0007", -- SD_ADDR_lp2_delay         -- light pulser 2 delay
      X"0001", -- SD_ADDR_coin_n_p          -- majority coincidence n (for physics)
      X"001E", -- SD_ADDR_coin_n_c          -- majority coincidence n (for calibration)
      X"0000", -- SD_ADDR_trigger_delay     -- trigger delay
      X"0000", -- SD_ADDR_timemarker_delay  -- timemarker delay
      --X"0019", -- SD_ADDR_dead_time         -- dead time, 8ns + 4x25ns = 108ns
      X"0000", -- SD_ADDR_dead_time         -- dead time, 8ns + 4x25ns = 108ns
      X"0003", -- SD_ADDR_cc_R0_HI          -- clock conditioner R0 bits 31...16
      X"8000", -- SD_ADDR_cc_R0_LO          -- clock conditioner R0 bits 15...0
      X"0001", -- SD_ADDR_cc_R1_HI          -- clock conditioner R1 bits 31...16
      X"0101", -- SD_ADDR_cc_R1_LO          -- clock conditioner R1 bits 15...0
      X"1000", -- SD_ADDR_cc_R8_HI          -- clock conditioner R8 bits 31...16
      X"0908", -- SD_ADDR_cc_R8_LO          -- clock conditioner R8 bits 15...0
      X"A003", -- SD_ADDR_cc_R9_HI          -- clock conditioner R9 bits 31...16
      X"2A09", -- SD_ADDR_cc_R9_LO          -- clock conditioner R9 bits 15...0
      X"0082", -- SD_ADDR_cc_R11_HI         -- clock conditioner R11 bits 31...16
      X"000B", -- SD_ADDR_cc_R11_LO         -- clock conditioner R11 bits 15...0
      X"020A", -- SD_ADDR_cc_R13_HI         -- clock conditioner R13 bits 31...16
      X"000D", -- SD_ADDR_cc_R13_LO         -- clock conditioner R13 bits 15...0
      X"0830", -- SD_ADDR_cc_R14_HI         -- clock conditioner R14 bits 31...16
      X"280E", -- SD_ADDR_cc_R14_LO         -- clock conditioner R14 bits 15...0
      X"1400", -- SD_ADDR_cc_R15_HI         -- clock conditioner R15 bits 31...16
      X"FA0F", -- SD_ADDR_cc_R15_LO         -- clock conditioner R15 bits 15...0
      X"0001", -- SD_ADDR_coin_win_p        -- majority coincidence window (for physics), 8ns + 4x1ns = 12ns
      X"0001", -- SD_ADDR_coin_win_c        -- majority coincidence window (for calibration), 8ns + 4x1ns = 12ns
      X"001F"  --                           -- Spare
  );

  -- default values for all FTUs
  constant sd_block_ftu_default_array : sd_block_ftu_default_array_type := (
      X"01FF", -- enables patch 0 board x crate y
      X"01FF", -- enables patch 1 board x crate y
      X"01FF", -- enables patch 2 board x crate y
      X"01FF", -- enables patch 3 board x crate y
      X"01F4", -- DAC_A board x crate y
      X"01F4", -- DAC_B board x crate y
      X"01F4", -- DAC_C board x crate y
      X"01F4", -- DAC_D board x crate y
      X"0010", -- DAC_H board x crate y
      X"0001"  -- Prescaling board x crate y
  );

  --default values for active FTU lists
  constant sd_block_default_ftu_active_list : sd_block_default_ftu_active_list_type := (
    X"0000",
    X"0000",
    X"0000",
    X"0000"
  );

  -- Light pulser interface
  constant low_PLC : integer := 16;   -- minimal pulse duration in units of 4 ns
  constant width_PLC : integer := 6;  -- counter width pulse duration 

  -- Timing counter
  --  constant tc_width : integer := 48;    -- width (number of bits) of timing counter
  --  constant zero : unsigned (tc_width - 1 downto 0) := (others => '0');  

  --======================================================================================
  -- Constants for calibration and pedestal triggers generation
  --======================================================================================
  constant LOW_SPEED_CLOCK_FREQ   : real          := 50000000.0;
  constant LOW_SPEED_CLOCK_PERIOD : real          := 1.0/LOW_SPEED_CLOCK_FREQ;
  constant MS_PERIOD              : real          := 0.001;
  constant MAX_MS_COUNTER_WIDTH   : integer       := integer(ceil(log2(real(MS_PERIOD/LOW_SPEED_CLOCK_PERIOD))));
  constant MAX_MS_COUNTER_VAL     : integer       := integer(MS_PERIOD/LOW_SPEED_CLOCK_PERIOD);
  --======================================================================================
  
  --======================================================================================
  -- Constants for trigger and TIM signals width (8ns+value*4ns)
  --======================================================================================
  constant TRIG_SIGNAL_PULSE_WIDTH        : integer range 0 to 15 := 10;
  constant TIM_SIGNAL_PULSE_WIDTH         : integer range 0 to 15 := 0;
  --======================================================================================
  
end ftm_constants;
