--=======================================================================================
-- TITLE        : Trigger manager
-- DESCRIPTION  : Top architecture file to detect events and generate triggers
-- FILE         : trigger_manager.vhd
-- COMPANY      : Micro-Cameras & Space Exploration SA
--=======================================================================================
-- CREATION
-- DATE 			AUTHOR 	PROJECT 		REVISION
-- 09/03/2011 JGi                110309a
--=======================================================================================
-- MODIFICATION HISTORY
-- DATE 			AUTHOR	PROJECT			REVISION	COMMENTS
-- 09/03/2011 JGi 	 	            110309a 	Description
--=======================================================================================
-- Library Definition
library ieee;
  use ieee.std_logic_1164.all;
  use ieee.numeric_std.all;

-- Entity Definition
entity trigger_manager is
  port( --clocks
    		clk_50MHz  					: in 	std_logic;
    		clk_250MHz 					: in 	std_logic;
    		clk_250MHz_180			: in 	std_logic;
    		--trigger primitives from FTUs
    		trig_prim_0 				: in 	std_logic_vector(9 downto 0);  --crate 0
    		trig_prim_1 				: in 	std_logic_vector(9 downto 0);  --crate 1
    		trig_prim_2 				: in 	std_logic_vector(9 downto 0);  --crate 2
    		trig_prim_3 				: in 	std_logic_vector(9 downto 0);  --crate 3
    		--external signals
    		ext_trig_1 					: in 	std_logic;
    		ext_trig_2 					: in 	std_logic;
    		ext_veto   					: in 	std_logic;
    		FAD_busy_0 					: in 	std_logic;  --crate 0
    		FAD_busy_1 					: in 	std_logic;  --crate 1
    		FAD_busy_2 					: in 	std_logic;  --crate 2
    		FAD_busy_3 					: in 	std_logic;  --crate 3
    		--control signals from e.g. main control
    		start_run         	: in 	std_logic;  --enable trigger output
    		stop_run          	: in 	std_logic;  --disable trigger output
				new_config					: in	std_logic;
    		--settings register (see FTM Firmware Specifications)
    		general_settings   	: in 	std_logic_vector(15 downto 0);
    		LP_and_PED_freq    	: in 	std_logic_vector(15 downto 0);
    		LP1_LP2_PED_ratio  	: in 	std_logic_vector(15 downto 0);
    		maj_coinc_n_phys   	: in 	std_logic_vector(15 downto 0);
    		maj_coinc_n_calib  	: in 	std_logic_vector(15 downto 0);
    		trigger_delay      	: in 	std_logic_vector(15 downto 0);
    		TIM_delay          	: in 	std_logic_vector(15 downto 0);
    		dead_time          	: in 	std_logic_vector(15 downto 0);
    		coinc_window_phys  	: in 	std_logic_vector(15 downto 0);
    		coinc_window_calib	: in 	std_logic_vector(15 downto 0);
    		active_FTU_list_0  	: in 	std_logic_vector(15 downto 0);
    		active_FTU_list_1  	: in 	std_logic_vector(15 downto 0);
    		active_FTU_list_2  	: in 	std_logic_vector(15 downto 0);
    		active_FTU_list_3  	: in 	std_logic_vector(15 downto 0);
    		--control signals or information for other entities
				trigger_ID_read			: in 	std_logic;
				trig_cnt_copy_read	: in	std_logic;
				trigger_ID_ready		: out std_logic;
    		trigger_ID          : out std_logic_vector(55 downto 0);
    		trig_cnt_copy       : out std_logic_vector(31 downto 0);  --counter reading
    		trig_cnt_copy_valid : out std_logic;  --trigger counter reading is valid
    		trigger_active      : out std_logic;  --phys triggers are enabled/active
				config_done					: out	std_logic;
    		LP1_pulse           : out std_logic;  --send start signal to light pulser 1
    		LP2_pulse           : out std_logic;  --send start signal to light pulser 2
    		--trigger and time marker output signals to FADs
    		trigger_signal      : out std_logic;
    		TIM_signal          : out std_logic);
end trigger_manager;

-- Architecture Definition
architecture RTL of trigger_manager is

	component interface_sync_250MHz is
	  port( clk_250MHz								: in  std_logic;
	    		start_run         				: in 	std_logic;
	    		stop_run          				: in 	std_logic;
					new_config								: in	std_logic;
	    		general_settings   				: in 	std_logic_vector(15 downto 0);
	    		LP_and_PED_freq    				: in 	std_logic_vector(15 downto 0);
	    		LP1_LP2_PED_ratio  				: in 	std_logic_vector(15 downto 0);
	    		maj_coinc_n_phys   				: in 	std_logic_vector(15 downto 0);
	    		maj_coinc_n_calib  				: in 	std_logic_vector(15 downto 0);
	    		trigger_delay      				: in 	std_logic_vector(15 downto 0);
	    		TIM_delay          				: in 	std_logic_vector(15 downto 0);
	    		dead_time          				: in 	std_logic_vector(15 downto 0);
	    		coinc_window_phys  				: in 	std_logic_vector(15 downto 0);
	    		coinc_window_calib				: in 	std_logic_vector(15 downto 0);
	    		active_FTU_list_0  				: in 	std_logic_vector(15 downto 0);
	    		active_FTU_list_1  				: in 	std_logic_vector(15 downto 0);
	    		active_FTU_list_2  				: in 	std_logic_vector(15 downto 0);
	    		active_FTU_list_3  				: in 	std_logic_vector(15 downto 0);
	    		config_done								: out	std_logic;
	    		sync_start_run         		:	out	std_logic;
	    		sync_stop_run          		:	out	std_logic;
	    		sync_general_settings   	:	out	std_logic_vector(7 downto 0);
	    		sync_LP_and_PED_freq    	:	out	std_logic_vector(9 downto 0);
	    		sync_LP1_LP2_PED_ratio  	:	out	std_logic_vector(14 downto 0);
	    		sync_maj_coinc_n_phys   	:	out	std_logic_vector(5 downto 0);
	    		sync_maj_coinc_n_calib		:	out	std_logic_vector(5 downto 0);
	    		sync_trigger_delay      	:	out	std_logic_vector(9 downto 0);
	    		sync_TIM_delay          	:	out	std_logic_vector(9 downto 0);
	    		sync_dead_time          	:	out	std_logic_vector(15 downto 0);
	    		sync_coinc_window_phys  	:	out	std_logic_vector(3 downto 0);
	    		sync_coinc_window_calib		:	out	std_logic_vector(3 downto 0);
	    		sync_active_FTU_list_0  	:	out	std_logic_vector(9 downto 0);
	    		sync_active_FTU_list_1  	:	out	std_logic_vector(9 downto 0);
	    		sync_active_FTU_list_2  	:	out	std_logic_vector(9 downto 0);
	    		sync_active_FTU_list_3  	:	out	std_logic_vector(9 downto 0));
	end component;

	signal i_config_done							: std_logic	:= '0';
	signal i_sync_start_run         	: std_logic	:= '0';
	signal i_sync_stop_run          	: std_logic	:= '0';
	signal i_sync_general_settings  	: std_logic_vector(7 downto 0)	:= (others => '0');
	signal i_sync_LP_and_PED_freq   	: std_logic_vector(9 downto 0)	:= (others => '0');
	signal i_sync_LP1_LP2_PED_ratio 	: std_logic_vector(14 downto 0)	:= (others => '0');
	signal i_sync_maj_coinc_n_phys  	: std_logic_vector(5 downto 0)	:= (others => '0');
	signal i_sync_maj_coinc_n_calib 	: std_logic_vector(5 downto 0)	:= (others => '0');
	signal i_sync_trigger_delay     	: std_logic_vector(9 downto 0)	:= (others => '0');
	signal i_sync_TIM_delay         	: std_logic_vector(9 downto 0)	:= (others => '0');
	signal i_sync_dead_time         	: std_logic_vector(15 downto 0)	:= (others => '0');
	signal i_sync_coinc_window_phys 	: std_logic_vector(3 downto 0)	:= (others => '0');
	signal i_sync_coinc_window_calib	: std_logic_vector(3 downto 0)	:= (others => '0');
	signal i_sync_active_FTU_list_0 	: std_logic_vector(9 downto 0)	:= (others => '0');
	signal i_sync_active_FTU_list_1 	: std_logic_vector(9 downto 0)	:= (others => '0');
	signal i_sync_active_FTU_list_2 	: std_logic_vector(9 downto 0)	:= (others => '0');
	signal i_sync_active_FTU_list_3		: std_logic_vector(9 downto 0)	:= (others => '0');

	component interface_sync_50MHz is
	  port( clk_50MHz									: in  std_logic;
	  			clk_250MHz								: in  std_logic;
	  			config_done								:	in	std_logic;
	  			trigger_active						:	in	std_logic;
	  			trigger_ID_done						:	in	std_logic;
	  			trigger_ID								:	in	std_logic_vector(55 downto 0);
	  			trigger_ID_read						: in	std_logic;
	  			trigger_cnt_read					:	in	std_logic;
	  			sync_config_done					: out	std_logic;
	  			sync_trigger_active				: out	std_logic;
	  			sync_trigger_ID_ready			: out	std_logic;
	  			sync_trigger_ID						:	out	std_logic_vector(55 downto 0);
	  			trigger_cnt_valid					:	out	std_logic;
	  			trigger_cnt_copy					:	out	std_logic_vector(31 downto 0));
	end component;

	component FTU_trigger_counter is
	  port( clk_250MHz								: in  std_logic;
					clk_250MHz_180						: in  std_logic;
					phys_coinc_window					: in	std_logic_vector(3 downto 0);
					calib_coinc_window				: in	std_logic_vector(3 downto 0);
					active_FTU_list_0					: in	std_logic_vector(9 downto 0);
					active_FTU_list_1					: in	std_logic_vector(9 downto 0);
					active_FTU_list_2					: in	std_logic_vector(9 downto 0);
					active_FTU_list_3					: in	std_logic_vector(9 downto 0);
	    		trig_prim_0								: in 	std_logic_vector(9 downto 0);
	    		trig_prim_1								: in 	std_logic_vector(9 downto 0);
	    		trig_prim_2								: in 	std_logic_vector(9 downto 0);
	    		trig_prim_3								: in 	std_logic_vector(9 downto 0);
	    		phys_events								: out	std_logic_vector(5 downto 0);
	    		calib_events							: out	std_logic_vector(5 downto 0));
	end component;

	signal i_phys_events							: std_logic_vector(5 downto 0)	:= (others => '0');
	signal i_calib_events							: std_logic_vector(5 downto 0)	:= (others => '0');

	component calibration_pedestal is
	  port( clk_50MHz									: in  std_logic;
	  			new_config								: in	std_logic;
	  			general_settings					: in 	std_logic_vector(7 downto 0);
	    		LP_and_PED_freq						: in 	std_logic_vector(9 downto 0);
	    		LP1_LP2_PED_ratio					: in 	std_logic_vector(14 downto 0);
	    		LP1_pulse									: out std_logic;
	    		LP2_pulse									: out std_logic;
	    		PED_pulse									: out	std_logic);
	end component;

	signal i_LP1_pulse								: std_logic;
	signal i_LP2_pulse								: std_logic;
	signal i_PED_pulse								: std_logic;

	component trigger_generator is
	  port( clk_250MHz								: in  std_logic;
	  			start_run									: in	std_logic;
	  			stop_run									: in	std_logic;
	  			general_settings   				:	in	std_logic_vector(7 downto 0);
		    	maj_coinc_n_phys   				:	in	std_logic_vector(5 downto 0);
		    	maj_coinc_n_calib					:	in	std_logic_vector(5 downto 0);
		    	trigger_delay      				:	in	std_logic_vector(9 downto 0);
		    	TIM_delay          				:	in	std_logic_vector(9 downto 0);
		    	dead_time          				:	in	std_logic_vector(15 downto 0);
	    		ext_trig_1 								: in 	std_logic;
	    		ext_trig_2 								: in 	std_logic;
	    		ext_veto   								: in 	std_logic;
	    		FAD_busy_0 								: in 	std_logic;
	    		FAD_busy_1 								: in 	std_logic;
	    		FAD_busy_2 								: in 	std_logic;
	    		FAD_busy_3 								: in 	std_logic;
	    		phys_events								: in	std_logic_vector(5 downto 0);
	    		calib_events							: in	std_logic_vector(5 downto 0);
	    		LP1_pulse									: in	std_logic;
	    		LP2_pulse									: in	std_logic;
	    		PED_pulse									: in	std_logic;
					trigger_ID_done						: out	std_logic;
	  			trigger_ID								: out	std_logic_vector(55 downto 0);
	  			trigger_active						: out	std_logic;
	  			trigger_signal						:	out	std_logic;
	  			TIM_signal								:	out	std_logic);
	end component;

	signal i_trigger_active   				: std_logic;
	signal i_trigger_ID_done  				: std_logic;
	signal i_trigger_ID								: std_logic_vector(55 downto 0);

begin

	-- Component instantiation
	inst_settings_sync: interface_sync_250MHz
		port map(	clk_250MHz							=> clk_250MHz,
							start_run         			=> start_run,
							stop_run          			=> stop_run,
							new_config							=> new_config,
							general_settings   			=> general_settings,
							LP_and_PED_freq    			=> LP_and_PED_freq,
							LP1_LP2_PED_ratio  			=> LP1_LP2_PED_ratio,
							maj_coinc_n_phys   			=> maj_coinc_n_phys,
							maj_coinc_n_calib  			=> maj_coinc_n_calib,
							trigger_delay      			=> trigger_delay,
							TIM_delay          			=> TIM_delay,
							dead_time          			=> dead_time,
							coinc_window_phys  			=> coinc_window_phys,
							coinc_window_calib			=> coinc_window_calib,
							active_FTU_list_0  			=> active_FTU_list_0,
							active_FTU_list_1  			=> active_FTU_list_1,
							active_FTU_list_2  			=> active_FTU_list_2,
							active_FTU_list_3  			=> active_FTU_list_3,
    					config_done							=> i_config_done,
							sync_start_run         	=> i_sync_start_run,
							sync_stop_run          	=> i_sync_stop_run,
							sync_general_settings   => i_sync_general_settings,
							sync_LP_and_PED_freq    => i_sync_LP_and_PED_freq,
							sync_LP1_LP2_PED_ratio  => i_sync_LP1_LP2_PED_ratio,
							sync_maj_coinc_n_phys   => i_sync_maj_coinc_n_phys,
							sync_maj_coinc_n_calib	=> i_sync_maj_coinc_n_calib,
							sync_trigger_delay      => i_sync_trigger_delay,
							sync_TIM_delay          => i_sync_TIM_delay,
							sync_dead_time          => i_sync_dead_time,
							sync_coinc_window_phys  => i_sync_coinc_window_phys,
							sync_coinc_window_calib	=> i_sync_coinc_window_calib,
							sync_active_FTU_list_0  => i_sync_active_FTU_list_0,
							sync_active_FTU_list_1  => i_sync_active_FTU_list_1,
							sync_active_FTU_list_2  => i_sync_active_FTU_list_2,
							sync_active_FTU_list_3  => i_sync_active_FTU_list_3);

	inst_interface_sync: interface_sync_50MHz
		port map( clk_50MHz								=> clk_50MHz,
							clk_250MHz							=> clk_250MHz,
							config_done							=> i_config_done,
							trigger_active					=> i_trigger_active,
							trigger_ID_done					=> i_trigger_ID_done,
							trigger_ID							=> i_trigger_ID,
							trigger_ID_read					=> trigger_ID_read,
							trigger_cnt_read				=> trig_cnt_copy_read,
							sync_config_done				=> config_done,
							sync_trigger_active			=> trigger_active,
							sync_trigger_ID_ready		=> trigger_ID_ready,
							sync_trigger_ID					=> trigger_ID,
							trigger_cnt_valid				=> trig_cnt_copy_valid,
							trigger_cnt_copy				=> trig_cnt_copy);

	inst_FTU_trig: FTU_trigger_counter
		port map( clk_250MHz							=> clk_250MHz,
							clk_250MHz_180					=> clk_250MHz_180,
							phys_coinc_window				=> i_sync_coinc_window_phys,
							calib_coinc_window			=> i_sync_coinc_window_calib,
							active_FTU_list_0				=> i_sync_active_FTU_list_0,
							active_FTU_list_1				=> i_sync_active_FTU_list_1,
							active_FTU_list_2				=> i_sync_active_FTU_list_2,
							active_FTU_list_3				=> i_sync_active_FTU_list_3,
							trig_prim_0							=> trig_prim_0,
							trig_prim_1							=> trig_prim_1,
							trig_prim_2							=> trig_prim_2,
							trig_prim_3							=> trig_prim_3,
							phys_events							=> i_phys_events,
							calib_events						=> i_calib_events);

	inst_internal_trig: calibration_pedestal
		port map( clk_50MHz								=> clk_50MHz,
							new_config         			=> new_config,
							general_settings				=> general_settings(7 downto 0),
							LP_and_PED_freq					=> LP_and_PED_freq(9 downto 0),
							LP1_LP2_PED_ratio				=> LP1_LP2_PED_ratio(14 downto 0),
							LP1_pulse								=> i_LP1_pulse,
							LP2_pulse								=> i_LP2_pulse,
							PED_pulse								=> i_PED_pulse);

	inst_trig_gen: trigger_generator
		port map( clk_250MHz							=> clk_250MHz,
							start_run								=> i_sync_start_run,
							stop_run								=> i_sync_stop_run,
							general_settings  			=> i_sync_general_settings,
							maj_coinc_n_phys  			=> i_sync_maj_coinc_n_phys,
							maj_coinc_n_calib				=> i_sync_maj_coinc_n_calib,
							trigger_delay     			=> i_sync_trigger_delay,
							TIM_delay         			=> i_sync_TIM_delay,
							dead_time         			=> i_sync_dead_time,
							ext_trig_1 							=> ext_trig_1,
							ext_trig_2 							=> ext_trig_2,
							ext_veto   							=> ext_veto,
							FAD_busy_0 							=> FAD_busy_0,
							FAD_busy_1 							=> FAD_busy_1,
							FAD_busy_2 							=> FAD_busy_2,
							FAD_busy_3 							=> FAD_busy_3,
							phys_events							=> i_phys_events,
							calib_events						=> i_calib_events,
							LP1_pulse								=> i_LP1_pulse,
							LP2_pulse								=> i_LP2_pulse,
							PED_pulse								=> i_PED_pulse,
							trigger_ID_done					=> i_trigger_ID_done,
							trigger_ID							=> i_trigger_ID,
							trigger_active    			=> i_trigger_active,
							trigger_signal					=> trigger_signal,
							TIM_signal							=> TIM_signal);

	LP1_pulse	<= i_LP1_pulse;
	LP2_pulse	<= i_LP2_pulse;

end RTL;