// **************************************************************************
/** @class FactFits

@brief FITS writter for the FACT project. 

The FactFits class is able to open, manage and update FITS files. 

The file columns should be given to the class before the file is openned. Once
a file has been created, the structure of its columns cannot be changed. Only
row can be added.

This class relies on the CCfits and CFitsIO packages.

*/
// **************************************************************************
#include "Fits.h"

#include "Time.h"
#include "Converter.h"

//for file stats
#include <sys/stat.h>

using namespace std;

// --------------------------------------------------------------------------
//
//! This gives a standard variable to the file writter. 
//! This variable should not be related to the DIM service being logged. 
//! @param desc the description of the variable to add
//! @param dataFormat the FITS data format corresponding to the variable to add.
//! @param dataPointer the memory location where the variable is stored
//! @param numDataBytes the number of bytes taken by the variable
//
void Fits::AddStandardColumn(Description& desc, std::string dataFormat, void* dataPointer, long unsigned int numDataBytes)
{
	//check if entry already exist
	for (std::vector<Description>::iterator it=fStandardColDesc.begin(); it != fStandardColDesc.end(); it++)
		if (it->name == desc.name)
			return;
	fStandardColDesc.push_back(desc);
	fStandardFormats.push_back(dataFormat);
	fStandardPointers.push_back(dataPointer);
	fStandardNumBytes.push_back(numDataBytes);
}
// --------------------------------------------------------------------------
//
//! This gives the file writter access to the DIM data
//! @param desc a vector containing the description of all the columns to log
//! @param dataFormat a vector containing the FITS data format of all the columsn to log
//! @param dataPointer the memory location where the DIM data starts
//! @param numDataBytes the number of bytes taken by the DIM data. 
//	
void Fits::InitDataColumns(std::vector<Description> desc, std::vector<std::string>& dataFormat, void* dataPointer, int numDataBytes)
{//we will copy this information here. It duplicates the data, which is not great, but it is the easiest way of doing it right now
	if (desc.size() == dataFormat.size())
	{
	 	fDataColDesc = desc;
	}
	else
	{
		fDataColDesc.clear();
		for (unsigned int i=0;i<dataFormat.size();i++)
		{
			std::stringstream stt;
			stt << "Data" << i;
			fDataColDesc.push_back(Description(stt.str(), "comment", "unit"));
		}
	}		
	fDataFormats = dataFormat;
	fDataPointer = dataPointer;
	fDataNumBytes = numDataBytes; 	
}
// --------------------------------------------------------------------------
//
//! This opens the FITS file (after the columns have been passed)
//! @param fileName the filename with complete or relative path of the file to open
//! @param tableName the name of the table that will receive the logged data.
//! @param file a pointer to an existing FITS file. If NULL, file will be opened and managed internally
//
void Fits::Open(const std::string& fileName, const std::string& tableName, FITS* file, int* fitsCounter, std::ostream& out)
{		
	if (fMess)
		delete fMess;
	fMess = new MessageImp(out);
	
	fFileName = fileName;
	if (file == NULL)
	{
		try
		{
			fFile = new FITS(fileName, RWmode::Write);
		}
		catch (CCfits::FitsError e)
		{			
			std::stringstream str;
			str << "Could not open FITS file " << fileName << " reason: " << e.message();
			fMess->Error(str);
			fFile = NULL;
			return;
		}	
		fOwner = true;
		fNumOpenFitsFiles = fitsCounter;
		(*fNumOpenFitsFiles)++;
	}
	else
	{
		fFile = file;
		fOwner = false;
	}
	//concatenate the standard and data columns
	//do it the inneficient way first: its easier and faster to code.
	std::vector<std::string> allNames;
	std::vector<std::string> allDataTypes;
	std::vector<std::string> allUnits;
	fTotalNumBytes = 0;
	for (unsigned int i=0;i<fStandardColDesc.size();i++)
	{
		allNames.push_back(fStandardColDesc[i].name);
		allDataTypes.push_back(fStandardFormats[i]);
		allUnits.push_back(fStandardColDesc[i].unit);
		fTotalNumBytes += fStandardNumBytes[i];
	}
	//for (int i=static_cast<int>(fDataColDesc.size())-1;i>=0;i--)
	for (int i=0; i< static_cast<int>(fDataColDesc.size()); i++)
	{
		if (fDataColDesc[i].name != "")
			allNames.push_back(fDataColDesc[i].name);
		else 
		{
			std::stringstream stt;
			stt << "Data" << i;
			allNames.push_back(stt.str());
		}
		allDataTypes.push_back(fDataFormats[i]);
		allUnits.push_back(fDataColDesc[i].unit);	
	}
	fTotalNumBytes += fDataNumBytes;
	
	bool updating = false;
	try
	{
		std::string factTableName = "FACT-" + tableName;
		fTable = fFile->addTable(factTableName, 0, allNames, allDataTypes, allUnits);
		fCopyBuffer = new unsigned char[fTotalNumBytes]; 
		fNumRows = fTable->rows();
		if (fNumRows !=0)
		{//If the file already existed, then we must load its data to memory before writing to it.
			std::vector<std::string> tableNameString;
			tableNameString.push_back(tableName);
			fFile->read(tableNameString);	
			std::map<std::string, Column*> cMap = fTable->column();
			std::map<std::string, Column*>::iterator cMapIt;
			for (cMapIt = cMap.begin(); cMapIt != cMap.end(); cMapIt++)
			{
				//TODO this only works for scalar columns I assume. upgrade it to fully read vector columns
				cMapIt->second->readData(1, fNumRows);		
			}	
			updating = true;
		}
	}
	catch(CCfits::FitsError e)
	{
		std::stringstream str;
		str << "Could not open or create FITS table " << tableName << " in  file " << fileName << " reason: " << e.message();
		fMess->Error(str);
		fTable = NULL;
	}
			
	fRefMjD = * static_cast<double*>(fStandardPointers[0]);
	if (!updating)
		WriteHeaderKeys();
}
// --------------------------------------------------------------------------
//
//! This writes the standard header 
//
void Fits::WriteHeaderKeys()
{
	if (!fTable)
		return;
	std::string name;
	std::string comment;
	
	float floatValue;
	double doubleValue;
	std::string stringValue;
	try
	{	
		name = "EXTREL";
		comment = "Release Number";
		floatValue = 1.0f;
		fTable->addKey(name, floatValue, comment);
				
		name = "BASETYPE";
		comment = "Base type of the data structure";
		stringValue = "??????";
		fTable->addKey(name, stringValue, comment);
				
		name = "TELESCOP";
		comment = "Telescope that acquired this data";
		stringValue = "FACT";
		fTable->addKey(name, stringValue, comment);
				
		name = "ORIGIN";
		comment = "Institution that wrote the file";
		stringValue = "ISDC";
		fTable->addKey(name, stringValue, comment);
				
		name = "CREATOR";
		comment = "Program that wrote this file";
		stringValue = "FACT++_DataLogger";
		fTable->addKey(name, stringValue, comment);
				
		name = "DATE";
		stringValue = Time().GetAsStr();
		stringValue[10] = 'T';
		comment = "File creation date";
		fTable->addKey(name, stringValue, comment);
		
		name = "TIMESYS";
		stringValue = "TT";
		comment = "Time frame system";
		fTable->addKey(name, stringValue, comment);
		
		name = "TIMEUNIT";
		stringValue = "d";
		comment = "Time unit";
		fTable->addKey(name, stringValue, comment);
		
		name = "TIMEREF";
		stringValue = "UTC";
		comment = "Time reference frame";
		fTable->addKey(name, stringValue, comment);
			
		name = "MJDREF";
		doubleValue = fRefMjD;
		comment = "Modified Julian Date of origin";
		fTable->addKey(name, doubleValue, comment);
	}
	catch (CCfits::FitsError e)
	{
		std::stringstream str;
		str << "Could not add header keys in file " << fFileName << " reason: " << e.message();
		fMess->Error(str);
	}			
	//More ?
}
// --------------------------------------------------------------------------
//
//! This writes one line of data to the file.
//! @param conv the converter corresponding to the service being logged
//
void Fits::Write(Converter* conv)
{

	fTable->makeThisCurrent();
	try
	{
		fTable->insertRows(fNumRows);
	}
	catch(CCfits::FitsError e)
	{
		std::stringstream str;
		str << "Could not insert row in file " << fFileName << " reason: " << e.message();
		fMess->Error(str);
	}
	fNumRows++;
			
	//the first standard variable is the current MjD
	if (fEndMjD == 0)
	{
		std::string name = "TSTART";
		double doubleValue = *static_cast<double*>(fStandardPointers[0]) - fRefMjD;
		std::string comment = "Time of the first received data";
		fTable->addKey(name, doubleValue, comment);	
	}
	fEndMjD = * static_cast<double*>(fStandardPointers[0]);
			
	//first copy the standard variables to the copy buffer
	int shift = 0;

	for (unsigned int i=0;i<fStandardNumBytes.size();i++)
	{
		for (int j=0; j<fStandardNumBytes[i]; j++)
			fCopyBuffer[shift+j] = static_cast<char*>(fStandardPointers[i])[fStandardNumBytes[i]-(j+1)];
		shift+= fStandardNumBytes[i];	
	}

	//now take care of the DIM data. The Converter is here for that purpose
	conv->ToFits(static_cast<void*>(&fCopyBuffer[shift]), fDataPointer, fDataNumBytes);
			 
	//data copied to buffer, can write to fits
	int status = 0;
	//TODO check the status after the write operation
   	fits_write_tblbytes(fFile->fitsPointer(), fNumRows, 1, fTotalNumBytes, fCopyBuffer, &status);
	if (status)
	{
		char text[30];//max length of cfitsio error strings (from doc)
		fits_get_errstatus(status, text);
		std::stringstream str;
		str << "Error while writing FITS row in " << fFileName << ". Message: " << text << " [" << status << "]";
		fMess->Error(str);	
	}
	//This forces the writting of each row to the disk. Otherwise all rows are written when the file is closed.
	///TODO check whether this consumes too much resources or not. If it does, flush every N write operations instead
	try
	{
		fFile->flush();
	}
	catch (CCfits::FitsError e)
	{
		std::stringstream str;
		str << "Error while flushing bytes to disk. File: " << fFileName << " reason: " << e.message();
		fMess->Error(str);	
	}
}
// --------------------------------------------------------------------------
//
//! This closes the currently openned FITS file. 
//! it also updates the header to reflect the time of the last logged row
//	
void Fits::Close() 
{
//WARNING: do NOT delete the table as it gets deleted by the fFile object
//			if (fTable != NULL)
//				delete fTable;
	std::string name = "TEND";
	double doubleValue = fEndMjD - fRefMjD;
	std::string comment = "Time of the last received data";
	fTable->addKey(name, doubleValue, comment);
	
	if (fFile != NULL && fOwner)
		delete fFile;
	fFile = NULL;
	if (fCopyBuffer != NULL)
		delete [] fCopyBuffer;
	fCopyBuffer = NULL;
	if (fOwner && fNumOpenFitsFiles != NULL)
		(*fNumOpenFitsFiles)--;
	if (fMess)
		delete fMess;
	fMess = NULL;
}

// --------------------------------------------------------------------------
//
//! This closes the currently openned FITS file. 
//! it also updates the header to reflect the time of the last logged row
//	
int Fits::GetWrittenSize()
{
	if (!IsOpen())
		return 0;
		
	struct stat st;
	if (stat(fFileName.c_str(), &st))
		return 0;
	else
		return st.st_size;
}
