//****************************************************************
/** @class DataLogger
  
  @brief Logs all message and infos between the services
  
  This is the main logging class facility. 
  It derives from StateMachineDim and DimInfoHandler. the first parent is here to enforce 
  a state machine behaviour, while the second one is meant to make the dataLogger receive
  dim services to which it subscribed from.
  The possible states and transitions of the machine are:
  \dot
  digraph datalogger {
  		node [shape=record, fontname=Helvetica, fontsize=10];
  	e [label="Error" color="red"];
   r [label="Ready"]
   d [label="DailyOpen"]
   w [label="WaitingRun"]
  	l [label="Logging"]
   b [label="BadDailyconfig" color="red"]
   c [label="BadRunConfig" color="red"]
  
  e -> r
  r -> e
  r -> d
  r -> b
  d -> w
  d -> r
  w -> r
  l -> r
  l -> w
  b -> d
  w -> c
  w -> l
  b -> r
  c -> r
  c -> l
  }
  \enddot
  
  @todo
  	- Retrieve also the messages, not only the infos
 */
 //****************************************************************
#include "Event.h"
#include "Time.h"
#include "StateMachineDim.h"
#include "WindowLog.h"
#include "Configuration.h"
#include "ServiceList.h"
#include "Converter.h"
#include "MessageImp.h"
#include "LocalControl.h"
#include "DimDescriptionService.h"

#include "Description.h"

//for getting stat of opened files
#include <unistd.h>
//for getting disk free space
#include <sys/statvfs.h>
//for getting files sizes
#include <sys/stat.h>

#define HAS_FITS
//#define ONE_RUN_FITS_ONLY

#include <fstream>

#include <boost/bind.hpp>
#include <boost/thread.hpp>

#ifdef HAS_FITS
#include "Fits.h"
#endif

//Dim structures
struct DataLoggerStats {
	long long sizeWritten;
	long long freeSpace;
	long writingRate;
};

struct NumSubAndFitsType {
	int numSubscriptions;
	int numOpenFits;
};
//For debugging DIM's services
class MyService
{
public:
	MyService(){};
	MyService(std::string, std::string, void*, int){};
	MyService(std::string, const char*){};
	void updateService(){};
	void updateService(void*, int){};
	void setQuality(int){};
};
class DataLogger : public StateMachineDim, DimInfoHandler
{
public:
	/// The list of existing states specific to the DataLogger
	enum
	{
		kSM_DailyOpen = 20, ///< Daily file openned and writing
		kSM_WaitingRun = 30, ///< waiting for the run number to open the run file
		kSM_Logging = 40, ///< both files openned and writing
		kSM_BadDailyConfig = 0x101, ///< the folder specified for daily logging does not exist or has bad permissions
		kSM_BadRunConfig = 0x102, ///<  the folder specified for the run logging does not exist or has wrong permissions or no run number
	} localstates_t;
	
        DataLogger(std::ostream &out);
	~DataLogger(); 
	
private:
	//Define all the data structure specific to the DataLogger here
 	/// ofstream for the dailyLogfile
	std::ofstream fDailyLogFile;
	/// ofstream for the run-specific Log file
	std::ofstream fRunLogFile; 

	/// ofstream for the daily report file
	std::ofstream fDailyReportFile;
	/// ofstream for the run-specific report file
	std::ofstream fRunReportFile;
	/// base path of the dailyfile
	std::string fDailyFileName; 
	///base path of the run file
	std::string fRunFileName; 
	///run number (-1 means no run number specified)
	int fRunNumber; 
	///Current Service Quality
	int fQuality;
	///Modified Julian Date
	double fMjD;
	
	///Define all the static names
	static const char* fConfigDay;
	static const char* fConfigRun;
	static const char* fConfigRunNumber;
	static const char* fConfigLog;
	static const char* fTransStart;
	static const char* fTransStop;
	static const char* fTransStartRun;
	static const char* fTransStopRun;
	static const char* fTransReset;
	static const char* fTransWait;
	static const char* fRunNumberInfo; ///< This is the name of the dimInfo received to specify the run number. It must be updated once the final name will be defined
	
	//overloading of DIM's infoHandler function
	void infoHandler(); 
	
	///for obtaining the name of the existing services
	ServiceList fServiceList;
	
	
	///A std pair to store both the DimInfo name and the actual DimInfo pointer 
//	typedef std::pair<std::string, DimStampedInfo*> subscriptionType; 
	///All the services to which we've subscribed to. Sorted by server name
//	std::map<const std::string, std::vector<subscriptionType > > fServiceSubscriptions; 

	///A std pair to store both the DimInfo pointer and the corresponding outputted fits file
	struct SubscriptionType
	{ 
#ifdef HAS_FITS
		///daily FITS output file
		Fits	dailyFile;
		///run-specific FITS output file
		Fits	runFile;
#endif
		///the actual dimInfo pointer
		DimStampedInfo* dimInfo;
		///the converter for outputting the data according to the format
		Converter* fConv;
		///the number of existing handlers to this structure.
		///This is required otherwise I MUST handle the deleting of dimInfo outside from the destructor
		int* numCopies;
		void operator = (const SubscriptionType& other)
		{
#ifdef HAS_FITS
			dailyFile = other.dailyFile;
			runFile = other.runFile;
#endif
			dimInfo = other.dimInfo;	
			numCopies = other.numCopies;
			fConv = other.fConv;
			(*numCopies)++;
		}
		SubscriptionType(const SubscriptionType& other)
		{
#ifdef HAS_FITS
			dailyFile = other.dailyFile;
			runFile = other.runFile;
#endif
			dimInfo = other.dimInfo;
			numCopies = other.numCopies;
			fConv = other.fConv;
			(*numCopies)++;	
		}
		SubscriptionType(DimStampedInfo* info)
		{
			dimInfo = info;	
			fConv = NULL;
			numCopies = new int(1);
		}
		SubscriptionType()
		{
			dimInfo = NULL;
			fConv = NULL;
			numCopies = new int(1);
		}
		~SubscriptionType()
		{
			if (numCopies)
			(*numCopies)--;
			if (numCopies)
			if (*numCopies < 1)
			{
				if (dimInfo)
				delete dimInfo;
#ifdef HAS_FITS
				if (dailyFile.IsOpen())
					dailyFile.Close();
				if (runFile.IsOpen())
					runFile.Close();
#endif
				if (numCopies)	
				delete numCopies;
				delete fConv;
				fConv = NULL;
				dimInfo = NULL;
				numCopies = NULL;
			}
		}
	};
	typedef std::map<const std::string, std::map<std::string, SubscriptionType>> SubscriptionsListType;
	///All the services to which we have subscribed to, sorted by server name.
	SubscriptionsListType fServiceSubscriptions;


	///Reporting method for the services info received
	void ReportPlease(DimInfo* I, SubscriptionType& sub);  

	///Configuration of the daily file path
	int ConfigureDailyFileName(const Event& evt); 
	///Configuration fo the file name
	int ConfigureRunFileName(const Event& evt); 
	///DEPREC - configuration of the run number
	int ConfigureRunNumber(const Event& evt); 
	///logging method for the messages
	int LogMessagePlease(const Event& evt); 
	///checks whether or not the current info being treated is a run number
	void CheckForRunNumber(DimInfo* I); 
	/// start transition
	int StartPlease(); 
	///from waiting to logging transition
	int StartRunPlease(); 
	/// from logging to waiting transition
	int StopRunPlease(); 
	///stop and reset transition
	int GoToReadyPlease(); 
	///from dailyOpen to waiting transition
	int DailyToWaitRunPlease(); 
#ifdef HAS_FITS
	///Open fits files
	void OpenFITSFilesPlease(SubscriptionType& sub);
	///Write data to FITS files
	void WriteToFITS(SubscriptionType& sub);
	///Allocate the buffers required for fits
	void AllocateFITSBuffers(SubscriptionType& sub);
		
#ifdef ONE_RUN_FITS_ONLY
	///FITS file for runs. only one, hence dealt with in the dataLogger itself
	FITS* fRunFitsFile;
#endif //one_run_fits_only
#endif//has_fits
public:	
	///checks with fServiceList whether or not the services got updated
	bool CheckForServicesUpdate(); 

private:	
	///monitoring notification loop
	void ServicesMonitoring();
	///services notification thread
	boost::thread fMonitoringThread;
	///end of the monitoring
	bool fContinueMonitoring;
	///required for accurate monitoring
	std::map<std::string, long long> fFileSizesMap;
	std::string fFullDailyLogFileName;
	std::string fFullDailyReportFileName;
	std::string fFullRunLogFileName;
	std::string fFullRunReportFileName;
	long long fBaseSizeDaily;
	long long fPreviousSize;
	long long fBaseSizeRun;
	///Service for opened files
	DimDescribedService* fOpenedDailyFiles;
	DimDescribedService* fOpenedRunFiles;
	DimDescribedService* fNumSubAndFits;
	NumSubAndFitsType fNumSubAndFitsData;
//	char* fDimBuffer;
	inline void NotifyOpenedFile(std::string name, int type, DimService* service);
	inline void NotifyOpenedFile(std::string , int , MyService* ){}
	
}; //DataLogger

//static members initialization
//since I do not check the transition/config names any longer, indeed maybe these could be hard-coded... but who knows what will happen in the future ?
const char* DataLogger::fConfigDay = "CONFIG_DAY";
const char* DataLogger::fConfigRun = "CONFIG_RUN";
const char* DataLogger::fConfigRunNumber = "CONFIG_RUN_NUMBER";
const char* DataLogger::fConfigLog = "LOG";
const char* DataLogger::fTransStart = "START";
const char* DataLogger::fTransStop = "STOP";
const char* DataLogger::fTransStartRun = "START_RUN";
const char* DataLogger::fTransStopRun = "STOP_RUN";
const char* DataLogger::fTransReset = "RESET";
const char* DataLogger::fTransWait = "WAIT_RUN_NUMBER";
const char* DataLogger::fRunNumberInfo = "RUN_NUMBER";


void DataLogger::ServicesMonitoring()
{
		//create the DIM service
		int dataSize = 2*sizeof(long long) + sizeof(long);

		DataLoggerStats statVar;
		statVar.sizeWritten = 0;
		statVar.freeSpace = 0;
		statVar.writingRate = 0;

		struct statvfs vfs;
		if (!statvfs(fDailyFileName.c_str(), &vfs))
			statVar.freeSpace = vfs.f_bsize*vfs.f_bavail;
		else
			statVar.freeSpace = -1;
//        DimDescribedService srvc((GetName()+"/STATS").c_str(), "x:2;l:1", &statVar, dataSize,//static_cast<void*>(data), dataSize,
//                                        "Add description here");
		DimDescribedService srvc ("DATA_LOGGER/STATS", "X:2;L:1", &statVar, dataSize, "Add description here");
		double deltaT = 1;
		fPreviousSize = 0;
		bool statWarning = false;
		//loop-wait for broadcast
		while (fContinueMonitoring)
		{
			sleep(deltaT);
			//update the fits files sizes
#ifdef HAS_FITS
			SubscriptionsListType::iterator x;
			std::map<std::string, SubscriptionType>::iterator y;
			bool runFileDone = false;
			for (x=fServiceSubscriptions.begin(); x != fServiceSubscriptions.end(); x++)
			{
				for (y=x->second.begin(); y != x->second.end(); y++)
				{
					if (y->second.runFile.IsOpen() && !runFileDone)
					{
							fFileSizesMap[y->second.runFile.fFileName] = y->second.runFile.GetWrittenSize();
							runFileDone = true;
					}
					if (y->second.dailyFile.IsOpen())
						fFileSizesMap[y->second.dailyFile.fFileName] = y->second.dailyFile.GetWrittenSize();
				}
			}
#endif
			struct stat st;
			//gather log and report files sizes on disk
			if (fDailyLogFile.is_open())
			{
				stat(fFullDailyLogFileName.c_str(), &st);
				fFileSizesMap[fFullDailyLogFileName] = st.st_size;	
			}
			if (fDailyReportFile.is_open())
			{
				stat(fFullDailyReportFileName.c_str(), &st);
				fFileSizesMap[fFullDailyReportFileName] = st.st_size;	
			}
			if (fRunLogFile.is_open())
			{
				stat(fFullRunLogFileName.c_str(), &st);
				fFileSizesMap[fFullRunLogFileName] = st.st_size;	
			}
			if (fRunReportFile.is_open())
			{
				stat(fFullRunReportFileName.c_str(), &st);
				fFileSizesMap[fFullRunReportFileName] = st.st_size;
			}	

			if (!statvfs(fDailyFileName.c_str(), &vfs))
			{
				statVar.freeSpace = vfs.f_bsize*vfs.f_bavail;
				statWarning = false;
			}
			else
			{
				std::stringstream str;
				str << "Unable to retrieve stats for " << fDailyFileName << ". Reason: " << strerror(errno) << " [" << errno << "]";
				if (!statWarning)
					Error(str);
				statWarning = true;
				statVar.freeSpace = -1;
			}
			
			//sum up all the file sizes. past and present
			statVar.sizeWritten = 0;
			for (std::map<std::string, long long>::iterator it=fFileSizesMap.begin(); it != fFileSizesMap.end();  it++)
				statVar.sizeWritten += it->second;
			statVar.sizeWritten -= fBaseSizeDaily;
			statVar.sizeWritten -= fBaseSizeRun;
			//FIXME get the actual time elapsed
			statVar.writingRate = (statVar.sizeWritten - fPreviousSize)/deltaT;  
			fPreviousSize = statVar.sizeWritten;
			if (statVar.writingRate != 0) //if data has been written
			{
//std::cout << "rate: " << statVar.writingRate << std::endl;
				srvc.updateService(&statVar, dataSize);//static_cast<void*>(data), dataSize);
			}
		}
}


// --------------------------------------------------------------------------
//
//! Default constructor. The name of the machine is given DATA_LOGGER
//! and the state is set to kSM_Ready at the end of the function.
//
//!Setup the allows states, configs and transitions for the data logger
//
DataLogger::DataLogger(std::ostream &out) : StateMachineDim(out, "DATA_LOGGER")
{	
		//initialize member data
		fDailyFileName = ".";//"/home/lyard/log";//
		fRunFileName = ".";//"/home/lyard/log";
		fRunNumber = 12345;
#ifdef HAS_FITS
#ifdef ONE_RUN_FITS_ONLY
		fRunFitsFile = NULL;
#endif
#endif
		//Give a name to this machine's specific states
		AddStateName(kSM_DailyOpen,      "DailyFileOpen",  "Add description here");
		AddStateName(kSM_WaitingRun,     "WaitForRun",     "Add description here");
		AddStateName(kSM_Logging,        "Logging",        "Add description here");
		AddStateName(kSM_BadDailyConfig, "ErrDailyFolder", "Add description here");
		AddStateName(kSM_BadRunConfig,   "ErrRunFolder",   "Add description here");

        /*Add the possible transitions for this machine*/
                AddTransition(kSM_DailyOpen, fTransStart, kSM_Ready, kSM_BadDailyConfig)
                    (boost::bind(&DataLogger::StartPlease, this))
                    ("start the daily logging. daily file location must be specified already");

		AddTransition(kSM_Ready, fTransStop, kSM_DailyOpen, kSM_WaitingRun, kSM_Logging)
                    (boost::bind(&DataLogger::GoToReadyPlease, this))
                   ("stop the data logging");

                AddTransition(kSM_Logging, fTransStartRun, kSM_WaitingRun, kSM_BadRunConfig)
                    (boost::bind(&DataLogger::StartRunPlease, this))
                    ("start the run logging. run file location must be specified already.");

                AddTransition(kSM_WaitingRun, fTransStopRun, kSM_Logging)
                    (boost::bind(&DataLogger::StopRunPlease, this))
                    ("");

                AddTransition(kSM_Ready, fTransReset, kSM_Error, kSM_BadDailyConfig, kSM_BadRunConfig, kSM_Error)
                    (boost::bind(&DataLogger::GoToReadyPlease, this))
                    ("transition to exit error states. dunno if required or not, would close the daily file if already openned.");

                AddTransition(kSM_WaitingRun, fTransWait, kSM_DailyOpen)
                    (boost::bind(&DataLogger::DailyToWaitRunPlease, this));

        /*Add the possible configurations for this machine*/
        
                AddConfiguration(fConfigDay, "C", kSM_Ready, kSM_BadDailyConfig)
                    (boost::bind(&DataLogger::ConfigureDailyFileName, this, _1))
                    ("configure the daily file location. cannot be done before the file is actually opened");

                AddConfiguration(fConfigRun, "C", kSM_Ready, kSM_BadDailyConfig, kSM_DailyOpen, kSM_WaitingRun, kSM_BadRunConfig)
                    (boost::bind(&DataLogger::ConfigureRunFileName, this, _1))
                    ("configure the run file location. cannot be done before the file is actually opened, and not in a dailly related error.");

		//Provide a logging command
		//I get the feeling that I should be going through the EventImp 
                //instead of DimCommand directly, mainly because the commandHandler
                //is already done in StateMachineImp.cc
                //Thus I'll simply add a configuration, which I will treat as the logging command
                AddConfiguration(fConfigLog, "C", kSM_DailyOpen, kSM_Logging, kSM_WaitingRun, kSM_BadRunConfig)
                    (boost::bind(&DataLogger::LogMessagePlease, this, _1));

 		fServiceList.SetHandler(this);
		CheckForServicesUpdate();
		
		//start the monitoring service
		fContinueMonitoring = true;
		fMonitoringThread = boost::thread(boost::bind(&DataLogger::ServicesMonitoring, this));
		fBaseSizeDaily = 0;
		fBaseSizeRun = 0;

		//gives the entire buffer size. Otherwise, dim overwrites memory at bizarre locations if smaller size is given at creation time.
//        fOpenedFiles = 	new DimDescribedService((GetName()+"/FILENAME").c_str(), "i:1;C", static_cast<void*>(fDimBuffer), 256, "Add description here");
		fOpenedDailyFiles = new DimDescribedService((GetName() + "/FILENAME_DAILY").c_str(), const_cast<char*>(""), "Add description here");//"i:1;C", static_cast<void*>(fDimBuffer), 256);
		fOpenedRunFiles = new DimDescribedService((GetName() + "/FILENAME_RUN").c_str(), const_cast<char*>(""), "Add description here");
		fOpenedDailyFiles->setQuality(0);
		fOpenedRunFiles->setQuality(0);
		fOpenedDailyFiles->updateService();
		fOpenedRunFiles->updateService();
		fNumSubAndFitsData.numSubscriptions = 0;
		fNumSubAndFitsData.numOpenFits = 0;
		fNumSubAndFits = new DimDescribedService((GetName() + "/NUM_SUBS").c_str(), "I:2", &fNumSubAndFitsData, sizeof(NumSubAndFitsType), "Add description here");

}
// --------------------------------------------------------------------------
//
//! Checks for changes in the existing services.
//! Any new service will be added to the service list, while the ones which disappeared are removed.
//! @todo
//! 	add the configuration (using the conf class ?)
//
//FIXME The service must be udpated so that I get the first notification. This should not be
bool DataLogger::CheckForServicesUpdate()
{ 
	bool serviceUpdated = false;
	//get the current server list
	const std::vector<std::string> serverList = fServiceList.GetServerList();
	//first let's remove the servers that may have disapeared
	//can't treat the erase on maps the same way as for vectors. Do it the safe way instead
	std::vector<std::string> toBeDeleted;
	for (SubscriptionsListType::iterator cListe = fServiceSubscriptions.begin(); cListe != fServiceSubscriptions.end(); cListe++)
	{
		std::vector<std::string>::const_iterator givenServers;
		for (givenServers=serverList.begin(); givenServers!= serverList.end(); givenServers++)
			if (cListe->first == *givenServers)
				break;
		if (givenServers == serverList.end())//server vanished. Remove it
		{	
			toBeDeleted.push_back(cListe->first);
			serviceUpdated = true;
		}
			
	} 
	for (std::vector<std::string>::const_iterator it = toBeDeleted.begin(); it != toBeDeleted.end(); it++)
		fServiceSubscriptions.erase(*it);
	//now crawl through the list of servers, and see if there was some updates
	for (std::vector<std::string>::const_iterator i=serverList.begin(); i!=serverList.end();i++)
	{
		//skip the two obvious excluded services
		if ((i->find("DIS_DNS") != std::string::npos) ||
		    (i->find("DATA_LOGGER") != std::string::npos))
			continue;
		//find the current server in our subscription list	
		SubscriptionsListType::iterator cSubs = fServiceSubscriptions.find(*i);
		//get the service list of the current server
		std::vector<std::string> cServicesList = fServiceList.GetServiceList(*i);
		if (cSubs != fServiceSubscriptions.end())//if the current server already is in our subscriptions
		{										 //then check and update our list of subscriptions
			//first, remove the services that may have dissapeared.
			std::map<std::string, SubscriptionType>::iterator serverSubs;
			std::vector<std::string>::const_iterator givenSubs;
			toBeDeleted.clear();
			for (serverSubs=cSubs->second.begin(); serverSubs != cSubs->second.end(); serverSubs++)
			{
				for (givenSubs = cServicesList.begin(); givenSubs != cServicesList.end(); givenSubs++)
					if (serverSubs->first == *givenSubs)
						break;
				if (givenSubs == cServicesList.end())
				{
					toBeDeleted.push_back(serverSubs->first);
					serviceUpdated = true;
				}	
			}
			for (std::vector<std::string>::const_iterator it = toBeDeleted.begin(); it != toBeDeleted.end(); it++)
				cSubs->second.erase(*it);
			//now check for new services
			for (givenSubs = cServicesList.begin(); givenSubs != cServicesList.end(); givenSubs++)
			{
				if (*givenSubs == "SERVICE_LIST")
					continue;
				if (cSubs->second.find(*givenSubs) == cSubs->second.end())
				{//service not found. Add it
					cSubs->second[*givenSubs].dimInfo = new DimStampedInfo(((*i) + "/" + *givenSubs).c_str(), const_cast<char*>(""), this);
					serviceUpdated = true;
				}	
			}
		}
		else //server not found in our list. Create its entry
		{
			fServiceSubscriptions[*i] = std::map<std::string, SubscriptionType>();
			std::map<std::string, SubscriptionType>& liste = fServiceSubscriptions[*i];
			for (std::vector<std::string>::const_iterator j = cServicesList.begin(); j!= cServicesList.end(); j++)
			{
				if (*j == "SERVICE_LIST")
					continue;
				liste[*j].dimInfo = new DimStampedInfo(((*i) + "/" + (*j)).c_str(), const_cast<char*>(""), this);
				serviceUpdated = true;
			}
		}	
	}
	return serviceUpdated;
}
// --------------------------------------------------------------------------
//
//! Destructor
//
DataLogger::~DataLogger()
{
	//release the services subscriptions
	fServiceSubscriptions.clear();
	//exit the monitoring loop
	fContinueMonitoring = false;
//	delete[] fDimBuffer;
	fMonitoringThread.join();
	//close the files
	if (fDailyLogFile.is_open())
		fDailyLogFile.close();
	if (fDailyReportFile.is_open())
		fDailyReportFile.close();
	if (fRunLogFile.is_open())
		fRunLogFile.close();
	if (fRunReportFile.is_open())
		fRunReportFile.close();
	delete fOpenedDailyFiles;
	delete fOpenedRunFiles;
	delete fNumSubAndFits;
//TODO notify that all files were closed
#ifdef HAS_FITS
#ifdef ONE_RUN_FITS_ONLY
	if (fRunFitsFile != NULL)
		delete fRunFitsFile;
	fRunFitsFile = NULL;
#endif
#endif
}

// --------------------------------------------------------------------------
//
//! Inherited from DimInfo. Handles all the Infos to which we subscribed, and log them
//
void DataLogger::infoHandler()
{
	DimInfo* I = getInfo();
	SubscriptionsListType::iterator x;
	std::map<std::string, SubscriptionType>::iterator y;
	if (I==NULL)
	{
		if (CheckForServicesUpdate())
		{
			//services were updated. Notify
			fNumSubAndFitsData.numSubscriptions = 0;
			for (x=fServiceSubscriptions.begin(); x != fServiceSubscriptions.end(); x++)
				fNumSubAndFitsData.numSubscriptions += x->second.size();
			fNumSubAndFits->updateService();
		}
		return;
	}
	//check if the service pointer corresponds to something that we subscribed to
	//this is a fix for a bug that provides bad Infos when a server starts
	bool found = false;
	for (x=fServiceSubscriptions.begin(); x != fServiceSubscriptions.end(); x++)
	{//find current service is subscriptions
		for (y=x->second.begin(); y!=x->second.end();y++)
			if (y->second.dimInfo == I)
			{
				found = true;	
				break;
			}
		if (found)
			break;
	}
	if (!found)
		return;
	if (I->getSize() <= 0)
		return;
	//check that the message has been updated by something, i.e. must be different from its initial value
	if (I->getTimestamp() == 0)
		return;

	CheckForRunNumber(I);
	ReportPlease(I, y->second);

}

// --------------------------------------------------------------------------
//
//! Checks whether or not the current info is a run number.
//! If so, then remember it. A run number is required to open the run-log file
//! @param I
//!		the current DimInfo
//
void DataLogger::CheckForRunNumber(DimInfo* I)
{
	if (strstr(I->getName(), fRunNumberInfo) != NULL)
	{//assumes that the run number is an integer
		//TODO check the format here
		fRunNumber = I->getInt();	
	}
}

// --------------------------------------------------------------------------
//
//! write infos to log files.
//! @param I
//! 	The current DimInfo 
//
void DataLogger::ReportPlease(DimInfo* I, SubscriptionType& sub)
{
	//should we log or report this info ? (i.e. is it a message ?)
	bool isItaReport = ((strstr(I->getName(), "Message") == NULL) && (strstr(I->getName(), "MESSAGE") == NULL));
	
	//TODO add service exclusion
	
	if (!fDailyReportFile.is_open())
		return;

	//create the converter for that service
	if (sub.fConv == NULL)
	{
		sub.fConv = new Converter(Out(), I->getFormat());	
		if (!sub.fConv)
		{
			std::stringstream str;
			str << "Couldn't properly parse the format... service " << sub.dimInfo->getName() << " ignored.";
			Error(str);
			return;	
		}
	}
		
	//construct the header
	std::stringstream header;
	Time cTime(I->getTimestamp(), I->getTimestampMillisecs()*1000);
	fQuality = I->getQuality();
	fMjD = cTime.Mjd();

	if (isItaReport)
	{
		//write text header
		header << I->getName() << " " << fQuality << " ";
		header << cTime.Y() << " " << cTime.M() << " " << cTime.D() << " ";
		header << cTime.h() << " " << cTime.m() << " " << cTime.s() << " ";
		header << cTime.ms() << " " << I->getTimestamp() << " ";

		std::string text;
        try
        {
        	text = sub.fConv->GetString(I->getData(), I->getSize());
        }
        catch (const std::runtime_error &e)
        {
        	Out() << kRed << e.what() << endl;
        	std::stringstream str;
        	str << "Could not properly parse the data for service " << sub.dimInfo->getName();
        	str << " reason: " << e.what() << ". Entry ignored";
            Error(str);
            return;
        }

		if (text.empty())
		{
			std::stringstream str;
			str << "Service " << sub.dimInfo->getName() << " sent an empty string";
			Info(str);
        	return;
		}
        //replace bizarre characters by white space
        replace(text.begin(), text.end(), '\n', '\\');
        replace_if(text.begin(), text.end(), std::ptr_fun<int, int>(&std::iscntrl), ' ');
        
        //write entry to daily report
		try
		{
			if (fDailyReportFile.is_open())
				fDailyReportFile << header.str() << text << std::endl;
		}
		catch (std::exception e)
		{
			std::stringstream str;
			str << "Error while writing to daily report file: " << e.what();
			Error(str);	
		}
		//write entry to run-report
		try
		{
			if (fRunReportFile.is_open())
				fRunReportFile << header.str() << text << std::endl;
		}
		catch (std::exception e)
		{
			std::stringstream str;
			str << "Error while writing to run report file: " << e.what();
			Error(str);
		}
	}
	else
	{//write entry to both daily and run logs
		std::string n = I->getName();
		std::stringstream msg;
		msg << n.substr(0, n.find_first_of('/')) << ": " << I->getString();
		try
		{
			MessageImp dailyMess(fDailyLogFile);
			dailyMess.Write(cTime, msg.str().c_str(), fQuality);
		}
		catch (std::exception e)
		{
			std::stringstream str;
			str << "Error while writing to daily log file: " << e.what();
			Error(str);	
		}
		if (fRunLogFile.is_open())
		{
			try 
			{
				MessageImp runMess(fRunLogFile);
				runMess.Write(cTime, msg.str().c_str(), fQuality);
			}
			catch (std::exception e)
			{
				std::stringstream str;
				str << "Error while writing to run log file: " << e.what();
				Error(str);	
			}
		}
	}

#ifdef HAS_FITS
	if (!sub.dailyFile.IsOpen() || !sub.runFile.IsOpen())
		OpenFITSFilesPlease(sub);	
	WriteToFITS(sub);
		
#endif

}

// --------------------------------------------------------------------------
//
//! write messages to logs. 
//! @param evt
//!		the current event to log
//! @returns 
//!		the new state. Currently, always the current state
//!
//! @deprecated
//!    I guess that this function should not be any longer
//
//TODO isn't that function not used any longer ? If so I guess that we should get rid of it...
//Otherwise re-write it properly with the MessageImp class
int DataLogger::LogMessagePlease(const Event& evt)
{
	if (!fDailyLogFile.is_open())
		return GetCurrentState();
	
	std::stringstream header;
	const Time& cTime = evt.GetTime();
	header << evt.GetName() << " " << cTime.Y() << " " << cTime.M() << " " << cTime.D() << " ";
	header << cTime.h() << " " << cTime.m() << " " << cTime.s() << " ";
	header << cTime.ms() << " ";
		
    const Converter conv(Out(), evt.GetFormat());
    if (!conv)
    {
    	Error("Couldn't properly parse the format... ignored.");
        return GetCurrentState();
    }

    std::string text;
    try
    {
    	text = conv.GetString(evt.GetData(), evt.GetSize());
    }
    catch (const std::runtime_error &e)
    {
    	Out() << kRed << e.what() << endl;
        Error("Couldn't properly parse the data... ignored.");
        return GetCurrentState();
    }

	if (text.empty())
        return GetCurrentState();

    //replace bizarre characters by white space
    replace(text.begin(), text.end(), '\n', '\\');
    replace_if(text.begin(), text.end(), std::ptr_fun<int, int>(&std::iscntrl), ' ');

	if (fDailyLogFile.is_open())
		fDailyLogFile << header;
	if (fRunLogFile.is_open())
		fRunLogFile << header;

	if (fDailyLogFile.is_open())
		fDailyLogFile << text;
	if (fRunLogFile.is_open())
		fRunLogFile << text;

	return GetCurrentState();
}
// --------------------------------------------------------------------------
//
//!	Sets the path to use for the daily log file.
//! @param evt
//! 	the event transporting the path
//! @returns
//!		currently only the current state.
//
int DataLogger::ConfigureDailyFileName(const Event& evt)
{
	if (evt.GetText() != NULL)
	{
		fDailyFileName = std::string(evt.GetText());	
		Message("New daily folder specified: " + fDailyFileName);
	}
	else
		Error("Empty daily folder given. Please specify a valid path.");

	return GetCurrentState();
}
// --------------------------------------------------------------------------
//
//! Sets the path to use for the run log file.
//! @param evt
//!		the event transporting the path
//! @returns
//! 	currently only the current state
int DataLogger::ConfigureRunFileName(const Event& evt)
{
	if (evt.GetText() != NULL)
	{
		fRunFileName = std::string(evt.GetText());
		Message("New Run folder specified: " + fRunFileName);
	}
	else
		Error("Empty daily folder given. Please specify a valid path");

	return GetCurrentState();
}
// --------------------------------------------------------------------------
//
//! Sets the run number.
//! @param evt
//!		the event transporting the run number
//! @returns
//! 	currently only the current state
//TODO remove this function as the run numbers will be distributed through a dedicated service
int DataLogger::ConfigureRunNumber(const Event& evt)
{
	fRunNumber = evt.GetInt();
	return GetCurrentState();
}
// --------------------------------------------------------------------------
//
//! Notifies the DIM service that a particular file was opened
//! @ param name the base name of the opened file, i.e. without path nor extension. 
//! 	WARNING: use string instead of string& because I pass values that do not convert to string&.
//!		this is not a problem though because file are not opened so often.
//! @ param type the type of the opened file. 0 = none open, 1 = log, 2 = text, 4 = fits
inline void DataLogger::NotifyOpenedFile(std::string name, int type, DimService* service)
{
	service->setQuality(type);
	service->updateService(const_cast<char*>(name.c_str()));
}
// --------------------------------------------------------------------------
//
//! Implements the Start transition.
//! Concatenates the given path for the daily file and the filename itself (based on the day), 
//! and tries to open it.
//! @returns 
//!		kSM_DailyOpen if success, kSM_BadDailyConfig if failure
int DataLogger::StartPlease()
{
	//TODO concatenate the dailyFileName and the formatted date and extension to obtain the full file name
	Time time;
	std::stringstream sTime;
	sTime << time.Y() << "_" << time.M() << "_" << time.D();

	fFullDailyLogFileName = fDailyFileName + '/' + sTime.str() + ".log"; 
	fDailyLogFile.open(fFullDailyLogFileName.c_str(), std::ios_base::out | std::ios_base::app); 
	if (errno != 0)
	{
		std::stringstream str;
		str << "Unable to open daily Log " << fFullDailyLogFileName << ". Reason: " << strerror(errno) << " [" << errno << "]";
		Error(str);	
	}
	fFullDailyReportFileName = fDailyFileName + '/' + sTime.str() + ".rep";
	fDailyReportFile.open(fFullDailyReportFileName.c_str(), std::ios_base::out | std::ios_base::app);
	if (errno != 0)
	{
		std::stringstream str;
		str << "Unable to open daily Report " << fFullDailyReportFileName << ". Reason: " << strerror(errno) << " [" << errno << "]";
		Error(str);	
	}

	if (!fDailyLogFile.is_open() || !fDailyReportFile.is_open())
	{	
		//TODO send an error message	
	    return kSM_BadDailyConfig;
	}
	//get the size of the newly opened file.
	struct stat st;
	stat(fFullDailyLogFileName.c_str(), &st);
	fBaseSizeDaily = st.st_size;	
	stat(fFullDailyReportFileName.c_str(), &st);
	fBaseSizeDaily += st.st_size;
	fFileSizesMap.clear();
	fBaseSizeRun = 0;
	fPreviousSize = 0;
	//notify that files were opened
	std::string actualTargetDir;
	if (fDailyFileName == ".")
	{
		char currentPath[FILENAME_MAX];
		getcwd(currentPath, sizeof(currentPath));
		if (errno != 0)
		{
			std::stringstream str;
			str << "Unable retrieve current path" << ". Reason: " << strerror(errno) << " [" << errno << "]";
			Error(str);	
		}
		actualTargetDir = currentPath;
	}
	else
	{
		actualTargetDir = fDailyFileName;	
	}

	NotifyOpenedFile(actualTargetDir + '/' + sTime.str(), 3, fOpenedDailyFiles);
	
	
	return kSM_DailyOpen; 	
}

#ifdef HAS_FITS
// --------------------------------------------------------------------------
//
//! open if required a the FITS files corresponding to a given subscription
//! @param sub
//! 	the current DimInfo subscription being examined
void DataLogger::OpenFITSFilesPlease(SubscriptionType& sub)
{
	std::string serviceName(sub.dimInfo->getName());
	for (unsigned int i=0;i<serviceName.size(); i++)
	{
		if (serviceName[i] == '/')
		{
			serviceName[i] = '_';
			break;	
		}	
	}
	Time time;
	std::stringstream sTime;
	sTime << time.Y() << "_" << time.M() << "_" << time.D();
	//we open the dailyFile anyway, otherwise this function shouldn't have been called.
	if (!sub.dailyFile.IsOpen())
	{
		std::string partialName = fDailyFileName + '/' + sTime.str() + '_' + serviceName + ".fits";
		AllocateFITSBuffers(sub);
		//get the size of the file we're about to open
		if (fFileSizesMap.find(partialName) == fFileSizesMap.end())
		{
			struct stat st;
			if (!stat(partialName.c_str(), &st))
				fBaseSizeDaily += st.st_size;
			fFileSizesMap[partialName] = 0;
		}
		sub.dailyFile.Open(partialName, serviceName, NULL, &fNumSubAndFitsData.numOpenFits, Out());
		//notify the opening
		std::string actualTargetDir;
		if (fDailyFileName == ".")
		{
			char currentPath[FILENAME_MAX];
			getcwd(currentPath, sizeof(currentPath));
			actualTargetDir = currentPath;
		}
		else
		{
			actualTargetDir = fDailyFileName;	
		}		
		NotifyOpenedFile(actualTargetDir + '/' + sTime.str(), 4, fOpenedDailyFiles);
		fNumSubAndFits->updateService();
	}
	if (!sub.runFile.IsOpen() && (GetCurrentState() == kSM_Logging))
	{//buffer for the run file have already been allocated when doing the daily file
		std::stringstream sRun;
		sRun << fRunNumber;
#ifdef ONE_RUN_FITS_ONLY
		std::string partialName = fRunFileName + '/' + sRun.str() + ".fits";//'_' + serviceName + ".fits";
		if (fRunFitsFile == NULL)
		{
#else
		std::string partialName = fRunFileName + '/' + sRun.str() + '_' + serviceName + ".fits";
#endif
			//get the size of the file we're about to open
			if (fFileSizesMap.find(partialName) == fFileSizesMap.end())
			{
				struct stat st;
				if (!stat(partialName.c_str(), &st))
					fBaseSizeRun += st.st_size;
				else
					fBaseSizeRun = 0;
				fFileSizesMap[partialName] = 0;	
			}
#ifdef ONE_RUN_FITS_ONLY
			try
			{
				fRunFitsFile = new FITS(partialName, RWmode::Write);	
				(fNumSubAndFitsData.numOpenFits)++;
			}	
			catch (CCfits::FitsError e)
			{
				std::stringstream str;
				str << "Could not open FITS Run file " << partialName << " reason: " << e.message();
				Error(str);
				fRunFitsFile = NULL;
			}
#endif
			std::string actualTargetDir;
			if (fRunFileName == ".")
			{
				char currentPath[FILENAME_MAX];
				getcwd(currentPath, sizeof(currentPath));
				actualTargetDir = currentPath;
			}
			else
			{
				actualTargetDir = fRunFileName;	
			}		
			NotifyOpenedFile(actualTargetDir + '/' + sRun.str(), 4, fOpenedRunFiles);// + '_' + serviceName, 4);
#ifdef ONE_RUN_FITS_ONLY
		}
		sub.runFile.Open(partialName, serviceName, fRunFitsFile, &fNumSubAndFitsData.numOpenFits, Out());
#else
		sub.runFile.Open(partialName, serviceName, NULL, &fNumSubAndFitsData.numOpenFits, Out());
#endif //one_run_fits_only
	   fNumSubAndFits->updateService();

	}
}	
// --------------------------------------------------------------------------
//
void DataLogger::AllocateFITSBuffers(SubscriptionType& sub)
{
	int size = sub.dimInfo->getSize();
	 
	//Init the time columns of the file
	Description dateDesc(std::string("Time"), std::string("Modified Julian Date"), std::string("MjD"));
	sub.dailyFile.AddStandardColumn(dateDesc, "1D", &fMjD, sizeof(double));
	sub.runFile.AddStandardColumn(dateDesc, "1D", &fMjD, sizeof(double));

	Description QoSDesc("Qos", "Quality of service", "None");
	sub.dailyFile.AddStandardColumn(QoSDesc, "1J", &fQuality, sizeof(int));
	sub.runFile.AddStandardColumn(QoSDesc, "1J", &fQuality, sizeof(int));

	const Converter::FormatList flist = sub.fConv->GetList();
    // Compilation failed
    if (flist.empty() || flist.back().first.second!=0)
    {
    	Error("Compilation of format string failed.");
        return;
    }

	//we've got a nice structure describing the format of this service's messages.
	//Let's create the appropriate FITS columns
	std::vector<std::string> dataFormatsLocal;
	for (unsigned int i=0;i<flist.size()-1;i++)
	{
	 	std::stringstream dataQualifier; 

	 	dataQualifier << flist[i].second.first;
	 	switch (flist[i].first.first->name()[0])
	 	{//TODO handle all the data format cases
	 		case 'c':
	 			dataQualifier <<  "S";
	 		break;
	 		case 's':
	 			dataQualifier << "I";
	 		break;
	 		case 'i':
	 			dataQualifier << "J";
	 		break;
	 		case 'l':
	 			dataQualifier << "J";
	 			//TODO triple check that in FITS, long = int
	 		break;
	 		case 'f':
	 			dataQualifier << "E";
	 		break;
	 		case 'd':
	 			dataQualifier << "D";
	 		break;
	 		case 'x':
	 		case 'X':
	 			dataQualifier << "K";
	 		break;
	 		case 'S':
	 			//for strings, the number of elements I get is wrong. Correct it
	 			dataQualifier.str(""); //clear
	 			dataQualifier << size-1 <<  "A";
	 			size = size-1;
	 		break;
	 		
	 		default:
	 			Error("THIS SHOULD NEVER BE REACHED. dataLogger.cc ln 948.");
	 	};
	 	dataFormatsLocal.push_back(dataQualifier.str());
	 }

	 sub.dailyFile.InitDataColumns(fServiceList.GetDescriptions(sub.dimInfo->getName()), dataFormatsLocal, sub.dimInfo->getData(), size);
	 sub.runFile.InitDataColumns(fServiceList.GetDescriptions(sub.dimInfo->getName()), dataFormatsLocal, sub.dimInfo->getData(), size);
}
// --------------------------------------------------------------------------
//
//! write a dimInfo data to its corresponding FITS files
//
void DataLogger::WriteToFITS(SubscriptionType& sub)
{
		//dailyFile status (open or not) already checked
		if (sub.dailyFile.IsOpen())
			sub.dailyFile.Write(sub.fConv);
		if (sub.runFile.IsOpen())
			sub.runFile.Write(sub.fConv);
}
#endif //if has_fits
// --------------------------------------------------------------------------
//
//! Implements the StartRun transition.
//! Concatenates the given path for the run file and the filename itself (based on the run number),
//! and tries to open it.
//! @returns
//!		kSM_Logging if success, kSM_BadRunConfig if failure.
int DataLogger::StartRunPlease()
{
	//attempt to open run file with current parameters
	if (fRunNumber == -1)
		return kSM_BadRunConfig;
	std::stringstream sRun;
	sRun << fRunNumber;
	fFullRunLogFileName = fRunFileName + '/' + sRun.str() + ".log";
	fRunLogFile.open(fFullRunLogFileName.c_str(), std::ios_base::out | std::ios_base::app); //maybe should be app instead of ate
	if (errno != 0)
	{
		std::stringstream str;
		str << "Unable to open run Log " << fFullRunLogFileName << ". Reason: " << strerror(errno) << " [" << errno << "]";
		Error(str);	
	}
	fFullRunReportFileName = fRunFileName + '/' + sRun.str() + ".rep";
	fRunReportFile.open(fFullRunReportFileName.c_str(), std::ios_base::out | std::ios_base::app);
	if (errno != 0)
	{
		std::stringstream str;
		str << "Unable to open run report " << fFullRunReportFileName << ". Reason: " << strerror(errno) << " [" << errno << "]";
		Error(str);	
	}
	
	if (!fRunLogFile.is_open() || !fRunReportFile.is_open())
	{
		//TODO send an error message
		return kSM_BadRunConfig;	
	}
	//get the size of the newly opened file.
	struct stat st;
	fBaseSizeRun = 0;
	if (fFileSizesMap.find(fFullRunLogFileName) == fFileSizesMap.end())
	{
		stat(fFullRunLogFileName.c_str(), &st);
		if (errno != 0)
		{
			std::stringstream str;
			str << "Unable to stat " << fFullRunLogFileName << ". Reason: " << strerror(errno) << " [" << errno << "]";
			Error(str);	
		}
		else
			fBaseSizeRun += st.st_size;
		fFileSizesMap[fFullRunLogFileName] = 0;
	}
	if (fFileSizesMap.find(fFullRunReportFileName) == fFileSizesMap.end())
	{
		stat(fFullRunReportFileName.c_str(), &st);
		if (errno != 0)
		{
			std::stringstream str;
			str << "Unable to stat " << fFullRunReportFileName << ". Reason: " << strerror(errno) << " [" << errno << "]";
			Error(str);	
		}
		else
			fBaseSizeRun += st.st_size;
		fFileSizesMap[fFullRunReportFileName] = 0;
	}
	std::string actualTargetDir;
	if (fRunFileName == ".")
	{
		char currentPath[FILENAME_MAX];
		getcwd(currentPath, sizeof(currentPath));
		if (errno != 0)
		{
			std::stringstream str;
			str << "Unable to retrieve the current path" << ". Reason: " << strerror(errno) << " [" << errno << "]";
			Error(str);	
		}		
		actualTargetDir = currentPath;
	}
	else
	{
		actualTargetDir = fRunFileName;	
	}		
	NotifyOpenedFile(actualTargetDir + '/' + sRun.str(), 3, fOpenedRunFiles);
	
	return kSM_Logging;
}
// --------------------------------------------------------------------------
//
//! Implements the StopRun transition.
//! Attempts to close the run file.
//! @returns
//!		kSM_WaitingRun if success, kSM_FatalError otherwise
int DataLogger::StopRunPlease()
{
	if (!fRunLogFile.is_open() || !fRunReportFile.is_open())
		return kSM_FatalError;
	
	fRunLogFile.close();
	fRunReportFile.close();
#ifdef HAS_FITS
	for (SubscriptionsListType::iterator i = fServiceSubscriptions.begin(); i != fServiceSubscriptions.end(); i++)
		for (std::map<std::string, SubscriptionType>::iterator j = i->second.begin(); j != i->second.end(); j++)
		{
				if (j->second.runFile.IsOpen())
					j->second.runFile.Close();	
		}
#ifdef ONE_RUN_FITS_ONLY
	if (fRunFitsFile != NULL)
	{
		delete fRunFitsFile;
		fRunFitsFile = NULL;	
		(fNumSubAndFitsData.numOpenFits)--;
	}
#endif
#endif
	NotifyOpenedFile("None", 0, fOpenedRunFiles);
	fNumSubAndFits->updateService();
	return kSM_WaitingRun;

}
// --------------------------------------------------------------------------
//
//! Implements the Stop and Reset transitions.
//! Attempts to close any openned file.
//! @returns
//! 	kSM_Ready
int DataLogger::GoToReadyPlease()
{
	if (fDailyLogFile.is_open())
		fDailyLogFile.close();
	if (fDailyReportFile.is_open())
		fDailyReportFile.close();

	if (fRunLogFile.is_open())
		fRunLogFile.close();
	if (fRunReportFile.is_open())
		fRunReportFile.close();
		
#ifdef HAS_FITS
	for (SubscriptionsListType::iterator i = fServiceSubscriptions.begin(); i != fServiceSubscriptions.end(); i++)
		for (std::map<std::string, SubscriptionType>::iterator j = i->second.begin(); j != i->second.end(); j++)
		{
				if (j->second.dailyFile.IsOpen())
					j->second.dailyFile.Close();
				if (j->second.runFile.IsOpen())
					j->second.runFile.Close();	
		}
#ifdef ONE_RUN_FITS_ONLY
	if (fRunFitsFile != NULL)
	{
		delete fRunFitsFile;
		fRunFitsFile = NULL;
		(fNumSubAndFitsData.numOpenFits)--;
	}
#endif
#endif
	if (GetCurrentState() == kSM_Logging)
		NotifyOpenedFile("None", 0, fOpenedRunFiles);
	if (GetCurrentState() == kSM_Logging || 
	    GetCurrentState() == kSM_WaitingRun || 
	    GetCurrentState() == kSM_DailyOpen)
	{ 
		NotifyOpenedFile("None", 0, fOpenedDailyFiles);
		fNumSubAndFits->updateService();
	}
	return kSM_Ready;
}
// --------------------------------------------------------------------------
//
//! Implements the transition towards kSM_WaitingRun
//! Does nothing really.
//!	@returns
//!		kSM_WaitingRun
int DataLogger::DailyToWaitRunPlease()
{
	return kSM_WaitingRun;	
}

// --------------------------------------------------------------------------

int RunDim(Configuration &conf)
{
    WindowLog wout;

    //log.SetWindow(stdscr);
    if (conf.Has("log"))
        if (!wout.OpenLogFile(conf.Get<std::string>("log")))
            wout << kRed << "ERROR - Couldn't open log-file " << conf.Get<std::string>("log") << ": " << strerror(errno) << std::endl;

    // Start io_service.Run to use the StateMachineImp::Run() loop
    // Start io_service.run to only use the commandHandler command detaching
    DataLogger logger(wout);
    logger.Run(true);

    return 0;
}

void RunThread(DataLogger* logger)
{
	// This is necessary so that the StateMachine Thread can signal the 
	// Readline thread to exit
	logger->Run(true);
	Readline::Stop();	
}

template<class T>
int RunShell(Configuration &conf)
{
    static T shell(conf.GetName().c_str(), conf.Get<int>("console")!=1);

    WindowLog &win  = shell.GetStreamIn();
    WindowLog &wout = shell.GetStreamOut();

    if (conf.Has("log"))
        if (!wout.OpenLogFile(conf.Get<std::string>("log")))
            win << kRed << "ERROR - Couldn't open log-file " << conf.Get<std::string>("log") << ": " << strerror(errno) << std::endl;

    DataLogger logger(wout);
    
//    if (conf.Has("black-list"))
//    	logger.setBlackList(conf.Get<std::string>("black-list"));
//    else if (conf.Has("white-list"))
//    	logger.setWhiteList(conf.Get<std::string>("white-list"));

    shell.SetReceiver(logger);

	boost::thread t(boost::bind(RunThread, &logger));
	
	shell.Run(); // Run the shell
	
	logger.Stop();
	
	//Wait until the StateMachine has finished its thread
	//before returning and destroyinng the dim objects which might 
	//still be in use.
	t.join();

    return 0;
}

/*
 Extract usage clause(s) [if any] for SYNOPSIS.
 Translators: "Usage" and "or" here are patterns (regular expressions) which
 are used to match the usage synopsis in program output.  An example from cp
 (GNU coreutils) which contains both strings:
  Usage: cp [OPTION]... [-T] SOURCE DEST
    or:  cp [OPTION]... SOURCE... DIRECTORY
    or:  cp [OPTION]... -t DIRECTORY SOURCE...
 */
void PrintUsage()
{
    cout << "\n"
        "The data logger connects to all available Dim services and "
        "writes them to ascii and fits files.\n"
        "\n"
        "Usage: dataLogger [-c type] [OPTIONS]\n"
        "  or:  dataLogger [OPTIONS]\n"
        "\n"
        "Options:\n"
        "The following describes the available commandline options. "
        "For further details on how command line option are parsed "
        "and in which order which configuration sources are accessed "
        "please refer to the class reference of the Configuration class.";
    cout << endl;

}

void PrintHelp()
{
    cout << "\n"
        "The default is that the program is started without user interaction. "
        "All actions are supposed to arrive as DimCommands. Using the -c "
        "option, a local shell can be initialized. With h or help a short "
        "help message about the usuage can be brought to the screen."
        << endl;
}

/*
 The first line of the --version information is assumed to be in one
 of the following formats:

   <version>
   <program> <version>
   {GNU,Free} <program> <version>
   <program> ({GNU,Free} <package>) <version>
   <program> - {GNU,Free} <package> <version>

 and separated from any copyright/author details by a blank line.

 Handle multi-line bug reporting sections of the form:

   Report <program> bugs to <addr>
   GNU <package> home page: <url>
   ...
*/
void PrintVersion(const char *name)
{
    cout <<
        name << " - "PACKAGE_STRING"\n"
        "\n"
        "Written by Thomas Bretz et al.\n"
        "\n"
        "Report bugs to <"PACKAGE_BUGREPORT">\n"
        "Home page: "PACKAGE_URL"\n"
        "\n"
        "Copyright (C) 2011 by the FACT Collaboration.\n"
        "This is free software; see the source for copying conditions.\n"
        << endl;
}


void SetupConfiguration(Configuration &conf)
{
    const string n = conf.GetName()+".log";

    po::options_description config("Program options");
    config.add_options()
        ("dns",       var<string>("localhost"),  "Dim nameserver host name (Overwites DIM_DNS_NODE environment variable)")
        ("log,l",     var<string>(n), "Write log-file")
        ("console,c", var<int>(),     "Use console (0=shell, 1=simple buffered, X=simple unbuffered)")
        ("black-list,b", var<string>(""), "Black-list of services")
        ("white-list,w", var<string>(""), "White-list of services")
        ;

    conf.AddEnv("dns", "DIM_DNS_NODE");

    conf.AddOptions(config);
}

int main(int argc, const char* argv[])
{
    Configuration conf(argv[0]);
    conf.SetPrintUsage(PrintUsage);
    SetupConfiguration(conf);

    po::variables_map vm;
    try
    {
        vm = conf.Parse(argc, argv);
    }
    catch (std::exception &e)
    {
#if BOOST_VERSION > 104000
        po::multiple_occurrences *MO = dynamic_cast<po::multiple_occurrences*>(&e);
        if (MO)
            cout << "Error: " << e.what() << " of '" << MO->get_option_name() << "' option." << endl;
        else
#endif
            cout << "Error: " << e.what() << endl;
        cout << endl;

        return -1;
    }

    if (conf.HasPrint())
        return -1;

    if (conf.HasVersion())
    {
        PrintVersion(argv[0]);
        return -1;
    }

    if (conf.HasHelp())
    {
        PrintHelp();
        return -1;
    }

    setenv("DIM_DNS_NODE", conf.Get<string>("dns").c_str(), 1);

    try
    {
        // No console access at all
        if (!conf.Has("console"))
            return RunDim(conf);

        // Console access w/ and w/o Dim
        if (conf.Get<int>("console")==0)
            return RunShell<LocalShell>(conf);
        else
            return RunShell<LocalConsole>(conf);
    }
    catch (std::exception& e)
    {
        cerr << "Exception: " << e.what() << endl;
        return -1;
    }

    return 0;
}
