#include "MainWindow.h"

#include <iostream>

#include <QTimer>

#include "dic.hxx"

#include "DockWindow.h"
#include "HtmlDelegate.h"
#include "CheckBoxDelegate.h"

using namespace std;

namespace Dim
{
static bool SendCommand(const char *command)
{
    return DimClient::sendCommand(command, NULL, 0);
}

template<typename T>
static bool SendCommand(const char *command, const T &t)
{
    return DimClient::sendCommand(command, (void*)&t, sizeof(t));
}
}

MainWindow::MainWindow(QWidget *p) : QMainWindow(p)
{
    // setupUi MUST be called before the DimNetwork is initilized
    // In this way it can be ensured that nothing from the
    // DimNetwork arrives before all graphical elements are
    // initialized. This is a simple but very powerfull trick.
    setupUi(this);

    // Now here we can do further setup which should be done
    // before the gui is finally displayed.
    fDimCmdServers->setItemDelegate(new CheckBoxDelegate);
    fDimCmdCommands->setItemDelegate(new CheckBoxDelegate);
    fDimCmdDescription->setItemDelegate(new HtmlDelegate);

    fDimSvcServers->setItemDelegate(new CheckBoxDelegate);
    fDimSvcServices->setItemDelegate(new CheckBoxDelegate);
    fDimSvcDescription->setItemDelegate(new HtmlDelegate);

    fStatusBar->showMessage(PACKAGE_STRING"   |   "PACKAGE_URL"   |   report bugs to <"PACKAGE_BUGREPORT">");

    fFtuLED[0] = fFtuLEDPrototype;

    for (int i=1; i<40; i++)
    {
        QPushButton *b = new QPushButton(static_cast<QWidget*>(fFtuLEDPrototype->parent()));

        b->setEnabled(fFtuLEDPrototype->isEnabled());
        b->setSizePolicy(fFtuLEDPrototype->sizePolicy());
        b->setMaximumSize(fFtuLEDPrototype->maximumSize());
        b->setIcon(fFtuLEDPrototype->icon());
        b->setIconSize(fFtuLEDPrototype->iconSize());
        b->setCheckable(fFtuLEDPrototype->isCheckable());
        b->setFlat(fFtuLEDPrototype->isFlat());

        fFtuLedLayout->addWidget(b, i/10+1, i%10+1, 1, 1);

        fFtuLED[i] = b;
    }

    for (int i=0; i<40; i++)
    {
        fFtuLED[i]->setObjectName(QString::fromUtf8("fFtuLED")+QString::number(i));
        QObject::connect(fFtuLED[i], SIGNAL(clicked()), this, SLOT(slot_fFtuLED_clicked()));
    }

    QTimer *timer = new QTimer(this);
    connect(timer, SIGNAL(timeout()), this, SLOT(slot_TimeUpdate()));
    timer->start(100);
}

void MainWindow::slot_TimeUpdate()
{
    fUTC->setDateTime(QDateTime::currentDateTimeUtc());
}

void MainWindow::SelectTab(const QString &name)
{
    for (int i=0; i<fTabWidget->count(); i++)
        if (fTabWidget->tabText(i)==name)
        {
            fTabWidget->setCurrentIndex(i);
            break;
        }
}


void MainWindow::on_fShutdown_clicked()
{
    Dim::SendCommand("DIS_DNS/KILL_SERVERS");
}


void MainWindow::on_fShutdownAll_clicked()
{
    Dim::SendCommand("DIS_DNS/KILL_SERVERS");
    Dim::SendCommand("DIS_DNS/EXIT", int(1));
}

void MainWindow::on_fTabWidget_tabCloseRequested(int which)
{
    // To get the correct size we have to switch to this tab
    // An alternative would be to take the size of the current tab
    fTabWidget->setCurrentIndex(which);

    QWidget *w = fTabWidget->currentWidget(); //fTabWidget->widget(which);
    if (!w)
    {
        cout << "Weird... the tab requested to be closed doesn't exist!" << endl;
        return;
    }

    QDockWidget *d = w->findChild<QDockWidget*>();
    if (!d)
    {
        cout << "Sorry, tab requested to be closed contains no QDockWidget!" << endl;
        return;
    }

    new DockWindow(d, fTabWidget->tabText(which));
    fTabWidget->removeTab(which);

    if (fTabWidget->count()==1)
        fTabWidget->setTabsClosable(false);
}

void MainWindow::SetTriggerSequence()
{
    const uint8_t d[3] =
    {
        uint8_t(fTriggerSeqPed->value()),
        uint8_t(fTriggerSeqLPint->value()),
        uint8_t(fTriggerSeqLPext->value())
    };

    Dim::SendCommand("FTM_CONTROL/SET_TRIGGER_SEQUENCE", d);
}

void on_fEnableTrigger_clicked(bool b)
{
    Dim::SendCommand("FTM_CONTROL/ENABLE_TRIGGER", b);
}

void on_fEnableExt1_clicked(bool b)
{
    Dim::SendCommand("FTM_CONTROL/ENABLE_EXT1", b);
}

void on_fEnableExt2_clicked(bool b)
{
    Dim::SendCommand("FTM_CONTROL/ENABLE_EXT2", b);
}

void on_fEnableVeto_clicked(bool b)
{
    Dim::SendCommand("FTM_CONTROL/ENABLE_VETO", b);
}

void MainWindow::SetTriggerCoincidence()
{
    const uint16_t d[2] =
    {
        uint16_t(fPhysicsCoincidence->value()),
        uint16_t(fPhysicsWindow->value()/4-2)
    };

    Dim::SendCommand("FTM_CONTROL/SET_TRIGGER_COINCIDENCE", d);
}

void MainWindow::SetCalibCoincidence()
{
    const uint16_t d[2] =
    {
        uint16_t(fCalibCoincidence->value()),
        uint16_t(fCalibWindow->value()/4-2)
    };

    Dim::SendCommand("FTM_CONTROL/SET_CALIBRATION_COINCIDENCE", d);
}

void MainWindow::on_fThresholdVal_valueChanged(int v)
{
    fThresholdVolt->setValue(2500./4095*v);

    const int32_t d[2] = { fThresholdIdx->value(), v };
    Dim::SendCommand("FTM_CONTROL/SET_THRESHOLD", d);
}

void MainWindow::on_fTriggerInterval_valueChanged(int val)
{
    Dim::SendCommand("FTM_CONTROL/SET_TRIGGER_INTERVAL", val);
}

void MainWindow::on_fTriggerDelay_valueChanged(int val)
{
    Dim::SendCommand("FTM_CONTROL/SET_TRIGGER_DELAY", val/4-2);
}

void MainWindow::on_fTimeMarkerDelay_valueChanged(int val)
{
    Dim::SendCommand("FTM_CONTROL/SET_TIME_MARKER_DELAY", val/4-2);
}

void MainWindow::on_fDeadTime_valueChanged(int val)
{
    Dim::SendCommand("FTM_CONTROL/SET_DEAD_TIME", val/4-2);
}

void MainWindow::on_fPrescalingVal_valueChanged(int val)
{
    Dim::SendCommand("FTM_CONTROL/SET_PRESCALING", val);
}

void MainWindow::slot_fFtuLED_clicked()
{
    for (int32_t i=0; i<40; i++)
        if (sender()==fFtuLED[i])
        {
            Dim::SendCommand("FTM_CONTROL/TOGGLE_FTU", i);
            break;
        }
}

void MainWindow::on_fPing_toggled(bool checked)
{
    if (checked)
        Dim::SendCommand("FTM_CONTROL/PING");
}

void MainWindow::on_fChatSend_clicked()
{
    if (Dim::SendCommand("CHAT/MSG", fChatMessage->displayText().constData()))
        fChatMessage->clear();
}

void MainWindow::on_fStatusLoggerLed_clicked()
{
    SelectTab("Logger");
}

void MainWindow::on_fStatusChatLed_clicked()
{
    SelectTab("Chat");
}

void MainWindow::on_fStatusFTMLed_clicked()
{
    SelectTab("Trigger");
}

void MainWindow::on_fStatusFTULed_clicked()
{
    SelectTab("FTUs");
}
