//-----------------------------------------------------------------------------
#include "typedefs.h"
#include "muxer_fsc.h"
#include "parser.h"
#include "application.h"
#include "output.h"
#include "spi_master.h"
#include "ad7719_adc.h"
#include "atmega_adc.h"    
#include "usart.h"
#include "macros.h"
#include "interpol.h"
#include "w5100_spi_interface.h"
#include <avr/interrupt.h>
#include <avr/wdt.h>
#include <stdlib.h>
//-----------------------------------------------------------------------------

// MAIN WORKFLOW GLOBAL VARIABLES
	bool heartbeat_enable = true;
	bool human_readable_interface_active = true;
	

// USART global variables
	U08 usart_rx_buffer[USART_RX_BUFFER_SIZE];
	U08 usart_tx_buffer[USART_TX_BUFFER_SIZE];
	U08 usart_received_chars;
	U08 usart_rx_buffer_index = 0;
	U08 usart_tx_buffer_index = 0;
  	U08 usart_last_char; 					// last received char

// USART FLAGS
	bool usart_tx_buffer_overflow = false;	// true if usart_tx_buffer was full.
	bool usart_rx_ready = false;			// EOL was received, parser needs to be called
	
// TIMER global variable
	volatile U32 local_ms = 0;

// AD7719 global variables
	U32 ad7719_values[TEMP_CHANNELS];
	U08 ad7719_enables[CHANNEL_BITMAP];
	U08 ad7719_channels_ready[CHANNEL_BITMAP];
	U08 ad7719_readings_since_last_muxing = 0;
	U08 ad7719_current_channel = 0;
	U32 ad7719_current_reading = 0;
	bool ad7719_measured_all = false;
	
// ATMEGA ADC global variables
	U08 adc_values[V_CHANNELS + I_CHANNELS + H_CHANNELS]; // stores measured voltage in steps of 16mV
	U08 adc_enables[V_BITMAP + I_BITMAP + H_BITMAP];
	U08 adc_channels_ready[V_BITMAP + I_BITMAP + H_BITMAP];
	U08 adc_readings_since_last_muxing = 0;
	U08 adc_current_channel = 0;
	U08 adc_current_reading = 0;
	bool adc_measured_all = false;

	bool once_told_you = true;
	bool debug_mode = false;

//-----------------------------------------------------------------------------
//   M A I N    ---   M A I N    ---   M A I N    ---   M A I N    ---  M A I N    
//-----------------------------------------------------------------------------
int main(void)
{
	app_init();		  // Setup: Watchdog and I/Os
	usart_init();		// Initialize serial interface   
	spi_init(); 		// Initialize SPI interface as master
	ad7719_init(); 		// Initialize AD7719 ADC as SPI slave 
	atmega_adc_init();

// TIMER2 is used as local clock:
// configure timer 2
	TCCR2 = (1<<WGM21); // CTC Modus
	TCCR2 |= (1<<CS21) | (1<<CS20); // Prescaler 64 --> counts up every 8us
	OCR2 = 125-1; 					// --> output compare interrupt occurs every 125 x 8us = 1ms
	// Compare Interrupt erlauben
	TIMSK |= (1<<OCIE2);

  //  Enable interrupts
  sei();              

  PORTB &= ~(1<<PB2);  // PULL PB2 low --> this is #RST of WIZ812MJ --> WIZ812MJ is inactive.

	static U08 welcome[]="\n\nwelcome to FACT FSC commandline interface v0.5\nready?";
	usart_write_str(welcome);

	
  
for ( U08 i=0; i<CHANNEL_BITMAP; ++i ) {
	ad7719_enables[i]=0x00;
	ad7719_channels_ready[i]=0;
	}
	ad7719_enables[0]=0x08;
for ( U08 i=0; i<V_BITMAP + I_BITMAP; ++i ) {
	adc_enables[i]=0xFF;
	adc_channels_ready[i]=0;
}
	adc_enables[V_BITMAP + I_BITMAP + H_BITMAP -1] = 0xF0;
	adc_channels_ready[V_BITMAP + I_BITMAP + H_BITMAP -1]=0;



//MAIN LOOP
while (1)
{
	if (heartbeat_enable) PORTB ^= (1<<PB3); // toggle Out2_spare --> heartbeat
//----------------------------------------------------------------------------
	//IF we need to send away one byte, and ready to send
	// THIS IS NOT USED AT THE MOMENT
	if ( (usart_tx_buffer_index > 0) && (UCSRA & (1<<UDRE)) ) { 
		UDR = usart_tx_buffer[0];
		// THis is shit
		for (U08 i=0 ; i < USART_TX_BUFFER_SIZE; ++i) {
			usart_tx_buffer[i] = usart_tx_buffer[i+1];
		}
		usart_tx_buffer_index--;
	}
//----------------------------------------------------------------------------

	//IF we just received one byte, and there is enough space in the RX_buffer
	if ( (UCSRA & (1<<RXC)) && (usart_rx_buffer_index < USART_RX_BUFFER_SIZE) ){
		usart_last_char = UDR;
		if (usart_last_char == '\n'){ // if EOL was received
			usart_rx_ready = true;
		}else {
		usart_rx_buffer[usart_rx_buffer_index] = usart_last_char;
		usart_rx_buffer_index++;
		}
		// here is still something strange .... better send an enter automatically
	} else if (UCSRA & (1<<RXC)) { // if there is no space in the buffer; read anyway.
		usart_last_char = UDR;
		usart_rx_buffer_index =0;
	}
//----------------------------------------------------------------------------

	//IF USART DOR bit is set, PC is sending data to fast!!!
	if ( UCSRA & (1<<DOR) ){
		// flush TX_buffer and write warning message in
		// maybe even switch off every measurement. ?
	}
//----------------------------------------------------------------------------

	//IF TX_BUFFER was overrun.	
	if (usart_tx_buffer_overflow) {
	
		// TX BUFFER is not used in this firmware version.
		
		// flash TX_buffer and write warning message in
		// maybe even switch off every measurement. ?
		//
		// this should only happen, in verbose mode and with low baudrates.
	}
//----------------------------------------------------------------------------
	
	//IF one command was received. 
	//	-It is not allowed to send more than one command between two '\n'
	if (usart_rx_ready){ 
		parse_ascii();
		usart_rx_buffer_index = 0;
		usart_rx_ready = false;
	}
//----------------------------------------------------------------------------

	//IF ATmega internal ADC did finish a conversion --every 200us
	if ( (ADCSRA & (1<<ADIF)) && !adc_measured_all) {
		adc_current_reading = ADCH;
		if (adc_readings_since_last_muxing == ADC_READINGS_UNTIL_SETTLED) {
			adc_values[adc_current_channel] = adc_current_reading;
			adc_readings_since_last_muxing=0;
			// note that this channel is ready, now and 
			//adc_output(adc_current_channel, adc_current_reading);
			// proceed to the next enabled channel.
			adc_channels_ready[adc_current_channel/8] |= (1<<(adc_current_channel%8));
			adc_current_channel = increase_adc (adc_current_channel);
			Set_V_Muxer(adc_current_channel);
			_delay_ms(10);
		} else { // the ADC did not settle yet, we discard the reading
			++adc_readings_since_last_muxing;
			// current reading is not used for anything else
		}
	}
//----------------------------------------------------------------------------

	//IF AD7719 ADC just finished a conversion -- every 60ms
	
	if (AD7719_IS_READY() && !ad7719_measured_all) {
			ad7719_current_reading = read_adc(); // --takes at 4MHz SCLK speed about 6us
		// AD7719 is only read out if settled. saves time.	
		if (ad7719_readings_since_last_muxing == AD7719_READINGS_UNTIL_SETTLED) {
			ad7719_values[ad7719_current_channel] = ad7719_current_reading;
			ad7719_readings_since_last_muxing=0;
			if (debug_mode) {
				usart_write_U32_hex(ad7719_current_reading);
				usart_write_char('\n');
				usart_write_char('\n');
			}
			// now prepare the data to be send away via USART.
			//ad7719_output(ad7719_current_channel, ad7719_current_reading);
			// note that this channel is ready, now and 
			// proceed to the next enabled channel.
			ad7719_channels_ready[ad7719_current_channel/8] |= (1<<(ad7719_current_channel%8));
			ad7719_current_channel = increase_ad7719 (ad7719_current_channel);
			Set_T_Muxer(ad7719_current_channel);
		} else { // the AD7719 did not settle yet, we discard the reading
			++ad7719_readings_since_last_muxing;
			if (debug_mode) {
				usart_write_U32_hex(ad7719_current_reading);
				usart_write_char('\n'); 
			}

			// current reading is not used for anything else
		}
	}
//----------------------------------------------------------------------------
	//IF one of the ADC measured all channels, we wanted to know.
	check_what_measurement_was_finished();
	
	if (ad7719_measured_all && adc_measured_all && !once_told_you)
	{
		adc_output_all();
		once_told_you = true;
	}
//----------------------------------------------------------------------------

} // end of MAIN LOOP
//-----------------------------------------------------------------------------
//    E N D     E N D     E N D     E N D     E N D     E N D     E N D     
//-----------------------------------------------------------------------------
} // end of main() function


ISR (TIMER2_COMP_vect)
{
 ++local_ms;
}


U08	increase_adc (U08 channel){
U08 effective_channel;
	for ( U08 increase = 1 ; increase <= V_CHANNELS + I_CHANNELS + H_CHANNELS; increase++)
	{
		effective_channel = (channel + increase) % (V_CHANNELS + I_CHANNELS + H_CHANNELS);
		if (adc_enables[effective_channel/8] & (1<<effective_channel%8))
			return effective_channel;
	}
	return channel;
} // end if increase_adc;

U08	increase_ad7719 (U08 channel){
U08 effective_channel;
	for ( U08 increase = 1 ; increase <= TEMP_CHANNELS; increase++)
	{
		effective_channel = (channel + increase) % (TEMP_CHANNELS);
		if (ad7719_enables[effective_channel/8] & (1<<effective_channel%8))
			return effective_channel;
	}
	return channel;
} // end if increase_adc;

void check_what_measurement_was_finished() {
	adc_measured_all = true;
	for ( U08 i=0; i<V_BITMAP + I_BITMAP + H_BITMAP; ++i ) {
		if ((adc_enables[i] ^ adc_channels_ready[i]) != 0x00) {
			adc_measured_all = false;
			break;
		}
	}
	ad7719_measured_all = true;
	for ( U08 i=0; i<CHANNEL_BITMAP; ++i ) {
		if ((ad7719_enables[i] ^ ad7719_channels_ready[i]) != 0x00) {
			ad7719_measured_all = false;
			break;
		}
	}


}



void set_ad7719_enable_register() {

	usart_write_str((pU08)"\n set enable bits of AD7719 Port ");
	if ((usart_rx_buffer_index>=5) && 
	(usart_rx_buffer[2] >= 'A' && usart_rx_buffer[2] <= 'H'))
	{
	usart_write_char(usart_rx_buffer[2]);
	usart_write_str((pU08)" to ");
	usart_write_U08_hex(usart_rx_buffer[4]);
	usart_write_char('\n');
		ad7719_enables[usart_rx_buffer[2]-'A']=usart_rx_buffer[4];
		ad7719_channels_ready[usart_rx_buffer[2]-'A']=0x00;
	}
	else if  ((usart_rx_buffer_index=3) && 
	(usart_rx_buffer[1] >= 'A' && usart_rx_buffer[1] <= 'H'))
	{
		usart_write_char(usart_rx_buffer[1]);
		if (usart_rx_buffer[2]!='0') {
			usart_write_str((pU08)" to 0xFF\n");
			ad7719_enables[usart_rx_buffer[1]-'A']=0xFF;
		} else
		{
			usart_write_str((pU08)" to 0x00\n");
			ad7719_enables[usart_rx_buffer[1]-'A']=0x00;
		}
		ad7719_channels_ready[usart_rx_buffer[1]-'A']=0x00;	
	}
	else
	{
		usart_write_str((pU08)"\n something wrong\n");
		usart_write_str((pU08)"usart_rx_buffer_index: ");
		usart_write_U08(usart_rx_buffer_index, 3);
		usart_write_str((pU08)"\n usart_rx_buffer[2]: ");
		usart_write_char(usart_rx_buffer[2]);
		usart_write_str((pU08)"\n usart_rx_buffer[4]: ");
		usart_write_U08_hex(usart_rx_buffer[4]);
		usart_write_char('\n');
	}
}

void set_adc_enable_register() {
	// TODO
	usart_write_str((pU08)"setting of ATmega internal ADC enable registers is not supported. yet.\n");
}
