#include <iostream>
#include <dic.hxx>

#include "FACT.h"
#include "Dim.h"
#include "Configuration.h"

using namespace std;

void SetupConfiguration(Configuration &conf)
{
    po::options_description config("Configuration");
    config.add_options()
        ("dns",               var<string>("localhost"),  "Dim nameserver host name (Overwites DIM_DNS_NODE environment variable)")
        ("schedule-database", var<string>(),  "Database name for scheduling preview")
        ;

    po::positional_options_description p;
    p.add("schedule-database", 1); // The first positional options

    conf.AddEnv("dns", "DIM_DNS_NODE");
    conf.AddOptions(config);
    conf.SetArgumentPositions(p);
}

void PrintUsage()
{
    cout <<
        "The triggerschedule triggers the scheduler.\n"
        "\n"
        "The default is that the program is started without user intercation. "
        "All actions are supposed to arrive as DimCommands. Using the -c "
        "option, a local shell can be initialized. With h or help a short "
        "help message about the usuage can be brought to the screen.\n"
        "\n"
        "Usage: triggerschedule [-c type] [OPTIONS] <schedule-database>\n"
        "  or:  triggerschedule [OPTIONS] <schedule-database>\n";
    cout << endl;
}

void PrintHelp()
{
    cout <<
        "\n"
        "The method sendCommand(...) will wait for the command to "
        "be actualy sent to the server and return a completion code "
        "of:\n"
        " 1 - if it was successfully sent.\n"
        " 0 - if it couldn't be delivered.\n "
        << endl;
    /* Additional help text which is printed after the configuration
     options goes here */
}

int main(int argc, const char* argv[])
{
    Configuration conf(argv[0]);
    conf.SetPrintUsage(PrintUsage);
    SetupConfiguration(conf);

    po::variables_map vm;
    try
    {
        vm = conf.Parse(argc, argv);
    }
    catch (std::exception &e)
    {
#if BOOST_VERSION > 104000
        po::multiple_occurrences *MO = dynamic_cast<po::multiple_occurrences*>(&e);
        if (MO)
            cout << "Error: " << e.what() << " of '" << MO->get_option_name() << "' option." << endl;
        else
#endif
            cout << "Error: " << e.what() << endl;
        cout << endl;

        return -1;
    }

    if (conf.HasPrint())
        return -1;

    if (conf.HasVersion())
    {
        FACT::PrintVersion(argv[0]);
        return -1;
    }

    if (conf.HasHelp())
    {
        PrintHelp();
        return -1;
    }

    if (!conf.Has("schedule-database"))
    {
        cout << "Please provide which database you want to use for scheduling using --schedule-database=<dbname>." << endl;
        return -1;
    }

    const char* dbname = conf.Get<string>("schedule-database").c_str();

    Dim::Setup(conf.Get<string>("dns"));

    const int rc = DimClient::sendCommand("SCHEDULER/SCHEDULE", dbname);
    if (!rc)
        cerr << "Sending failed!" << endl;
    else
        cout << "Command issued successfully." << endl;

    return rc;
}
