/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 (tbretz@uni-sw.gwdg.de)
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//   MTask                                                                 //
//                                                                         //
//   Base class for all tasks which can perfomed in a tasklist             //
//   For each event processed in the eventloop all the different           //
//   tasks in the tasklist will be processed.                              //
//                                                                         //
//   So all tasks must inherit from this baseclass.                        //
//                                                                         //
//   The inheritance from MInputStreamID is used to indicate the           //
//   type of event that this task is for. If it is "All" it is executed    //
//   independantly of the actual ID of the task list.                      //
//                                                                         //
//   Inside this abstract class, there are three fundamental function:     //
//                                                                         //
//   - PreProcess():   executed before the eventloop starts. Here you      //
//                     can initiate different things, open files, etc.     //
//                     As an argument this function gets a pointer to the  //
//                     parameter list. You can stop the execution by       //
//                     returning kFALSE instead of kTRUE. If an error      //
//                     occured and you return kFALSE make sure, that       //
//                     any action is closed correctly and all newly        //
//                     created object are deleted. The PostProcess in      //
//                     such a case won't be executed by the Tasklist or    //
//                     Eventloop.                                          //
//                                                                         //
//   - Process():      executed for each event in the eventloop. Do it     //
//                     one task after the other (as they occur in the      //
//                     tasklist). Only the tasks with a Stream ID          //
//                     which matches the actual ID of the tasklist         //
//                     are executed. A task can return kFALSE to           //
//                     stop the execuition of the tasklist or              //
//                     kCONTINUE to skip the pending tasks.                //
//                                                                         //
//   - PostProcess():  executed after the eventloop. Here you can close    //
//                     output files, start display of the run parameter,   //
//                     etc. PostProcess should be executed only if         //
//                     PreProcess was successfull (returned kTRUE)         //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MTask.h"

#include <TMethod.h>
#include <TOrdCollection.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MFilter.h"
#include "MGGroupFrame.h"

ClassImp(MTask);

MTask::MTask(const char *name, const char *title)
    : fFilter(NULL), fIsPreprocessed(kFALSE), fNumExecutions(0)
{
    fName  = name  ? name  : "MTask";
    fTitle = title ? title : "Base class for all tasks (dummy task).";

    fListOfBranches = new TOrdCollection;
    fListOfBranches->SetOwner();
}

MTask::~MTask()
{
    delete fListOfBranches;
}

// --------------------------------------------------------------------------
//
// This adds a branch to the list for the auto enabeling schmeme.
// This makes it possible for MReadTree to decide which branches
// are really needed for the eventloop. Only the necessary branches
// are read from disk which speeds up the calculation enormously.
//
// You can use TRegExp expressions like "*.fEnergy", but the
// recommended method is to call this function for exactly all
// branches you want to have, eg:
//  AddToBranchList("MMcTrig;1.fNumFirstLevel");
//  AddToBranchList("MMcTrig;2.fNumFirstLevel");
//
// We agreed on the convetion, that all branches are stored with
// a trailing dot '.' so that always the Master Branch name
// (eg. MMcTrig) is part of the branch name.
//
// Remark: The common place to call AddToBranchList is the
//         constructor of the derived classes (tasks)
//
void MTask::AddToBranchList(const char *b)
{
    TNamed *name = new TNamed(b, "");
    fListOfBranches->Add(name);
}

// --------------------------------------------------------------------------
//
// Copy constructor.
//
MTask::MTask(MTask &t)
{
    fFilter = t.fFilter;
    fListOfBranches->AddAll(t.fListOfBranches);
}

// --------------------------------------------------------------------------
//
// Mapper function for PreProcess.
// Sets the preprocessed flag dependend on the return value of PreProcess.
//
Bool_t MTask::CallPreProcess(MParList *plist)
{
    if (!PreProcess(plist))
        return kFALSE;

    fIsPreprocessed = kTRUE;
    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Mapper function for Process.
// Executes Process dependent on the existance of a filter and its possible
// return value.
// If Process is executed, the execution counter is increased.
//
Bool_t MTask::CallProcess()
{
    //
    // Check for the existance of a filter. If a filter is existing
    // check for its value. If the value is kFALSE don't execute
    // this task.
    //
    const Bool_t exec = fFilter ? fFilter->IsExpressionTrue() : kTRUE;

    if (!exec)
        return kTRUE;

    fNumExecutions++;
    return Process();
}

// --------------------------------------------------------------------------
//
// Mapper function for PreProcess.
// Calls Postprocess dependent on the state of the preprocessed flag,
// resets this flag.
//
Bool_t MTask::CallPostProcess()
{
    if (!fIsPreprocessed)
        return kTRUE;

    fIsPreprocessed = kFALSE;

    return PostProcess();
}

// --------------------------------------------------------------------------
//
// This is processed before the eventloop starts
//
// It is the job of the PreProcess to connect the tasks
// with the right container in the parameter list.
//
// the virtual implementation returns kTRUE
//
Bool_t MTask::PreProcess(MParList *pList)
{
    return kTRUE;
}

// --------------------------------------------------------------------------
//
// This is processed for every event in the eventloop
//
// the virtual implementation returns kTRUE
//
Bool_t MTask::Process()
{
    return kTRUE;
}

// --------------------------------------------------------------------------
//
// This is processed after the eventloop starts
//
// the virtual implementation returns kTRUE
//
Bool_t MTask::PostProcess()
{
    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Prints the number of times this task has been processed.
// For convinience the lvl argument results in a number of spaces at the
// beginning of the line. So that the structur of a tasklist can be
// identified.
//
void MTask::PrintStatistics(const Int_t lvl) const
{
    *fLog << setw(lvl) << " " << GetDescriptor() << "\t";
    *fLog << dec << fNumExecutions << endl;
}

