/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Harald Kornmayer 1/2001
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

#include "MBrowser.h"

#include <TSystem.h>        // gSystem

#include <TGTab.h>          // TGTab
#include <TGMenu.h>         // TGPopupMenu
#include <TGButton.h>       // TGTextButton
#include <TGMsgBox.h>       // TGMsgBox
#include <TBrowser.h>       // TBrowser
#include <TGListBox.h>      // TGListBox
#include <TGTextEntry.h>    // TGTextEntry
#include <TGFSComboBox.h>   // TGFSComboBox, TGFSLboxEntry
#include <TGProgressBar.h>  // TGHProgressBar
#include <TGFSContainer.h>  // TGFileContainer

#include <TG3DLine.h>       // TGHorizontal3DLine
                            // use TGSplitter.h for root<3.00

#include <sys/stat.h>       // S_ISDIR

#include <iostream.h>

ClassImp(MBrowser);

enum {
    kFileTBrowser,
    kFileClose,
    kButDirUp,
    kButListMode,
    kButDetailMode,
    kCBDirectory,
    kTEFileName
};

void MBrowser::CreateMenuBar()
{
    //
    //  crate the menu bar
    //
    TGPopupMenu *filemenu = new TGPopupMenu(gClient->GetRoot());
    filemenu->AddEntry("Open &TBrowser", kFileTBrowser);
    filemenu->AddSeparator();
    filemenu->AddEntry("&Close", kFileClose);
    filemenu->Associate(this);
    fList->Add(filemenu);

    //
    //  the button messages are handled by main frame (this)
    //
    TGLayoutHints *laymenubar  = new TGLayoutHints(kLHintsTop|kLHintsLeft|kLHintsExpandX, 2, 2, 2, 2);
    TGLayoutHints *laymenuitem = new TGLayoutHints(kLHintsTop|kLHintsLeft, 0, 4, 0, 0);

    fList->Add(laymenubar);
    fList->Add(laymenuitem);

    TGMenuBar *menubar = new TGMenuBar(this, 1, 1, kHorizontalFrame);
    menubar->AddPopup("&File", filemenu, laymenuitem);
    AddFrame(menubar, laymenubar);
    fList->Add(menubar);
}

void MBrowser::CreateUpperFrame(TGCompositeFrame *frametop)
{
    TGLayoutHints *lay1 = new TGLayoutHints(kLHintsTop    |kLHintsExpandX, 2, 2, 2, 0);
    TGLayoutHints *lay2 = new TGLayoutHints(kLHintsCenterY|kLHintsExpandX, 2, 2, 2, 2);
    TGLayoutHints *lay3 = new TGLayoutHints(kLHintsBottom |kLHintsExpandX, 2, 2, 0, 2);
    fList->Add(lay1);
    fList->Add(lay2);
    fList->Add(lay3);

    //
    // *********** Create Contents of frame top (upper part) **********
    //
    fTop1 = new TGHorizontalFrame(frametop, 500, 50);
    fTop2 = new TGHorizontalFrame(frametop, 500, 50);
    fTop3 = new TGHorizontalFrame(frametop, 500, 50);

    // FIXME: If I use TGLayoutHints the Progress Bar doesn't disappear!
    frametop->AddFrame(fTop1);//, lay1);
    frametop->AddFrame(fTop2);//, lay2);
    frametop->AddFrame(fTop3);//, lay3);

    fList->Add(fTop1);
    fList->Add(fTop2);
    fList->Add(fTop3);
}

void MBrowser::CreateDirListMenu(TGCompositeFrame *frame)
{
    //
    // Create Dir-Listbox and buttons in first frame
    //
    fDir = new TGFSComboBox(frame, kCBDirectory);
    fDir->Resize(350, 20);

    //
    // Get the three picturs from the system (must be deleted by FreePicture)
    //
    fPic1 = fClient->GetPicture("tb_uplevel.xpm");
    fPic2 = fClient->GetPicture("tb_list.xpm");
    fPic3 = fClient->GetPicture("tb_details.xpm");

    //
    // Create the corresponding picture buttons
    //
    fCdup     = new TGPictureButton(frame, fPic1, kButDirUp);
    fListMode = new TGPictureButton(frame, fPic2, kButListMode);
    fDetail   = new TGPictureButton(frame, fPic3, kButDetailMode);

    //
    // setup the buttons
    //
    fCdup    ->SetToolTipText("One Level up!");
    fListMode->SetToolTipText("List Mode");
    fDetail  ->SetToolTipText("Details Mode");

    fListMode->SetState(kButtonUp);
    fDetail  ->SetState(kButtonEngaged);

    fListMode->AllowStayDown(kTRUE);
    fDetail  ->AllowStayDown(kTRUE);

    //
    // send messages to 'this' object
    //
    fDir     ->Associate(this);
    fCdup    ->Associate(this);
    fDetail  ->Associate(this);
    fListMode->Associate(this);

    //
    // Add to list for 'automatic' deletion
    //
    fList->Add(fDir);
    fList->Add(fCdup);
    fList->Add(fListMode);
    fList->Add(fDetail);

    //
    // Layout Dir-Listbox and buttons in one row (frame)
    //
    //  - layout:
    //    alignment: top, left
    //    padding:   5, 5, 5, 5
    //
    TGLayoutHints *laydir = new TGLayoutHints(kLHintsExpandX|kLHintsLeft|kLHintsCenterY); //, 5, 5, 5, 5);
    fList->Add(laydir);

    TGLayoutHints *layout = new TGLayoutHints(kLHintsRight|kLHintsCenterY, 10); //, 5, 5, 5);
    fList->Add(layout);

    frame->AddFrame(fDir,      laydir);
    frame->AddFrame(fListMode, layout);
    frame->AddFrame(fDetail,   layout);
    frame->AddFrame(fCdup,     layout);
}

void MBrowser::CreateDirListBox(TGCompositeFrame *frame)
{
    //
    // Create file viewer (browser)
    //
    fFileView = new TGListView(frame, 540, 380);
    fList->Add(fFileView);

    TGViewPort *port = fFileView->GetViewPort();
    port->SetBackgroundColor(fgWhitePixel);

    fFileCont = new TGFileContainer(port, 100, 100, kVerticalFrame, fgWhitePixel);
    fList->Add(fFileCont);

    fFileView->SetContainer(fFileCont);
    fFileView->SetViewMode(kLVDetails);

    fFileCont->SetFilter("*");
    fFileCont->Associate(this);
    fFileCont->Sort(kSortByName);

    TGLayoutHints *layview = new TGLayoutHints(kLHintsTop|kLHintsExpandX|kLHintsExpandY); //, 5, 5, 5, 5);
    fList->Add(layview);

    frame->AddFrame(fFileView, layview);
}

void MBrowser::CreateTab1()
{
    TGCompositeFrame *frame = CreateNewTab("Input File");

    //
    // Create three frames for the first tab
    //
    TGHorizontalFrame *tab1a = new TGHorizontalFrame(frame, 100, 100);
    TGHorizontalFrame *tab1b = new TGHorizontalFrame(frame, 100, 100);
    TGVerticalFrame   *tab2  = new TGVerticalFrame  (frame, 100, 100);

    TGLayoutHints *laytab1  = new TGLayoutHints(kLHintsNormal|kLHintsExpandX,  10, 10, 10);
    TGLayoutHints *laytab2  = new TGLayoutHints(kLHintsExpandX|kLHintsExpandY, 10, 10, 10, 10);
    TGLayoutHints *layentry = new TGLayoutHints(kLHintsNormal|kLHintsExpandX); //,  5, 5, 5);

    frame->AddFrame(tab1a, laytab1);
    frame->AddFrame(tab1b, laytab1);
    frame->AddFrame(tab2,  laytab2);

    fEntry = new TGTextEntry(tab1a, "", kTEFileName);
    fEntry->Resize(350, 20);
    fEntry->Associate(this);
    tab1a->AddFrame(fEntry, layentry);

    CreateDirListMenu(tab1b);
    CreateDirListBox(tab2);

    fList->Add(laytab1);
    fList->Add(laytab2);
    fList->Add(tab1a);
    fList->Add(tab1b);
    fList->Add(tab2);
    fList->Add(layentry);
    fList->Add(fEntry);
}

TGCompositeFrame *MBrowser::CreateNewTab(const char *name)
{
    return fTabs->AddTab(name);
}

void MBrowser::CreateLowerFrame(TGCompositeFrame *framelow)
{
    //
    // *********** Create Contents of frame low (downer part) **********
    //

    //
    // ----- Create Object holding the Tabs -----
    //
    fTabs = new TGTab(framelow, 400, 400);
    fList->Add(fTabs);

    TGLayoutHints *laytabs = new TGLayoutHints(kLHintsBottom|kLHintsExpandX|kLHintsExpandY, 5, 5, 5, 5);
    fList->Add(laytabs);

    framelow->AddFrame(fTabs, laytabs);

    //
    // --- Create the first tab of the tabs ---
    //
    CreateTab1();
}

MBrowser::MBrowser(const TGWindow *main, const TGWindow *p,
                     const UInt_t w, const UInt_t h)
: TGTransientFrame(p?p:gClient->GetRoot(),
                   main?main:gClient->GetRoot(), w, h)
{
    fInputFile[0] = '\0';

    fList = new TList;
    fList->SetOwner();

    CreateMenuBar();

    TGLayoutHints *laylinesep  = new TGLayoutHints(kLHintsTop|kLHintsExpandX);
    fList->Add(laylinesep);

    TGHorizontal3DLine *linesep = new TGHorizontal3DLine(this);
    fList->Add(linesep);
    AddFrame(linesep, laylinesep);


    //
    // ---- Create the top window with a lot of buttons ----
    //
    TGLayoutHints *layframetop = new TGLayoutHints(kLHintsExpandX);
    fList->Add(layframetop);

    TGCompositeFrame *frametop = new TGCompositeFrame(this, 300, 100, kVerticalFrame);
    fList->Add(frametop);
    AddFrame(frametop, layframetop);

    linesep = new TGHorizontal3DLine(this);
    fList->Add(linesep);
    AddFrame(linesep, laylinesep);

    //
    // ---- Create the low window with a tabs in it ----
    //
    TGLayoutHints *layframelow = new TGLayoutHints(kLHintsExpandX|kLHintsExpandY);
    fList->Add(layframelow);

    TGCompositeFrame *framelow = new TGCompositeFrame(this, 300, 100, kHorizontalFrame);
    fList->Add(framelow);

    AddFrame(framelow, layframelow);

    CreateUpperFrame(frametop);
    CreateLowerFrame(framelow);

    //
    //   Map the window, set up the layout, etc.
    //
    ChangeDir();
    SetWMSizeHints(400, 350, 1000, 1000, 10, 10); // set the smallest and biggest size of the Main frame
    Move(rand()%100+50, rand()%100+50);
}


MBrowser::~MBrowser()
{
    fClient->FreePicture(fPic1);
    fClient->FreePicture(fPic2);
    fClient->FreePicture(fPic3);

    delete fList;
} 

TGProgressBar *MBrowser::CreateProgressBar(TGHorizontalFrame *frame)
{
    static TGLayoutHints laybar(kLHintsCenterY|kLHintsRight/*|kLHintsExpandX*/,
                                10, 10);

    TGHProgressBar *bar=new TGHProgressBar(frame);

    bar->SetWidth(150);
    bar->ShowPosition();

    frame->AddFrame(bar, &laybar);

    Layout();
    MapSubwindows();

    return bar;
}

void MBrowser::DestroyProgressBar(TGProgressBar *bar)
{
    TGHorizontalFrame *frame = (TGHorizontalFrame*)bar->GetParent();

    frame->RemoveFrame(bar);

    Layout();
    MapSubwindows();

    delete bar;
}

void MBrowser::CloseWindow()
{
   // Got close message for this MainFrame. Calls parent CloseWindow()
   // (which destroys the window) and terminate the application.
   // The close message is generated by the window manager when its close
   // window menu item is selected.

    delete this;
}

Bool_t MBrowser::InputFileSelected() 
{
    //
    // Checks if there is a selected input root file
    //
    return fInputFile[0]!='\0';
}

void MBrowser::DisplError(const char *txt)
{
    Int_t retval;
    new TGMsgBox(fClient->GetRoot(), this, "Error!", txt,
                 kMBIconExclamation, 4, &retval);
}

void MBrowser::DisplWarning(const char *txt)
{
    Int_t retval;
    new TGMsgBox(fClient->GetRoot(), this, "Warning!", txt,
                 kMBIconExclamation, 4, &retval);
}

void MBrowser::DisplInfo(const char *txt)
{
    Int_t retval;
    new TGMsgBox(fClient->GetRoot(), this, "Information!", txt,
                 kMBIconExclamation, 4, &retval);
}

void MBrowser::ChangeDir(const char *txt)
{
    fFileCont->ChangeDirectory(txt?txt:gSystem->WorkingDirectory());

    const char *dir = fFileCont->GetDirectory();
    fDir->Update(dir);
}

void MBrowser::SetFileName(const char *name)
{
    //
    // determine the file type by extensions
    //
    const char *ext=strrchr(name, '.');

    Bool_t failed = kFALSE;

    if (!ext)
        failed = kTRUE;
    else
        if (strcasecmp(ext, ".root"))
            failed = kTRUE;

    char *fname=NULL;
    if (!failed)
    {
        const char *dir = fFileCont->GetDirectory();

        fname = Form("%s/%s", dir, name);

        failed = gSystem->AccessPathName(fname, kFileExists);
    }

    if (!failed)
        strcpy(fInputFile, fname);
    else
        fname = Form(fInputFile);

    char *slash = strrchr(fname, '/');

    fEntry->SetText(slash ? slash+1 : "");

    if (!slash)
        return;

    *slash = '\0';
    ChangeDir(fname);
}

// --------------------------------------------------------------------------
//
// Process events generated by the gui elements in the frame.
//
Bool_t MBrowser::ProcessMessage(Long_t msg, Long_t parm1, Long_t)
{
    switch (GET_MSG(msg))
    {
    case kC_TEXTENTRY:
        if (GET_SUBMSG(msg)!=kTE_ENTER)
            return kTRUE;

        SetFileName(fEntry->GetText());
        return kTRUE;

    case kC_COMMAND:
        switch (GET_SUBMSG(msg))
        {
        case kCM_MENU:
            switch (parm1)
            {
            case kFileClose:
                CloseWindow();
                return kTRUE;

            case kFileTBrowser:
                new TBrowser();
                return kTRUE;
            }
            return kTRUE;

        case kCM_BUTTON:
            switch (parm1)
            {
            case kButDirUp:
                //
                //  goto the parent directory
                //
                ChangeDir("..");
                return kTRUE;

            case kButListMode:
                fFileView->SetViewMode(kLVList);
                fDetail->SetState(kButtonUp);
                return kTRUE;

            case kButDetailMode:
                fFileView->SetViewMode(kLVDetails);
                fListMode->SetState(kButtonUp);
                return kTRUE;
            }
            return kTRUE;

        case kCM_COMBOBOX:
            if (parm1 != kCBDirectory)
                return kTRUE;

            ChangeDir(((TGTreeLBEntry*)fDir->GetSelectedEntry())->GetPath()->GetString());
            return kTRUE;
        }
        return kTRUE;

    case kC_CONTAINER:
        switch (GET_SUBMSG(msg))
        {

            //      case kCT_ITEMCLICK:
            // 	printf ("itemclick\n");
            //      break;

        case kCT_ITEMDBLCLICK:
            //
            //  process the double click in the file view container
            //
            if (parm1 != kButton1 || fFileCont->NumSelected() != 1)
                return kTRUE;

            //
            //  one file selected
            //
            void *dummy = NULL;
            const TGFileItem *item = (TGFileItem *)fFileCont->GetNextSelected(&dummy);

            const char *str = item->GetItemName()->GetString();

            //
            // if the user choose a directory
            // change to this directory
            //
            if (S_ISDIR(item->GetType()))
            {
                ChangeDir(str);
                return kTRUE;
            }

            SetFileName(str);
            return kTRUE;
        }
    }
    return kTRUE;
}
