library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.all;

ENTITY w5300_modul IS
  generic(
    TIME_UNTIL_READ_READY : integer := 4;
	TIME_UNTIL_READ_READY : integer := 4;
  );
PORT(
	clk							: IN	std_logic;						-- expecting 50MHz
	
	wiz_reset					: OUT	std_logic;
	cs							: OUT	std_logic;
	wr							: OUT	std_logic;
	rd							: OUT	std_logic;
	int							: IN	std_logic;
	wiz_data					: INOUT	std_logic_vector (15 DOWNTO 0);
	wiz_addr					: OUT	std_logic_vector (9 DOWNTO 0);
	
	read_i						: IN	std_logic;
	write_i						: IN	std_logic;
	addr_i						: IN	std_logic_vector (9 DOWNTO 0);
	data_o						: OUT	std_logic_vector (15 DOWNTO 0)	:= (others => '0');
	data_i						: IN	std_logic_vector (15 DOWNTO 0);
	
);
END w5300_modul ;

architecture Behavioral of w5300_modul is
	signal read_not_write_signal : std_logic := '1';
	signal data_signal : std_logic_vector (15 DOWNTO 0) := (others => 'Z');
	signal addr_signal : std_logic_vector (9 DOWNTO 0) := (others => '0');

	
	-- this counter counts the time, the cs signal is low in units of 1/clk_period
	-- since it is increased already in IDLE state, it really 
	signal wait_ctr : integer range 0 to 15 := 0;  
	
	
main_process: process(clk)
	begin
	if rising__edge(clk) then
		-- synch in read & write commands
		read_sr 	<= read_sr(0) 	& read_i;
		write_sr 	<= write_sr(0) 	& write_i;

		case state is
			-- this state seems to lose 20ns of worthy time, but this is only true for single transmissions
			-- in case of continuous transmissions the W5300 datasheet demands min. 28ns time with CS high.
			-- this means min. 2 states @ a 50MHz clock.
			-- this is ensured by the IDLE state and a certain 'wait'-state and the end of each transmission.
			when IDLE =>
				if (read_sr = "01" ) then
					data_signal <= (others => 'Z');
					addr_sig <= addr_i;
					wait_ctr <= wait_ctr + 1;
					state <= READ_STATE;
				elsif (write_sr = "01" ) then
					data_signal <= data_i;	
					addr_sig <= addr_i;
					wait_ctr <= wait_ctr + 1;
					state <= WRITE_STATE;
				end if;
			
			when READ_STATE =>
				cs <= '0';
				wr <= '1';
				rd <= '0';
				wiz_addr <= addr_sig;
				wiz_data <= data_signal;
				wait_ctr <= wait_ctr + 1;
				state <= READ_WAIT;
				
			when WRITE_STATE =>
				cs <= '0';
				wr <= '0';
				rd <= '1';
				wiz_addr <= addr_sig;
				wiz_data <= data_signal;
				wait_ctr <= wait_ctr + 1;
				state <= WRITE_WAIT;

			-- actually WRITE ACCESS needs less time than READ access. 
			-- but in order to make an easy timed interface to this entity possible 
			-- I decided to wait exactly the same time.
			-- anyway after min. 42ns (i.e. 60ns in case of 50MHz) the read reasult may be synched in.
			
			when READ_WAIT =>
				if (wait_ctr = TIME_UNTIL_READ_READY - 1) then
					wait_ctr = 0;
					data_signal <= wiz_data;
					state <= CLEANUP;
				end if;
				wait_ctr <= wait_ctr + 1;
				
			when WRITE_WAIT
				if (wait_ctr = TIME_UNTIL_WRITE_READY - 1) then
					wait_ctr = 0;
					state <= CLEANUP;
				end if;
				wait_ctr <= wait_ctr + 1;
			
			when CLEANUP =>
				cs <= '1';
				wr <= '1';
				rd <= '1';
				data_o <= data_sigal;
				wiz_addr <= (others => 'Z');
				wiz_data <= (others => 'Z');
				state <= IDLE;
				
			end case;
			
	end if; -- rising_edge(clk)
						
end process main_process;