library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.NUMERIC_STD.all;

library FACT_FAD_lib;
use FACT_FAD_lib.fad_definitions.ALL;

ENTITY w5300_interface IS
PORT(
	clk							: IN	std_logic;						-- expecting 50MHz
	
	wiz_reset					: OUT	std_logic;
	cs							: OUT	std_logic;
	wr							: OUT	std_logic;
	rd							: OUT	std_logic;
	int							: IN	std_logic;
	wiz_data					: INOUT	std_logic_vector (15 DOWNTO 0);
	wiz_addr					: OUT	std_logic_vector (9 DOWNTO 0);
	
	read_i						: IN	std_logic;
	write_i						: IN	std_logic;
	addr_i						: IN	std_logic_vector (9 DOWNTO 0);
	data_o						: OUT	std_logic_vector (15 DOWNTO 0)	:= (others => '0');
	data_i						: IN	std_logic_vector (15 DOWNTO 0);
	ready_o						: OUT	std_logic := '0';
	reset_i						: IN	std_logic
);
END w5300_interface ;

architecture Behavioral of w5300_interface is

	type state_type is (
		RESET_WIZ,
		IDLE,
		--READ_STATE, 
		READ_WAIT,
		--WRITE_STATE, 
		WRITE_WAIT
--		,CLEANUP
		
	);
	signal state : state_type;

	signal data_signal : std_logic_vector (15 DOWNTO 0) := (others => 'Z');
	signal addr_signal : std_logic_vector (9 DOWNTO 0) := (others => '0');

    signal read_sr : std_logic_vector (1 DOWNTO 0) := (others => '0');
    signal write_sr : std_logic_vector (1 DOWNTO 0) := (others => '0');
	
	-- this counter counts the time, the cs signal is low in units of 1/clk_period
	-- since it is increased already in IDLE state, it really 
	signal wait_ctr : integer range 0 to 15 := 0;  
	signal RST_TIME : integer range 0 to 500000 := 500000;  

	signal reset_counter : integer range 0 to 500000 := 0;  
	signal ready : std_logic := '0';
begin
	
main_process : process(clk)
	begin
	
	ready_o <= ready and not write_i and not read_i;
	
	if rising_edge(clk) then
		-- synch in read & write commands
		read_sr 	<= read_sr(0) 	& read_i;
		write_sr 	<= write_sr(0) 	& write_i;

		--synthesis translate_off
		RST_TIME <= 288;
		--synthesis translate_on


		case state is
			-- this state seems to lose 20ns of worthy time, but this is only true for single transmissions
			-- in case of continuous transmissions the W5300 datasheet demands min. 28ns time with CS high.
			-- this means min. 2 states @ a 50MHz clock.
			-- this is ensured by the IDLE state and a certain 'wait'-state and the end of each transmission.
			when IDLE =>
				cs <= '1';
				wr <= '1';
				rd <= '1';
				ready <= '1';
				
				
				if (read_i = '1' ) then
					cs <= '0';
					wr <= '1';
					rd <= '0';
					ready <= '0';
					
					wiz_addr <= addr_i;
					wiz_data <= (others => 'Z');
					state <= READ_WAIT;
					
				elsif (write_i = '1' ) then
					cs <= '0';
					wr <= '0';
					rd <= '1';
					ready <= '0';
					
					wiz_addr <= addr_i;
					wiz_data <= data_i;

					state <= WRITE_WAIT;
				end if;
				
				if (reset_i = '1') then
					state <= RESET_WIZ;
				end if;
			
--			when READ_STATE =>
--				cs <= '0';
--				wr <= '1';
--				rd <= '0';
--				wiz_addr <= addr_signal;
--				wiz_data <= data_signal;
--				wait_ctr <= wait_ctr + 1;
--				state <= READ_WAIT;
--				
--			when WRITE_STATE =>
--				cs <= '0';
--				wr <= '0';
--				rd <= '1';
--				wiz_addr <= addr_signal;
--				wiz_data <= data_signal;
--				wait_ctr <= wait_ctr + 1;
--				state <= WRITE_WAIT;
--
			-- actually WRITE ACCESS needs less time than READ access. 
			-- but in order to make an easy timed interface to this entity possible 
			-- I decided to wait exactly the same time.
			-- anyway after min. 42ns (i.e. 60ns in case of 50MHz) the read reasult may be synched in.
			
			when READ_WAIT =>
				wait_ctr <= wait_ctr + 1;
				if (wait_ctr = TIME_UNTIL_READ_READY - 1) then
					cs <= '1';
					wr <= '1';
					rd <= '1';
					ready <= '1';
					
					wait_ctr <= 0;
					data_o <= wiz_data;

					state <= IDLE;
				end if;
				
				
			when WRITE_WAIT =>
				wait_ctr <= wait_ctr + 1;
				if (wait_ctr = TIME_UNTIL_WRITE_READY - 1) then
					cs <= '1';
					wr <= '1';
					rd <= '1';
					ready <= '1';
					
					wait_ctr <= 0;
					data_o <= (others => 'Z');

					state <= IDLE;
				end if;
				
			
--			when CLEANUP =>
--				ready <= '1';
--				cs <= '1';
--				wr <= '1';
--				rd <= '1';
--				data_o <= data_signal;
--				wiz_addr <= (others => 'Z');
--				wiz_data <= (others => 'Z');
--				state <= IDLE;
				
			when RESET_WIZ => 
				ready <= '0';
				cs <= '1';
				wr <= '1';
				rd <= '1';
				
				reset_counter <= reset_counter + 1;
				wiz_reset <= '0';
				if (reset_counter >= 100) then -- wait 2µs
					wiz_reset <= '1';
				end if;	
				if (reset_counter >= RST_TIME -1) then -- wait 10ms
					reset_counter <= 0;
					
					state <= IDLE;
				end if;				
				
			end case;
			
	end if; -- rising_edge(clk)
						
end process main_process;

end Behavioral;