-- Module Name:    w5300_modul2 - Behavioral 
library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
library FACT_FAD_lib;
use FACT_FAD_lib.fad_definitions.ALL;

ENTITY w5300_modul2 IS

   PORT( 
      state          : OUT    std_logic_vector (7 DOWNTO 0);  -- state is encoded here ... useful for debugging.
	  debug_data_ram_empty : OUT    std_logic;
	  debug_data_valid : OUT    std_logic;
	  data_generator_idle_i : IN std_logic;
     
      clk            : IN     std_logic;
      
	  wiz_reset      : OUT    std_logic                     := '1';
      addr           : OUT    std_logic_vector (9 DOWNTO 0);
      data           : INOUT  std_logic_vector (15 DOWNTO 0);
      cs             : OUT    std_logic                     := '1';
      wr             : OUT    std_logic                     := '1';
	  rd             : OUT    std_logic                     := '1';
	  
      
      int            : IN     std_logic;
      write_length   : IN     std_logic_vector (16 DOWNTO 0);
      ram_start_addr : IN     std_logic_vector (W5300_RAM_ADDR_WIDTH-1 DOWNTO 0);
      ram_data       : IN     std_logic_vector (15 DOWNTO 0);
      ram_addr       : OUT    std_logic_vector (W5300_RAM_ADDR_WIDTH-1 DOWNTO 0);
      data_valid     : IN     std_logic;
      data_valid_ack : OUT    std_logic := '0';
      busy           : OUT    std_logic                     := '1';
      write_header_flag, write_end_flag : IN std_logic;
      fifo_channels : IN std_logic_vector (3 downto 0);
      -- softtrigger:
      s_trigger : OUT std_logic := '0';
      c_trigger_enable: out std_logic := '0';
      c_trigger_mult: out std_logic_vector (15 DOWNTO 0) := conv_std_logic_vector(0 ,16); --subject to changes
      
      
      -- to be removed:
      led : out std_logic_vector (7 downto 0) := X"00";
      socket_tx_free_out : out std_logic_vector (16 downto 0) := '0' & X"0000";

	  -- FAD configuration signals:
	  ------------------------------------------------------------------------------
		memory_manager_config_start_o	: out std_logic := '0';
		memory_manager_config_valid_i	: in std_logic;
		
		spi_interface_config_start_o	: out std_logic := '0';
		spi_interface_config_valid_i	: in std_logic;
		
		--data_generator_config_start_o	: out std_logic := '0';
		--data_generator_config_valid_i	: in std_logic;
		
		dac_setting						: out dac_array_type := DEFAULT_DAC;		--<<-- default defined in fad_definitions.vhd
		roi_setting						: out roi_array_type := DEFAULT_ROI;		--<<-- default defined in fad_definitions.vhd
		
		runnumber						: out std_logic_vector (31 DOWNTO 0) := conv_std_logic_vector(0 ,32); 
		reset_trigger_id				: out std_logic := '0';

		data_ram_empty 					: IN std_logic;
		
	  ------------------------------------------------------------------------------
      
	  -- MAC/IP calculation signals:
	  ------------------------------------------------------------------------------
      MAC_jumper : in std_logic_vector (1 downto 0);
      BoardID : in std_logic_vector (3 downto 0);
      CrateID : in std_logic_vector (1 downto 0);
	  ------------------------------------------------------------------------------
      
	  -- user controllable enable signals
	  ------------------------------------------------------------------------------
	  trigger_enable : out std_logic; 
	  
      denable : out std_logic := '0'; -- default domino wave on. ... in case if REFCLK error ... REFCLK counter will override.
      dwrite_enable : out std_logic := '1'; -- default DWRITE low.
      sclk_enable : out std_logic := '1'; -- default DWRITE HIGH.
      srclk_enable : out std_logic := '1'; -- default SRCLK on.
	  ------------------------------------------------------------------------------
	  
	  -- ADC CLK generator, is able to shift phase with respect to X_50M
	  -- these signals control the behavior of the digital clock manager (DCM)
	  ------------------------------------------------------------------------------
	  ps_direction : out std_logic := '1'; -- default phase shift upwards
      ps_do_phase_shift : out std_logic := '0'; --pulse this to phase shift once
      ps_reset : out std_logic := '0'; -- pulse this to reset the variable phase shift
      ps_ready : in std_logic;
      ------------------------------------------------------------------------------
      	  
	  -- signals used to control FAD LED bahavior:
	  -- one of the three LEDs is used for com-status info
	  ------------------------------------------------------------------------------
      socks_waiting : out std_logic;
      socks_connected: out std_logic
	  ------------------------------------------------------------------------------
	
	
	  
   );
END w5300_modul2 ;

architecture Behavioral of w5300_modul2 is 

type state_init_type is (
	INTERRUPT, RESET, WAIT_FOR_RESET,
	INIT, 
		LOCATE, IM, MT,
		STX0, STX1, STX2, STX3,
		SRX0, SRX1, SRX2, SRX3,
		MAC0, MAC1, MAC2, 
		GW0, GW1, 
		SNM0, SNM1, 
		IP0, IP1,
	-- Socket initialisiation
	SI_MR, SI_IMR, SI_PORTOR, SI_PORT, SI_SSR, SI_CR_OPEN, SI_IS_OPEN, SI_CR_LISTEN,
	ESTABLISH, EST1,
	
	MAIN, 
	CONFIG, 
		WAIT_100NS, WAIT_UNTIL_DG_IDLE, 
		CONFIG_MEMORY_MANAGER, WAIT_FOR_CONFIG_MEMORY_MANAGER,
		CONFIG_DAC_ONLY, WAIT_FOR_CONFIG_DAC_ONLY,
	MAIN1, 
		READ_DATA,
	MAIN2, MAIN3,
		WRITE_DATA
	
-- these states do not exist anymore.		
--	WRITE_REG, READ_REG, 
--	--TIMEOUT, 
--	RETRY,
--	CONFIG_DATA_GENERATOR, WAIT_FOR_CONFIG_DATA_GENERATOR,
-- CHK_RECEIVED, 
);

type state_write_type is (
	WR_START,
		WR_GET_EVT_ID_WAIT1, WR_GET_EVT_ID1, WR_GET_EVT_ID_WAIT2, WR_GET_EVT_ID2,
		WR_MOD7_STARTED, WR_WAIT_FOR_MOD7,
		WR_CHECK_FOR_FIFO_SPACE_01, WR_CHECK_FOR_FIFO_SPACE_02, WR_CHECK_FOR_FIFO_SPACE_03, WR_CHECK_FOR_FIFO_SPACE_04,
		WR_HEADER_FETCH, WR_HEADER_WAIT, WR_HEADER,
		WR_ADC, WR_ADC1, WR_ADC2,
		WR_STRANGE_WAIT,
		WR_ENDFLAG_WAIT, WR_ENDFLAG, 
		WR_ENDFLAG2_WAIT, WR_ENDFLAG2,
		WR_05_PREPARE_LENGTH_INFO, WR_05_POSTPREPARE_LENGTH_INFO,
		WR_MESSAGE_LEN_HIGH_WORD, WR_MESSAGE_LEN_LOW_WORD, WR_SEND_COMMAND,
		WR_ACK, WR_WAIT_FOR_ACK
); 
type state_interrupt_1_type is (IR1_01, IR1_02, IR1_03, IR1_04);
type state_interrupt_2_type is (
	IR2_GET_IR, 
	IR2_CHECK_WHICH_SN_IRQ, 
	IR2_GET_SN_IR, 
	IR2_CLEAR_SN_IR, 
	IR2_SEND_DISCON, 
	IR2_GET_SOCKn_STATUS, 
	IR2_WAIT_SOCKn_CLOSED, 
	IR2_GOTO_RESET);
type state_read_data_type is (
	RD_1, 
	RD_2, 
	RD_3, 
	RD_4, 
	RD_5,
	WAIT_FOR_TRIGGER_ID_RESET_1, WAIT_FOR_TRIGGER_ID_RESET_2,
	RD_6,
	READ_COMMAND_DATA_SECTION,
	PUT_COMMAND_DATA_SECTION,
	RD_END
);
signal state_init : state_init_type := RESET;
signal state_write : state_write_type := WR_START;
signal state_interrupt_1 : state_interrupt_1_type := IR1_01;
signal state_interrupt_2 : state_interrupt_2_type := IR2_GET_IR;
signal state_read_data : state_read_data_type := RD_1;

signal RST_TIME : std_logic_vector(19 downto 0) := X"7A120";	-- 0x7A120=500000 @50MHz --> 10ms

signal wiz_addr : std_logic_vector (9 downto 0) := (OTHERS => '0');
signal wiz_data : std_logic_vector (15 downto 0) := (OTHERS => '0');
signal data_read : std_logic_vector (15 downto 0) := (OTHERS => '0');
signal adc_data_addr : std_logic_vector (W5300_RAM_ADDR_WIDTH-1 DOWNTO 0);

signal interrupt_ignore : std_logic := '1';
signal int_flag : std_logic := '0';

signal data_cnt : integer range 0 to 7680 := 0; 
signal data_end : integer range 0 to 7680 := 0;
signal drs_cnt : integer range 0 to 3 := 0;
signal channel_cnt : integer range 0 to 9 :=0;
signal socket_cnt : std_logic_vector (2 downto 0 ) := "000";
-- should be integer range 0 to 7 := 0; , but I didn't manage to get it working... 
signal roi_max : std_logic_vector (10 downto 0) := (others => '0');


signal socket_tx_free : std_logic_vector (16 downto 0) := (others => '0');
signal write_length_bytes : std_logic_vector (16 downto 0);

signal socket_rx_received : std_logic_vector (16 downto 0) := (others => '0');
signal chk_recv_cntr : integer range 0 to 10000 := 0;

-- --
signal wait_cntr : integer range 0 to 10000 := 0;
-- --

signal rx_packets_cnt : std_logic_vector (15 downto 0);

signal update_of_rois : std_logic := '1';
signal update_of_lessimportant : std_logic := '1';


signal trigger_enable_sig : std_logic := '0';
signal trigger_enable_storage_sig : std_logic;

signal local_write_length   : std_logic_vector (16 DOWNTO 0);
signal local_ram_start_addr : std_logic_vector (W5300_RAM_ADDR_WIDTH-1 DOWNTO 0);
signal local_ram_addr       : std_logic_vector (W5300_RAM_ADDR_WIDTH-1 downto 0);
signal local_socket_nr      : std_logic_vector (2 DOWNTO 0);
signal local_write_header_flag, local_write_end_flag : std_logic;
signal local_fifo_channels : std_logic_vector (3 downto 0);

signal wait_100ns_sig : std_logic_vector (2 downto 0) := "000";

signal config_addr : integer range 0 to 44;
type config_data_type is array (0 to 46) of std_logic_vector(15 downto 0); 
signal config_setting : config_data_type := (
--		X"000A", X"000A", X"000A", X"000A", X"000A", X"000A", X"000A", X"000A", X"000A", 		--<<-- ROIs = 10 TESTING ONLY
--		X"000A", X"000A", X"000A", X"000A", X"000A", X"000A", X"000A", X"000A", X"000A", 		--<<-- ROIs = 10 TESTING ONLY
--		X"000A", X"000A", X"000A", X"000A", X"000A", X"000A", X"000A", X"000A", X"000A", 		--<<-- ROIs = 10 TESTING ONLY
--		X"000A", X"000A", X"000A", X"000A", X"000A", X"000A", X"000A", X"000A", X"000A", 		--<<-- ROIs = 10 TESTING ONLY

		X"0400", X"0400", X"0400", X"0400", X"0400", X"0400", X"0400", X"0400", X"0400", 		--<<-- ROIs = 1024
		X"0400", X"0400", X"0400", X"0400", X"0400", X"0400", X"0400", X"0400", X"0400", 		--<<-- ROIs = 1024
		X"0400", X"0400", X"0400", X"0400", X"0400", X"0400", X"0400", X"0400", X"0400",  		--<<-- ROIs = 1024
		X"0400", X"0400", X"0400", X"0400", X"0400", X"0400", X"0400", X"0400", X"0400",  		--<<-- ROIs = 1024

		X"61A8", X"0000", X"0000", X"0000", X"7080", X"7080", X"7080", X"7080",					 --<<-- DACs
		X"0000",
		X"0000", X"0000"  -- MSword // LSword
		);
		
		

-- signals used for MAC/IP calculation:
-- these 2 synch in, the jumper state.
signal FAD_in_cam : std_logic := '0';
signal FAD_at_ETHZ : std_logic := '0';
-- these 2 synch in the FMP lines encoding the FAD position in the cam. 
signal bid : std_logic_vector (3 downto 0);
signal cid : std_logic_vector (1 downto 0);
-- these are just used as local variables, to make reading easier.
signal mac_loc : mac_type;
signal ip_loc : ip_type;
signal gateway_loc : ip_type;
signal netmask_loc : ip_type;

	-- signals for synching in asynchronous input signals
		------------------------------------------------------------------------------
		signal w5300_interrupt_sr : std_logic_vector(1 downto 0) := "11";
		 --?? not sure if this init value is good
		 -- but should be no problem, because interrupt_ignore is not true during the first 2 clock cycles.
		 -- I hope!
		signal data_valid_sr : std_logic_vector(1 downto 0) := "00"; 
		------------------------------------------------------------------------------
		-- synch in CONFIG inputs
		signal memory_manager_config_valid_i_sr 	: std_logic_vector(1 downto 0) := "00";
		signal spi_interface_config_valid_i_sr		: std_logic_vector(1 downto 0) := "00";
		--signal data_generator_config_valid_i_sr		: std_logic_vector(1 downto 0) := "00";
		signal data_ram_empty_sr 					: std_logic_vector(1 downto 0) := (OTHERS => '0');


		signal data_generator_idle_sr : std_logic_vector(2 downto 0) := "000";

-- only for debugging
--signal error_cnt : std_logic_vector (7 downto 0) := (others => '0');
--signal last_trigger_id : std_logic_vector (15 downto 0) := (others => '0');
signal DG_run_mode_temp_storage_signal : std_logic;
signal data_generator_run_mode_signal : std_logic := '1'; -- default triggers will be accepted
-- signals for different socket modes: DN 04.01.11
signal socket_send_mode : std_logic := '0'; -- if 0 data is send via socket 0; if 1 data is send via the other sockets.
-- signals for Sockek Number calculation
signal event_number : std_logic_vector(31 downto 0);
signal mod7_start : std_logic := '0';
signal mod7_started : std_logic;
signal mod7_valid : std_logic;
signal mod7_result : std_logic_vector(2 downto 0);


signal state_sig : std_logic_vector(7 downto 0) := X"FF";

signal number_of_words_written_to_fifo :  std_logic_vector(15 downto 0) := (others => '0');
signal number_of_bytes_written_to_fifo :  std_logic_vector(16 downto 0) := (others => '0');

signal ready_wi_sr : std_logic_vector(1 downto 0) := "00";

signal read_wi : std_logic  := '0';
signal write_wi : std_logic := '0';
signal data_out_wi : std_logic_vector (15 downto 0);
signal ready_wi : std_logic;
signal reset_wi : std_logic  := '0';
signal wi_started : std_logic := '0';


COMPONENT mod7
	 PORT (
			clk       : IN     std_logic;
			number    : IN     std_logic_vector (31 DOWNTO 0);
			start     : IN     std_logic;
			remainder : OUT    std_logic_vector (2 DOWNTO 0) := (others => '0');
			started   : OUT    std_logic                     := '0';
			valid     : OUT    std_logic                     := '0'
	 );
	 END COMPONENT;

COMPONENT w5300_interface IS
	PORT(
		clk							: IN	std_logic;						-- expecting 50MHz
		
		wiz_reset					: OUT	std_logic;
		cs							: OUT	std_logic;
		wr							: OUT	std_logic;
		rd							: OUT	std_logic;
		int							: IN	std_logic;
		wiz_data					: INOUT	std_logic_vector (15 DOWNTO 0);
		wiz_addr					: OUT	std_logic_vector (9 DOWNTO 0);
		
		read_i						: IN	std_logic;
		write_i						: IN	std_logic;
		addr_i						: IN	std_logic_vector (9 DOWNTO 0);
		data_o						: OUT	std_logic_vector (15 DOWNTO 0)	:= (others => '0');
		data_i						: IN	std_logic_vector (15 DOWNTO 0);
		ready_o						: OUT	std_logic := '0';
		reset_i						: IN	std_logic
		
	);
	END COMPONENT;
	 

begin
	
	mod7_calculator : mod7
	PORT MAP (
		--locals => actuals
		clk			=>clk	,
		number		=>event_number	,
		start		=>mod7_start	,
		remainder	=>mod7_result	,
		started		=>mod7_started	,
		valid		=>mod7_valid
	);
	
	inst_w5300_interface : w5300_interface
	PORT MAP(
		clk				=> clk,
		wiz_reset		=> wiz_reset,
		cs				=> cs,		
		wr				=> wr,				
		rd				=> rd,					
		int				=> int,					
		wiz_data		=> data,
		wiz_addr		=> addr,
		read_i			=> read_wi,
		write_i			=> write_wi,
		addr_i			=> wiz_addr,
		data_o			=> data_read,
		data_i			=> wiz_data,
		ready_o			=> ready_wi,
		reset_i			=> reset_wi
	);


  --synthesis translate_off
  RST_TIME <= X"00120";
  --synthesis translate_on
  
  -- concurrent statemnets
  
-- output config settings as DAC and ROI arrays.
state <= state_sig;
debug_data_ram_empty <= int_flag;
debug_data_valid <= interrupt_ignore;
--debug_data_ram_empty <= data_ram_empty_sr(1);
--debug_data_valid <= data_valid_sr(1);
roi_mapping : for i in 0 to 35 generate
	roi_setting(i) <= conv_integer(config_setting(i)) when (conv_integer(config_setting(i)) < 1025) else 1024;
end generate roi_mapping;
dac_mapping : for i in 0 to 7 generate
	dac_setting(i) <= conv_integer(config_setting(i+36));
end generate dac_mapping;
c_trigger_mult <= config_setting(44);

runnumber <= config_setting(45) & config_setting(46);

trigger_enable <= trigger_enable_sig;


	w5300_proc : process (clk)
	begin
		if rising_edge (clk) then
		write_wi <= '0';
		read_wi <= '0';
		
	
		memory_manager_config_valid_i_sr	<= memory_manager_config_valid_i_sr(0) 	& memory_manager_config_valid_i;
		spi_interface_config_valid_i_sr		<= spi_interface_config_valid_i_sr(0)	& spi_interface_config_valid_i;
		data_ram_empty_sr 					<= data_ram_empty_sr(0) 		& data_ram_empty;		
		data_generator_idle_sr				<= data_generator_idle_sr( 1 downto 0 ) & data_generator_idle_i;
		
		w5300_interrupt_sr <= w5300_interrupt_sr(0) & int;
		data_valid_sr <= data_valid_sr(0) & data_valid;
		
		ready_wi_sr <= ready_wi_sr(0) & ready_wi;
		
			-- interrupt is handled synchronously
			-- W5300 pulls low its interrpt line in case of:
			--	When Sockets time out and 
			-- When sockets receive disconnection request.
			
			if (w5300_interrupt_sr(1) = '0') and (interrupt_ignore = '0') then
				case state_interrupt_1 is
					when IR1_01 =>
						int_flag <= '1';
						state_interrupt_1 <= IR1_02;  --wait one cycle
					when IR1_02 =>
						state_interrupt_1 <= IR1_03;
					when IR1_03 =>
						state_init <= INTERRUPT;
						socket_cnt <= "000";
						int_flag <= '0';
						interrupt_ignore <= '1';
						state_interrupt_1 <= IR1_04;  --this state is not existing? bad coding habit???
					when others =>
						null;
				end case;
			end if; -- int = '0'
			
			if int_flag = '0' then
				case state_init is
					-- Interrupt
					when INTERRUPT =>
						case state_interrupt_2 is
							when IR2_GET_IR =>
								if (ready_wi = '1') then
									wiz_addr <= W5300_IR;
									read_wi <= '1';
									state_interrupt_2 <= IR2_CHECK_WHICH_SN_IRQ;
								end if;
								
							-- check if it was a Sx Interrupt of Socket n
							-- therfor: loop over all channel.
							-- if the loop ended but no Sx Interrupt was found --> IR2_GOTO_RESET
							-- if an Sx Interrupt was found go to --> IR2_GET_SN_IR and check what happened.
							when IR2_CHECK_WHICH_SN_IRQ =>
								if (ready_wi = '1') then
									if (data_read(conv_integer(socket_cnt)) = '1') then -- Sx Interrupt
										state_interrupt_2 <= IR2_GET_SN_IR;
									else
										socket_cnt <= socket_cnt + 1;
										if (socket_cnt = 7) then
											state_interrupt_2 <= IR2_GOTO_RESET;
										else
											state_interrupt_2 <= IR2_CHECK_WHICH_SN_IRQ;  -- go on with loop
										end if; 
									end if;
								end if;
								
							-- check the Interrupt register of the Socket to be able to clear them
							when IR2_GET_SN_IR =>
								if (ready_wi = '1') then
									wiz_addr <= W5300_S0_IR + socket_cnt * W5300_S_INC; -- Sx Interrupt Register
									read_wi <= '1';
									state_interrupt_2 <= IR2_CLEAR_SN_IR;
								end if;
							-- clear IRQs by writing ones.
							when IR2_CLEAR_SN_IR =>
								if (ready_wi = '1') then
									wiz_addr <= W5300_S0_IR + socket_cnt * W5300_S_INC;
									wiz_data <= data_read; -- clear Interrupts
									write_wi <= '1';
									state_interrupt_2 <= IR2_SEND_DISCON;
								end if;
								
							-- send the command to close this socket
							-- then go back and read the main Interrupt register again, 
							-- if this was not Socket 7 ... if it was Socket 7, we're done anyway.
							when IR2_SEND_DISCON =>
								if (ready_wi = '1') then
									wiz_addr <= W5300_S0_CR + socket_cnt * W5300_S_INC;
									wiz_data <= X"0008"; -- DISCON  --> W5300 wil send FIN packet
									write_wi <= '1';
									socket_cnt <= socket_cnt + 1;
								
									if (socket_cnt = 7) then
										socket_cnt <= "000";
										state_interrupt_2 <= IR2_WAIT_SOCKn_CLOSED;
									else
										state_interrupt_2 <= IR2_GET_IR;
									end if; 
								end if;
								
							when IR2_GET_SOCKn_STATUS =>
								if (ready_wi = '1') then
									wiz_addr <= W5300_S0_CR + socket_cnt * W5300_S_INC;
									read_wi <= '1';
									state_interrupt_2 <= IR2_WAIT_SOCKn_CLOSED;
								end if;
								
							when IR2_WAIT_SOCKn_CLOSED =>
								if (ready_wi = '1') then
									if (data_read = X"0000") then --closed
										if (socket_cnt = 7) then
											socket_cnt <= "000";
											state_interrupt_2 <= IR2_GOTO_RESET;
										else
											socket_cnt <= socket_cnt + 1;
											state_interrupt_2 <= IR2_GET_SOCKn_STATUS;
										end if;
									end if;
								end if;
							
								
							
							when IR2_GOTO_RESET =>
							state_sig <= X"F6";
								state_interrupt_1 <= IR1_01;
								state_interrupt_2 <= IR2_GET_IR;
								socket_cnt <= "000";
								state_init <= RESET;
							--when others =>
								--null;
							end case;
						
					-- reset W5300
					when RESET =>
						state_sig <= X"01";  --debugging
						
						socket_send_mode <= '0';	-- stuff, not connected to W5300
						busy <= '1';
						socks_waiting <= '0';
						socks_connected <= '0';
						socket_cnt <= "000";
						interrupt_ignore <= '0';
						
						reset_wi <= '1';
						if (ready_wi = '0') then
							reset_wi <= '0';
							state_init <= WAIT_FOR_RESET;
						end if;
					
					when WAIT_FOR_RESET =>
						if (ready_wi = '1') then
							state_init <= INIT;
						end if;
						
					-- Init
					when INIT =>
					  state_sig <= X"02";
						-- status of MAC jumpers is synched in
						-- and Board- and CrateID are synched in
						FAD_in_cam <= MAC_jumper(1); -- see position of jumpers in FACT logbook
						FAD_at_ETHZ <= MAC_jumper(0); -- MAC_jumper(1) is where D_T(7) was; MAC_jumper(0) is where D_T(6) was;
						bid <= BoardID;
						cid <= CrateID;
						
						wiz_addr <= W5300_MR;
						wiz_data <= X"0000";
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= LOCATE;
						end if;
						
						
					when LOCATE =>
						state_sig <= X"03";					  
						state_init <= IM;

						if (FAD_in_cam = '1') then
							if (bid = "1111" and cid="11") then
								-- if BID = "1111" and CID="11" then FAD is not really in cam
								-- I don't know what to do now.
								-- I could wait a long time, and then just assume, I was not in cam.
								-- this should never happen!!!!!
								-- impossible to find this out, if in cam
								gateway_loc <= ETHZ_GATEWAY;
								netmask_loc <= ETHZ_NETMASK;					        
								mac_loc <= MAC_FAD2; 
								ip_loc <= IP_ETHZ_FAD2;
								--state_init <= INIT;
							else -- everything is fine
								-- IP & MAC are calculated from BID & CID
								-- code here
								gateway_loc <= CAM_GATEWAY;
								netmask_loc <= CAM_NETMASK;					        
								mac_loc <= (CAM_MAC_prefix (0), CAM_MAC_prefix (1)  , conv_std_logic_vector ( conv_integer(cid)*10+conv_integer(bid)  , 16)   ); 
								ip_loc <= ( CAM_IP_PREFIX(0) , CAM_IP_PREFIX(1) , IP_offset + conv_integer(cid) , IP_offset + conv_integer(bid) );
							end if;
						else -- FAD is tested, either at ETHZ or at TUDO AND either with FMP or without.
							if ( FAD_at_ETHZ = '0' ) then
								-- easy FAD is at TUDO -> only one choice.
								mac_loc <= MAC_FAD0;
								ip_loc <= IP_TUDO;
								gateway_loc <= TUDO_GATEWAY;
								netmask_loc <= TUDO_NETMASK;
							else -- FAD is at ETHZ but not in cam --> IP lookup table is needed.
								if (bid = "1111" and cid="11") then -- FAD is not in crate
									mac_loc <= MAC_FAD0;
									ip_loc <= IP_ETHZ_FAD0;
									gateway_loc <= ETHZ_GATEWAY;
									netmask_loc <= ETHZ_NETMASK;					        
								else
									-- FAD is at ETHZ and in crate:
									-- crate ID is not of importance.
									-- we only have 3 MACs and IPs so far, so only the first boardIDs are allowed.
									if ( conv_integer(bid) < MAC_LIST'length) then
										gateway_loc <= ETHZ_GATEWAY;
										netmask_loc <= ETHZ_NETMASK;					        
										mac_loc <= MAC_LIST(conv_integer(bid));
										ip_loc <= IP_LIST(conv_integer(bid));
									end if; -- conv_integer
								end if; -- bid=1111 & cid=11
							end if; --FAD_at_ETHZ = 0
						end if; --FAD_in_cam = 1
						
					-- Interrupt Mask
					when IM =>
						state_sig <= X"04";
						wiz_addr <= W5300_IMR;
						wiz_data <= X"00FF"; -- S0-S7 Interrupts
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= MT;
						end if;
						
						
					-- Memory Type
					when MT =>
						wiz_addr <=	W5300_MTYPER;
						wiz_data <= X"7FFF"; -- 8K RX, 120K TX-Buffer
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= STX0;
						end if;
					  
					-- Socket TX Memory Size
					when STX0 =>
						wiz_addr <= W5300_TMS01R;
						wiz_data <= X"0F0F"; -- 15K TX
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= STX1;
						end if;
					when STX1 =>
						wiz_addr <= W5300_TMS23R;
						wiz_data <= X"0F0F"; -- 15K TX
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= STX2;
						end if;
										
					when STX2 =>
						wiz_addr <= W5300_TMS45R;
						wiz_data <= X"0F0F"; -- 15K TX
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= STX3;
						end if;

					when STX3 =>
						wiz_addr <= W5300_TMS67R;
						wiz_data <= X"0F0F"; -- 15K TX
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= SRX0;
						end if;
			 		
					-- Socket RX Memory Size
					when SRX0 =>
						wiz_data <= X"0101"; -- 1K RX
						wiz_addr <= W5300_RMS01R;
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= SRX1;
						end if;
					when SRX1 =>
						wiz_data <= X"0101"; -- 1K RX
						wiz_addr <= W5300_RMS23R;
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= SRX2;
						end if;
					when SRX2 =>
						wiz_data <= X"0101"; -- 1K RX
						wiz_addr <= W5300_RMS45R;
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= SRX3;
						end if;
					when SRX3 =>
						wiz_data <= X"0101"; -- 1K RX
						wiz_addr <= W5300_RMS67R;
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= MAC0;
						end if;

					-- MAC
					when MAC0 =>
						wiz_addr <= W5300_SHAR;
						wiz_data <= mac_loc(0);
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= MAC1;
						end if;

					when MAC1 =>
						wiz_addr <= W5300_SHAR + 2;
						wiz_data <= mac_loc(1);
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= MAC2;
						end if;
					when MAC2 =>
						wiz_addr <= W5300_SHAR + 4;
						wiz_data <= mac_loc(2);
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= GW0;
						end if;
						
					-- Gateway
					when GW0 =>
						wiz_addr <= W5300_GAR;
						wiz_data (15 downto 8) <= conv_std_logic_vector(gateway_loc(0),8);
						wiz_data (7 downto 0) <= conv_std_logic_vector(gateway_loc(1),8);
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= GW1;
						end if;
						
					when GW1 =>
						wiz_addr <= W5300_GAR + 2;
						wiz_data (15 downto 8) <= conv_std_logic_vector(gateway_loc(2),8);
						wiz_data (7 downto 0) <= conv_std_logic_vector(gateway_loc(3),8);
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= SNM0;
						end if;
						
					-- Subnet Mask
					when SNM0 =>
						wiz_addr <= W5300_SUBR;
						wiz_data (15 downto 8) <= conv_std_logic_vector(netmask_loc(0),8);
						wiz_data (7 downto 0) <= conv_std_logic_vector(netmask_loc(1),8);
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= SNM1;
						end if;
					when SNM1 =>
						wiz_addr <= W5300_SUBR + 2;
						wiz_data (15 downto 8) <= conv_std_logic_vector(netmask_loc(2),8);
						wiz_data (7 downto 0) <= conv_std_logic_vector(netmask_loc(3),8);
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= IP0;
						end if;
					-- Own IP-Address
					when IP0 =>
						wiz_addr <= W5300_SIPR;
						wiz_data (15 downto 8) <= conv_std_logic_vector(ip_loc(0),8);
						wiz_data (7 downto 0) <= conv_std_logic_vector(ip_loc(1),8);
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= IP1;
						end if;
					when IP1 =>
						wiz_addr <= W5300_SIPR + 2;
						wiz_data (15 downto 8) <= conv_std_logic_vector(ip_loc(2),8);
						wiz_data (7 downto 0) <= conv_std_logic_vector(ip_loc(3),8);
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= SI_MR;
						end if;

					-- Socket Init
					when SI_MR =>
					  state_sig <= X"05";
						wiz_addr <= W5300_S0_MR + socket_cnt * W5300_S_INC;
						wiz_data <= X"0101"; -- ALIGN, TCP
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= SI_IMR;
						end if;
					-- Sx Interrupt Mask
					when SI_IMR =>
						wiz_addr <= W5300_S0_IMR + socket_cnt * W5300_S_INC;
						wiz_data <= X"000A"; -- TIMEOUT, DISCON
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= SI_PORTOR;
						end if;
					when SI_PORTOR =>
						wiz_addr <= W5300_S0_PORTOR + socket_cnt * W5300_S_INC;
						wiz_data <= X"0606"; -- send automatic KEEP ALIVE every 30s.
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= SI_PORT;
						end if;
					when SI_PORT =>
						wiz_addr <= W5300_S0_PORTR + socket_cnt * W5300_S_INC;
						wiz_data <= conv_std_logic_vector(FIRST_PORT + unsigned(socket_cnt), 16);
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= SI_CR_OPEN;
						end if;
					when SI_CR_OPEN =>
						wiz_addr <= W5300_S0_CR + socket_cnt * W5300_S_INC;
						wiz_data <= X"0001"; -- OPEN
						if (ready_wi = '1') then 
							write_wi <= '1';
							state_init <= SI_SSR;
						end if;
					when SI_SSR =>
						wiz_addr <= W5300_S0_SSR + socket_cnt * W5300_S_INC;
						if (ready_wi = '1') then 
							read_wi <= '1';
							state_init <= SI_IS_OPEN;
						end if;
					when SI_IS_OPEN =>
						if (ready_wi = '1') then
							if (data_read (7 downto 0) = X"13") then -- is open?
								state_init <= SI_CR_LISTEN;
							else
								state_init <= SI_CR_OPEN;
							end if;
						end if;
					when SI_CR_LISTEN =>
						wiz_addr <= W5300_S0_CR + socket_cnt * W5300_S_INC;
						wiz_data <= X"0002"; -- LISTEN
						if (ready_wi = '1') then 
							write_wi <= '1';
							socket_cnt <= socket_cnt + 1;
							if (socket_cnt = 7) then
								socket_cnt <= "000";
								state_init <= ESTABLISH;
							else
								state_init <= SI_MR;
							end if;
						end if;

						
				  -- End Socket Init
						
					when ESTABLISH =>
						state_sig <= X"07";
						socks_waiting <= '1';
						socks_connected <= '0';
						wiz_addr <= W5300_S0_SSR + socket_cnt * W5300_S_INC;
						if (ready_wi = '1') then 
							read_wi <= '1';
							state_init <= EST1;
						end if;
					when EST1 =>
						state_sig <= X"08";
						if (ready_wi = '1') then
							case data_read (7 downto 0) is
								when X"17" => -- established
									if (socket_cnt = 7) then
									  socket_cnt <= "000";
									  busy <= '0';
									  state_init <= MAIN;
									  --state_init <= CONFIG_MEMORY_MANAGER;
									else
									  socket_cnt <= socket_cnt + 1;
									  state_init <= ESTABLISH;
									end if;
								when others =>
									state_init <= ESTABLISH;
							end case;
						end if;
						
					when CONFIG =>   -- Triggers are disabled here!
					  state_sig <= X"06";
						trigger_enable_storage_sig <= trigger_enable_sig; 				-- store last value of this signal.
						trigger_enable_sig <= '0'; 										--no triggers must occur, while configurating.
						state_init <= WAIT_100NS;	 									
						
					-- this wait for 100ns is not entirely understood, maybe it is not even 
					-- necessary, but I guess under certain circumstances 
					-- DG might be triggered, just when I pull the trigger enable line low.
					-- when I check no immediatly, if DG is in IDLE, it might *just* be.
					-- but about to go into another state.
					-- DG and WM live in different clock domains.
					-- so I wait 100ns .. in order to let DG make one last transition out of IDLE
					-- then in the next state I wait until Dg returns.
					-- this must always happen, there is no way for DG to stay for a long time in any other state.
							-- in order to ensure this, the DG MM 'data was just written into RAM' handshake was tweaked!
					when WAIT_100NS =>   
						state_sig <= X"15";
						wait_100ns_sig <= wait_100ns_sig + 1;
						if (wait_100ns_sig = "100") then
							wait_100ns_sig <= "000";
							state_init <= WAIT_UNTIL_DG_IDLE;	
						end if;
						
					when WAIT_UNTIL_DG_IDLE =>
						state_sig <= X"16";
						if (data_generator_idle_sr = "111") then
							state_init <= CONFIG_MEMORY_MANAGER;
							--state_init <= MAIN;
						end if;
					
					when CONFIG_MEMORY_MANAGER =>
					  state_sig <= X"18";
						memory_manager_config_start_o <= '1';
						if (memory_manager_config_valid_i_sr = "00") then
							state_init <= WAIT_FOR_CONFIG_MEMORY_MANAGER;
						end if;
					
					when WAIT_FOR_CONFIG_MEMORY_MANAGER =>
					  state_sig <= X"19";
						memory_manager_config_start_o <= '0';
						if (memory_manager_config_valid_i_sr(1) = '1') then
							trigger_enable_sig <= trigger_enable_storage_sig;
							state_init <= MAIN;
						end if;
					
					when CONFIG_DAC_ONLY =>
					  state_sig <= X"1C";
						spi_interface_config_start_o <= '1';
						if (spi_interface_config_valid_i_sr ="00") then
							state_init <= WAIT_FOR_CONFIG_DAC_ONLY;
						end if;
						
					when WAIT_FOR_CONFIG_DAC_ONLY =>
					  state_sig <= X"1D";
						spi_interface_config_start_o <= '0';
						if (spi_interface_config_valid_i_sr ="11") then
							state_init <= MAIN;
						end if;
							
----------------------------------------------------------------------------------
-- MAIN "loop" -------------------------------------------------------------------
----------------------------------------------------------------------------------
          
					when MAIN =>
						state_sig <= X"20";
						socks_waiting <= '0';
						socks_connected <= '1';
						ps_do_phase_shift <= '0';
						ps_reset <= '0';
						data_valid_ack <= '0';
					
						
						if (update_of_rois = '1') then
							update_of_rois <= '0';
							state_init <= CONFIG;
--							if (trigger_enable_sig = '1') then 
--								trigger_enable_storage_sig <= trigger_enable_sig;
--							end if;
--							trigger_enable_sig <= '0';
--
--							update_of_rois <= '0';
--							state_init <= CONFIG_MEMORY_MANAGER;
--							if (data_ram_empty_sr(1) = '1') then
--								update_of_rois <= '0';
--								state_init <= CONFIG_MEMORY_MANAGER;
--							else 
--								state_init <= MAIN2;
--							end if;
							
						elsif (update_of_lessimportant = '1') then				
							update_of_lessimportant <= '0';
							state_init <= CONFIG_DAC_ONLY;
						else -- update_of_rois='0' and update_of_lessimportant='0' 
							state_init <= MAIN1;
							--data_valid_int <= data_valid;
						end if;
			
					when MAIN1 =>
					  state_sig <= X"21";
						if (chk_recv_cntr = 1000) then
							chk_recv_cntr <= 0;
							state_init <= READ_DATA;
							busy <= '1';
						else
							chk_recv_cntr <= chk_recv_cntr + 1;  
							state_init <= MAIN2;
						end if;
					when MAIN2 =>
					  state_sig <= X"22";
						busy <= '0';
						
						if (data_valid_sr(1) = '1') then
							busy <= '1';
							local_write_length <= write_length;
							local_ram_start_addr <= ram_start_addr;
							local_ram_addr <= (others => '0');
							local_write_header_flag <= write_header_flag;
							local_write_end_flag <= write_end_flag;
							local_fifo_channels <= fifo_channels;
							state_init <= MAIN3;
						else
							state_init <= MAIN1; 
						end if;
					when MAIN3 =>
					  state_sig <= X"23";
						-- needed for the check: if there is enough space in W5300 FIFO
						write_length_bytes <= (local_write_length (15 downto 0) & '0')   ; -- shift left (*2) 
						state_init <= WRITE_DATA;
						
						
----------------------------------------------------------------------------------
-- END OF MAIN         -----------------------------------------------------------
----------------------------------------------------------------------------------

					-- read data from socket 0  
					when READ_DATA =>
						case state_read_data is
						when RD_1 =>
						  state_sig <= X"30";
							if (ready_wi = '1') then 
								wiz_addr <= W5300_S0_RX_RSR;
								read_wi <= '1';
								state_read_data <= RD_2;
							end if;
							
						when RD_2 =>
						  state_sig <= X"31";
							if (ready_wi = '1') then
								wiz_addr <= W5300_S0_RX_RSR + X"2";
								read_wi <= '1';
								socket_rx_received (16) <= data_read(0);
								state_read_data <= RD_3;
							end if;
						when RD_3 =>
						  state_sig <= X"32";
							if (ready_wi = '1') then
								socket_rx_received (15 downto 0) <= data_read;
								state_read_data <= RD_4;
							end if;
							
						when RD_4 =>
						  state_sig <= X"33";
							if (socket_rx_received (16 downto 0) > ('0' & X"000")) then
								rx_packets_cnt <= socket_rx_received (16 downto 1); -- socket_rx_received / 2
								state_read_data <= RD_5;
							else
								busy <= '0';
								state_read_data <= RD_1;
								state_init <= MAIN;
							end if;
						
						when RD_5 =>
						  state_sig <= X"34";
							s_trigger <= '0';
							ps_do_phase_shift <= '0';
							reset_trigger_id <= '0';
							
							if (rx_packets_cnt > 0) then
								if (ready_wi = '1') then
									wiz_addr <= W5300_S0_RX_FIFOR;
									read_wi <= '1';
									rx_packets_cnt <= rx_packets_cnt - '1';
									state_read_data <= RD_6;
								end if;
							else
								state_read_data <= RD_END;
							end if;
							
						when RD_6 =>
						  state_sig <= X"35";
						  if (ready_wi = '1') then
								-- The next 16bit word is assumed to contain a 'command' so it is 
								-- beeing parsed in this state
								case data_read (15 downto 8) is
								when CMD_START =>
									
									state_read_data <= RD_5;
								when CMD_STOP => 
									
									state_read_data <= RD_5;
								when CMD_MODE_ALL_SOCKETS =>  -- all data will be send via socket 1..7 
									socket_send_mode <= '1';
									state_read_data <= RD_5;
								when CMD_MODE_COMMAND => -- all data will be send via socket 0
									socket_send_mode <= '0';
									state_read_data <= RD_5;   
								when CMD_TRIGGER =>
									s_trigger <= '1';
									state_read_data <= RD_5;
								when CMD_DWRITE_RUN =>
									dwrite_enable <= '1';
									state_read_data <= RD_5;
								when CMD_DWRITE_STOP =>
									dwrite_enable <= '0';
									state_read_data <= RD_5;
								when CMD_SCLK_ON =>
									sclk_enable <= '1';
									state_read_data <= RD_5;
								when CMD_SCLK_OFF =>
									sclk_enable <= '0';
									state_read_data <= RD_5;
								when CMD_DENABLE =>
									denable <= '1';
									state_read_data <= RD_5;
								when CMD_DDISABLE =>
									denable <= '0';
									state_read_data <= RD_5;
								when CMD_TRIGGER_C =>
									c_trigger_enable <= '1';
									state_read_data <= RD_5;
								when CMD_TRIGGER_S =>
								  c_trigger_enable <= '0';
								  state_read_data <= RD_5;

								-- phase shift commands here:
								when CMD_PS_DO =>
								  ps_do_phase_shift <= '1';
								  state_read_data <= RD_5;
								when CMD_PS_DIRINC =>
								  ps_direction <= '1';
								  state_read_data <= RD_5;
								when CMD_PS_RESET =>
								  ps_reset <= '1';
								  state_read_data <= RD_5;
								when CMD_SRCLK_ON =>
								  srclk_enable <= '1';
								  state_read_data <= RD_5;
								when CMD_SRCLK_OFF =>
								  srclk_enable <= '0';
								  state_read_data <= RD_5;
								when CMD_TRIGGERS_ON =>
								  trigger_enable_sig <= '1';
								  state_read_data <= RD_5;
								when CMD_TRIGGERS_OFF =>
								  trigger_enable_sig <= '0';
								  state_read_data <= RD_5;
								when CMD_PS_DIRDEC =>
								  ps_direction <= '0';
								  state_read_data <= RD_5;
								  
								when CMD_RESET_TRIGGER_ID =>
									reset_trigger_id <= '1';
									state_read_data <= WAIT_FOR_TRIGGER_ID_RESET_1;
									
								when CMD_WRITE =>
									config_addr <= conv_integer(data_read (7 downto 0));
									state_read_data <= READ_COMMAND_DATA_SECTION;
								when others =>
									state_read_data <= RD_5;
								end case;
							end if;
							
						when WAIT_FOR_TRIGGER_ID_RESET_1 =>
						  state_sig <= X"36";
							state_read_data <= WAIT_FOR_TRIGGER_ID_RESET_2;
						when WAIT_FOR_TRIGGER_ID_RESET_2 =>
						  state_sig <= X"37";
							state_read_data <= RD_5;
						-- these states are beeing processed, if the 'command' was a 'write command'
						--	so it is assumed, that some data in config RAM changed, and we need full (re)config
						when READ_COMMAND_DATA_SECTION =>
						  state_sig <= X"38";
							if (rx_packets_cnt > 0) then
								if (ready_wi = '1') then
									rx_packets_cnt <= rx_packets_cnt - '1';
									wiz_addr <= W5300_S0_RX_FIFOR;
									state_read_data <= PUT_COMMAND_DATA_SECTION;
								end if;
							else
								state_read_data <= RD_END;
							end if;
			
						when PUT_COMMAND_DATA_SECTION =>
							state_sig <= X"39";
							if (ready_wi = '1') then
								config_setting(config_addr) <= data_read;
								if (config_addr < 36) then
									update_of_rois <= '1';
								else 
									if (config_addr < 45 ) then
										update_of_lessimportant <= '1';
									end if;
								end if;
								state_read_data <= RD_5;
							end if;

						when RD_END =>
							state_sig <= X"3A";
							
							if (ready_wi = '1') then
								wiz_addr <= W5300_S0_CR;
								wiz_data <= X"0040"; -- RECV
								write_wi <= '1';
								state_read_data <= RD_1;
								state_init <= MAIN;
							end if;

						--when others =>
							--state_sig <= X"3F";
						end case; -- state_data_read
                
-----------------------------------------------------------------------------------------------
-----------------------------------------------------------------------------------------------
-- WRITE DATA
-----------------------------------------------------------------------------------------------
-----------------------------------------------------------------------------------------------

					when WRITE_DATA =>
						case state_write is
							
						when WR_START =>
						    state_sig <= X"40";
						    if (local_write_header_flag = '1') then
								ram_addr <= local_ram_start_addr + 6; -- Address of HIGH word of Event ID
								state_write <= WR_GET_EVT_ID_WAIT1;
							else
					      		state_write <= WR_CHECK_FOR_FIFO_SPACE_01;
						    end if;
						      
						when WR_GET_EVT_ID_WAIT1 =>
						    state_sig <= X"41";
						    state_write <= WR_GET_EVT_ID1;
						when WR_GET_EVT_ID1 =>
						    state_sig <= X"42";
							event_number(31 downto 16) <= ram_data;
							ram_addr <= local_ram_start_addr + 9; -- Address of LOW word of Event ID
							state_write <= WR_GET_EVT_ID_WAIT2;
						when WR_GET_EVT_ID_WAIT2 =>
							state_write <= WR_GET_EVT_ID2;
						when WR_GET_EVT_ID2 =>
							state_sig <= X"43";
							event_number(15 downto 0) <= ram_data;
							mod7_start <= '1';
							if (mod7_valid = '0') then
								state_write <= WR_MOD7_STARTED;
							else
								state_write <= WR_GET_EVT_ID2;
							end if;
						when WR_MOD7_STARTED =>
							state_sig <= X"44";
							if (mod7_started = '1') then
								mod7_start <= '0';
								state_write <= WR_WAIT_FOR_MOD7;
							end if;
						when WR_WAIT_FOR_MOD7 =>
							state_sig <= X"45";
							if (mod7_valid = '1') then
								if (socket_send_mode = '1') then -- send via all sockets
									local_socket_nr <= mod7_result + 1;
								else -- only send via socket 0\
									local_socket_nr <= "000";
								end if;
								data_cnt <= 0;
								state_write <= WR_CHECK_FOR_FIFO_SPACE_01;  
							else 
								state_write <= WR_WAIT_FOR_MOD7;
							end if;
								
							-- Check FIFO Size
						when WR_CHECK_FOR_FIFO_SPACE_01 =>
							state_sig <= X"46";
							if (ready_wi = '1') then
								wiz_addr <= W5300_S0_TX_FSR + local_socket_nr * W5300_S_INC;
								read_wi <= '1';
								state_write <= WR_CHECK_FOR_FIFO_SPACE_02;
							end if;
							
						when WR_CHECK_FOR_FIFO_SPACE_02 =>
							state_sig <= X"47";
							if (ready_wi = '1') then
								socket_tx_free (16) <= data_read(0);
								wiz_addr <= W5300_S0_TX_FSR + (local_socket_nr * W5300_S_INC) + X"2";
								state_write <= WR_CHECK_FOR_FIFO_SPACE_03;
								read_wi <= '1';
							end if;
							
						when WR_CHECK_FOR_FIFO_SPACE_03 =>
							state_sig <= X"48";
							if (ready_wi = '1') then
								socket_tx_free (15 downto 0) <= data_read;
								state_write <= WR_CHECK_FOR_FIFO_SPACE_04;
							end if;
							
						when WR_CHECK_FOR_FIFO_SPACE_04 =>
							state_sig <= X"49";
--								if (socket_tx_free (16 downto 0) < write_length_bytes) then
							if (socket_tx_free (16 downto 0) < W5300_TX_FIFO_SIZE_8B) then
							
								state_write <= WR_CHECK_FOR_FIFO_SPACE_01;
							else
								if (local_write_header_flag = '1') then
									state_write <= WR_HEADER_FETCH;
								else
									state_write <= WR_ADC;
								end if; 
							end if;
							
							-- Fill FIFO

						------------------------------------------------
						-- Write Header
						------------------------------------------------
						-- at this point we will write data from RAM directly into the W5300 TX FIFO
						-- after the we got the data from RAM
						--	and we made sure wi is ready to transmit
						--	we give the order to transmit: write_wi <= '1'
						-- after giving this order we can immediatly ask new data from RAM and 
						-- return to WR_HEADER 
						when WR_HEADER_FETCH =>
							state_sig <= X"4A";
							ram_addr <= local_ram_start_addr + local_ram_addr;
							state_write <= WR_HEADER_WAIT;
						
						when WR_HEADER_WAIT =>
							state_write <= WR_HEADER;
							
						when WR_HEADER =>
							state_sig <= X"4B";
							if (ready_wi = '1') then
								wiz_addr <= W5300_S0_TX_FIFOR + local_socket_nr * W5300_S_INC;
								wiz_data <= ram_data;
								number_of_words_written_to_fifo <= number_of_words_written_to_fifo + 1;
								data_cnt <= data_cnt + 1;
								local_ram_addr <= local_ram_addr + 1;
								write_wi <= '1';
								
								if (data_cnt < PACKAGE_HEADER_LENGTH) then
									state_write <= WR_HEADER_FETCH;
								else
									state_write <= WR_ADC;
								end if;
							end if;
							-- End Write Header
							
						------------------------------------------------
						-- Write ADC
						------------------------------------------------
						-- Writing of ADC data is performed in a manner such that, groups of DRS channels 
						-- with equal channel indices fit completely into W5300 FIFO
						-- note: 
							-- drs index		: counts from 0 to 3
							-- channel index	: counts from 0 to 9
						-- e.g. when the ROI is large, not the whole event data fits completly 
						-- into the FIFO
						-- in this case the Header is written first, 
						-- followed by the channels with index = 0 of all 4 DRS chips.
						-- and further followed by the channels, having index = 1 of all DRS chips.
						-- the channels are written 'side-by-side' into the data ram, using the 64bit input.
						-- but they are read out 'as if written consecutivly' 
						-- so the address needs to be calculated like this
						-- addr(n+1) = addr(n) + 4; instead of 
						-- addr(n+1) = addr(n) + 1 in case of the header.
						
						when WR_ADC =>
							state_sig <= X"4C";
							
							-- when this line is executed for the first time adc_data_address points
							-- to the beginning of a channel header. e.g. to 0x0000 or 0x0039
							adc_data_addr <= local_ram_start_addr + local_ram_addr;
							
							drs_cnt <= 0;		-- counts the chis form 0..3
							data_cnt <= 0;		-- counts the words in the ADC data channel group
							channel_cnt <= 1;	-- counts the number of channels which fit into W5300 FIFO at once.
							
							roi_max <= (others => '0');
							data_end <= POSITION_OF_ROI_IN_CHANNEL_HEADER;
							state_write <= WR_ADC1;

						---- Write Channel
						when WR_ADC1 =>
							state_sig <= X"4D";
							-- read ROI and set end of Channel-Data
							
							-- when this line is executed for the first time it points to
							-- the beginning of a channel header, as well.
							ram_addr <= adc_data_addr + drs_cnt + (data_cnt * 4);
							if (data_cnt = POSITION_OF_ROI_IN_CHANNEL_HEADER) then
								data_end <= conv_integer (ram_data) + CHANNEL_HEADER_SIZE;
								if (ram_data > roi_max) then
									roi_max <= ram_data (10 downto 0);
								end if;
							end if;
							state_write <= WR_ADC2;
							
						when WR_ADC2 =>
							if (data_cnt < data_end) then
								if (ready_wi = '1') then
									wiz_addr <= W5300_S0_TX_FIFOR + local_socket_nr * W5300_S_INC;
									wiz_data <= ram_data;
									write_wi <= '1';
									number_of_words_written_to_fifo <= number_of_words_written_to_fifo + 1;
									data_cnt <= data_cnt + 1;
									state_write <= WR_ADC1;
								end if;
							else
								-- Next DRS
								if (drs_cnt < 3) then
									drs_cnt <= drs_cnt + 1;
									data_cnt <= 0;
									data_end <= POSITION_OF_ROI_IN_CHANNEL_HEADER;
									state_write <= WR_ADC1;
								else
									-- Next Channel
									if (channel_cnt < local_fifo_channels) then
										channel_cnt <= channel_cnt + 1;
										roi_max <= (others => '0');
										drs_cnt <= 0;
										data_cnt <= 0;
										data_end <= POSITION_OF_ROI_IN_CHANNEL_HEADER;
										adc_data_addr <= adc_data_addr + ((conv_integer(roi_max) + CHANNEL_HEADER_SIZE) * 4);
										state_write <= WR_ADC1;
									else
										-- Ready
										if (local_write_end_flag = '1') then
											ram_addr <= adc_data_addr + ((conv_integer(roi_max) + CHANNEL_HEADER_SIZE) * 4);
											state_write <= WR_ENDFLAG_WAIT;
										else
											state_write <= WR_STRANGE_WAIT;
										end if;
									end if;
								end if;    
							end if;
							-- End Write ADC-Data

						-- Write End Package Flag
						when WR_ENDFLAG_WAIT =>
							state_write <= WR_ENDFLAG;
						when WR_ENDFLAG =>				
							if (ready_wi = '1') then
								wiz_addr <= W5300_S0_TX_FIFOR + local_socket_nr * W5300_S_INC;
								wiz_data <= ram_data;
								write_wi <= '1';
								number_of_words_written_to_fifo <= number_of_words_written_to_fifo + 1;					
								ram_addr <= adc_data_addr + ((conv_integer(roi_max) + CHANNEL_HEADER_SIZE) * 4) + 1;
								state_write <= WR_ENDFLAG2_WAIT;
							end if;
						when WR_ENDFLAG2_WAIT =>
							state_write <= WR_ENDFLAG2;
				
						when WR_ENDFLAG2 =>
							if (ready_wi = '1') then
								wiz_addr <= W5300_S0_TX_FIFOR + local_socket_nr * W5300_S_INC;
								wiz_data <= ram_data;
								write_wi <= '1';
								number_of_words_written_to_fifo <= number_of_words_written_to_fifo + 1;
								state_write <= WR_STRANGE_WAIT;
							end if;
              
						when WR_STRANGE_WAIT =>
							if (wait_cntr < 10) then -- 3000 works???
								wait_cntr <= wait_cntr + 1;
							else
								wait_cntr <= 0;
								state_write <= WR_05_PREPARE_LENGTH_INFO;
							end if;
						--Send FIFO	
						when WR_05_PREPARE_LENGTH_INFO =>
							--number_of_words_written_to_fifo <= number_of_words_written_to_fifo - 1;
							state_write <= WR_05_POSTPREPARE_LENGTH_INFO;
						when WR_05_POSTPREPARE_LENGTH_INFO =>
							number_of_bytes_written_to_fifo <= number_of_words_written_to_fifo(15 downto 0) & '0';    
							state_write <= WR_MESSAGE_LEN_HIGH_WORD;
							
						when WR_MESSAGE_LEN_HIGH_WORD =>
							if (ready_wi = '1') then
								wiz_addr <= W5300_S0_TX_WRSR + local_socket_nr * W5300_S_INC;
								wiz_data <= (0 => write_length_bytes (16), others => '0');
								--wiz_data <= (0 => number_of_bytes_written_to_fifo(16), others => '0');
								state_write <= WR_MESSAGE_LEN_LOW_WORD;
							end if;
								
						when WR_MESSAGE_LEN_LOW_WORD =>
							if (ready_wi= '1') then
								wiz_addr <= W5300_S0_TX_WRSR + (local_socket_nr * W5300_S_INC) + X"2";
								wiz_data <= write_length_bytes (15 downto 0);
								--wiz_data <= number_of_bytes_written_to_fifo(15 downto 0);
								state_write <= WR_SEND_COMMAND;
							end if;

						when WR_SEND_COMMAND =>
							number_of_words_written_to_fifo <= (others => '0');
							if (ready_wi <= '1') then
								wiz_addr <= W5300_S0_CR + local_socket_nr * W5300_S_INC;
								wiz_data <= X"0020"; -- Send
								write_wi <= '1';
							end if;
							state_write <= WR_ACK;
						
						when WR_ACK =>
							data_valid_ack <= '1';
							state_write <= WR_WAIT_FOR_ACK;
						
						when WR_WAIT_FOR_ACK =>
							state_write <= WR_WAIT_FOR_ACK;
							if (data_valid_sr(1) = '0') then
								data_valid_ack <= '0';
								
								state_write <= WR_START;
								state_init <= MAIN;
								-- maybe MAIN1 is better ?!
							end if;
							
						
						--when others =>
							--null;
						end case;
						-- End WRITE_DATA
						
				
					--when others =>
						--null;
				end case;
			end if; -- int_flag = '0'

		end if; -- rising_edge (clk)

	end process w5300_proc;

end Behavioral;

