//-----------------------------------------------------------------------------

#include "application.h"
#include "usart.h"
#include <avr/wdt.h> 

// in order to implement the "registers" I work with a quite long 
//  char-array like this:

U08 FSCregister[FSC_REGISTER_LENGTH];
// but this register is not only accessible by
// FSCregister[i], but as well by special pointers like this:
U32 *status 							= (U32*)&FSCregister[0];
U32 *time_sec 							= (U32*)&(FSCregister[4]);
U16 *time_ms 							= (U16*)&(FSCregister[6]);
U16 *FR_period 							= (U16*)&(FSCregister[8]);
U16 *ref_resistor 						= (U16*)&(FSCregister[10]);


U32 *ad7719_values 						= (U32*)&FSCregister[68];
U08 *ad7719_enables 					= &FSCregister[30];
U08 *ad7719_channels_ready 				= &FSCregister[49];
U08 *ad7719_readings_since_last_muxing 	= &FSCregister[14];
U08 *ad7719_current_channel 			= &FSCregister[15];
U32 *ad7719_current_reading 			= (U32*)&FSCregister[16];

U16 *adc_values 						= (U16*) &FSCregister[324];
U08 *adc_enables 						= &FSCregister[38];
U08 *adc_channels_ready 				= &FSCregister[57];
U08 *adc_readings_since_last_muxing 	= &FSCregister[20];
U08 *adc_current_channel 				= &FSCregister[21];
U16 *adc_current_reading 				= (U16*) &FSCregister[22];

// using these pointers one can manipulate measurement values like this:
// res_value[3] = 453212;
// and then readout the most significant byte of this same value by accessing:
// FSCregister[92]; 
// I like this very much for asking the boards status ... this is just a copy of the registers,
// into the W5100 TX FIFO.
// submitting the measurement values is just a partial copy... 

//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------

volatile U08 app_reset_source;
//-----------------------------------------------------------------------------

void app_init(void) {
  app_reset_source = MCUSR; // Save last reset source
  MCUSR = 0x00; // Clear reset source for next reset cycle


	// Dangerous here: I still do not know much about the watchdog.
	// This code is still from Udo Juerss. 

  // The watchdog timer is disabled by default ("startup.asm")
	#ifdef USE_WATCHDOG
	 WDTCSR = WDTOE | (1 << WDE); // Enable watchdog reset (~16ms)
	#endif

	// define PORTS
	// USART
	DDRD &= ~(1<<PD0);				// PD0 = RXD is input
	DDRD |= 1<<PD1; 					// PD1 = TXD is output
	

	// SPARE OUT/-INPUTS
	DDRB |= (1<<PB2) | (1<<PB3); 	// set Out1_spare & out2_spare as outputs
	DDRA &= ~(1<<PA7);				// set In1_spare as input
	DDRC &= ~(1<<PC7);				// set In2_spare as input
	//PORTA |= (1<<PA7); 				// swtich on pullup on In1_spare
	//PORTC |= (1<<PC7); 				// swtich on pullup on In2_spare

	// ATmega internal ADC input 
	DDRA &= ~(1<<PA6);

	// MUXER ADDRESS OUTs
	DDRA |= 0x3F; // SA-pins -> output
	DDRC |= 0x7F; // SB-pins -> output

	// SPI
	// set all CS's: output
	DDRB |= (1 << SPI_E_CS);
	DDRD |= (1 << SPI_AD_CS) |(1 << SPI_M_CS) |(1 << SPI_A_CS);

	// set all Chips selects HIGH
	PORTB |= (1 << SPI_E_CS);
	PORTD |= (1 << SPI_AD_CS) |(1 << SPI_M_CS) |(1 << SPI_A_CS);

	// set MOSI and SCK: output & // set MISO: input
	SPI_DDR |= (1 << SPI_MOSI);
	SPI_DDR |= (1 << SPI_SCLK);
	SPI_DDR &= ~(1 << SPI_MISO);

	// set MOSI, SCK: HIGH. MISO leave alone.
	SPI_PRT |= (1 << SPI_MOSI);
	SPI_PRT |= (1 << SPI_SCLK);
	//SPI_PRT |= (1 << SPI_MISO);

	// ADC 
	DDRD &= ~(1<<PD6);					// PD6 is AD_READY input
	DDRD |= 1<<PD7;							// PD7 is AD_RESET output

	// ACCELEROMETER
	DDRD &= ~(1<<PD2);					// PD2 is ACC_READY input

	//MAX6662   <--- not assembled 
	// DDRB &= ~(1<<PB0); 			// PB0 is over temperature alert input
	// DDRB &= ~(1<<PB1);				// PB1 is general temperature altert input
}



//-----------------------------------------------------------------------------

void app_set_watchdog_prescaler(tWDT_PRESCALE wdt_prescale) // Set watchdog prescale
{
  U08 sreg_backup = SREG; // Copy status register to variable
  U08 wdtcsr_value = WDE + wdt_prescale; // Set new prescale value to variable

  cli(); // Disable interrups
  wdt_reset(); // Reset watchdog

  WDTCR |= (1 << WDTOE) | (1 << WDE); // Unlock register access, 4 cycles to store new value
  WDTCR = wdtcsr_value; // Set new watchdog prescaler
  SREG = sreg_backup; // Restore status register
}

void set_ad7719_enable_register() {

	usart_write_str((pU08)"\n set enable bits of AD7719 Port ");
	if ((usart_received_chars>=5) && 
	(usart_rx_buffer[2] >= 'A' && usart_rx_buffer[2] <= 'H'))
	{
	usart_write_char(usart_rx_buffer[2]);
	usart_write_str((pU08)" to ");
	usart_write_U08_hex(usart_rx_buffer[4]);
	usart_write_char('\n');
		ad7719_enables[usart_rx_buffer[2]-'A']=usart_rx_buffer[4];
		ad7719_channels_ready[usart_rx_buffer[2]-'A']=0x00;
	}
	else if  ((usart_received_chars=3) && 
	(usart_rx_buffer[1] >= 'A' && usart_rx_buffer[1] <= 'H'))
	{
		usart_write_char(usart_rx_buffer[1]);
		if (usart_rx_buffer[2]!='0') {
			usart_write_str((pU08)" to 0xFF\n");
			ad7719_enables[usart_rx_buffer[1]-'A']=0xFF;
		} else
		{
			usart_write_str((pU08)" to 0x00\n");
			ad7719_enables[usart_rx_buffer[1]-'A']=0x00;
		}
		ad7719_channels_ready[usart_rx_buffer[1]-'A']=0x00;	
	}
	else
	{
		usart_write_str((pU08)"\n something wrong\n");
		usart_write_str((pU08)"usart_rx_buffer_index: ");
		usart_write_U08(usart_received_chars, 3);
		usart_write_str((pU08)"\n usart_rx_buffer[2]: ");
		usart_write_char(usart_rx_buffer[2]);
		usart_write_str((pU08)"\n usart_rx_buffer[4]: ");
		usart_write_U08_hex(usart_rx_buffer[4]);
		usart_write_char('\n');
	}
}

void set_adc_enable_register() {
	// TODO
	usart_write_str((pU08)"setting of ATmega internal ADC enable registers is not supported. yet.\n");
}

U08	increase_adc (U08 channel){
U08 effective_channel;
	for ( U08 increase = 1 ; increase <= VOLTAGE_CHANNELS; increase++)
	{
		effective_channel = (channel + increase) % (VOLTAGE_CHANNELS);
		if (adc_enables[effective_channel/8] & (1<<effective_channel%8)) {
			if (debug_mode)
			{
				usart_write_U08(effective_channel,3);
				usart_write_crlf();
			}
			return effective_channel;
		}
	}
	if (debug_mode)
	{
		usart_write_U08(channel,3);
		usart_write_crlf();
	}
	return channel;
} // end if increase_adc;

U08	increase_ad7719 (U08 channel){
U08 effective_channel;
	for ( U08 increase = 1 ; increase <= RESISTANCE_CHANNELS; increase++)
	{
		effective_channel = (channel + increase) % (RESISTANCE_CHANNELS);
		if (ad7719_enables[effective_channel/8] & (1<<effective_channel%8))
			return effective_channel;
	}
	return channel;
} // end if increase_adc;

void check_if_measured_all() {
	adc_measured_all = true;
	for ( U08 i=0; i<(VOLTAGE_REGS); ++i ) {
		if ((adc_enables[i] ^ adc_channels_ready[i]) != 0x00) {
			adc_measured_all = false;
			break;
		}
	}
	ad7719_measured_all = true;
	for ( U08 i=0; i<(RESISTANCE_CHANNELS/8); ++i ) {
		if ((ad7719_enables[i] ^ ad7719_channels_ready[i]) != 0x00) {
			ad7719_measured_all = false;
			break;
		}
	}


}

// an U08 array containts bitmaps, which encode, which channel is enabled and which is not.
// a measurement is done, when all (or more) enabled channels were already measured.
// a similar U08 array contains a bitmap, encoding which channels are already done.
//  note: "or more" above is important, if we check to strictly, 
//	we will never finish, in case a disabled channel get measured by any mistake...
//
//	lets assume:
//	enabled = 1110.0011		1110.0011
// 	done 	= 1111.0011		0111.0010
//
//	and		= 1110.0011		0110.0010
//	nand	= 0001.0011		1001.1101
//	or		= 1111.0011		1111.0011
//	xor		= 0001.0000		1001.0001
//	
//											
// (en xor done) and enabled =
//	xor		= 0001.0000		1001.0001
//	enabled = 1110.0011		1110.0011
//			  ---------		---------
//			  0000.0000		1000.0001
// if this statement evaluates to zero, 
bool check_if_adc_measurement_done(){
	adc_measured_all = true;
	for ( U08 i=0; i<VOLTAGE_REGS; ++i ) {
		if (( (adc_enables[i] ^ adc_channels_ready[i]) & adc_enables[i] )  != 0x00) {
			adc_measured_all = false;
			break;
		}
	}
	return adc_measured_all;
}

bool check_if_ad7719_measurement_done(){
	ad7719_measured_all = true;
	for ( U08 i=0; i<RESISTANCE_CHANNELS/8; ++i ) {
		if (( (ad7719_enables[i] ^ ad7719_channels_ready[i]) & ad7719_enables[i]) != 0x00) {
			ad7719_measured_all = false;
			break;
		}
	}
	return ad7719_measured_all;
}
