#ifndef __W5100_SPI_INTERFACE_H
#define __W5100_SPI_INTERFACE_H
//-----------------------------------------------------------------------------

#include "typedefs.h"
#include "application.h"
#include "num_conversion.h"

extern volatile BOOL sock0_connection_established;


#define ETH_READ_BUFFER_SIZE 4
#define ETH_WRITE_BUFFER_SIZE 32
extern volatile U08 eth_read_buffer[ETH_READ_BUFFER_SIZE];
extern volatile U08 eth_write_buffer[ETH_WRITE_BUFFER_SIZE];
extern U08 eth_write_index;
//-----------------------------------------------------------------------------
// Port Definitions
// Pin Definitions
// W5100 is SPI device 0
// -- so there is no need to define special pins.
// -- refer to the device by using the global buffers:
// 	volatile U08 spi_read_buffer[SPI_READ_BUFFER_SIZE];
//	volatile U08 spi_write_buffer[SPI_WRITE_BUFFER_SIZE];
//
//	and the spi function:
// 	void spi_transfer(U08 bytes, U08 device)
// 
//	when only sending one byte, you can do:
//	U08 returnvalue;
//	CLR_BIT(SPI_DEVICE_SS_PRT[0], SPI_DEVICE_SS[0]);  	// Set CS low
//	returnvalue = spi_transfer_byte(0xFF);     	
//  SET_BIT(SPI_DEVICE_SS_PRT[0], SPI_DEVICE_SS[0]);	// Set CS high
//
//	W5100 supports SPI modes 0 and 3  --> cpol/cpha= 0/0 or 1/1

// we only use socket 0 in this application. So only a few of W5100 adresses are mentioned here
// W5100 MEM Addresses
// Common registers:
#define CM_MR    0x0000		// mode register
#define CM_GAR0  0x0001		// Gateway adress 
#define CM_GAR1  0x0002		// Gateway adress 
#define CM_GAR2  0x0003		// Gateway adress 
#define CM_GAR3  0x0004		// Gateway adress 
#define CM_SUBR0 0x0005		// Subnetmask adress 
#define CM_SUBR1 0x0006		// Subnetmask adress 
#define CM_SUBR2 0x0007		// Subnetmask adress 
#define CM_SUBR3 0x0008		// Subnetmask adress 
#define CM_SHAR0 0x0009		// Source Hardware adress: MAC
#define CM_SHAR1 0x000A		// Source Hardware adress: MAC
#define CM_SHAR2 0x000B		// Source Hardware adress: MAC
#define CM_SHAR3 0x000C		// Source Hardware adress: MAC
#define CM_SHAR4 0x000D		// Source Hardware adress: MAC
#define CM_SHAR5 0x000E		// Source Hardware adress: MAC
#define CM_SIPR0 0x000F		// Source IP adress
#define CM_SIPR1 0x0010		// Source IP adress
#define CM_SIPR2 0x0011		// Source IP adress
#define CM_SIPR3 0x0012		// Source IP adress
#define CM_IR    0x0015		// Interrupt
#define CM_IMR   0x0016		// Source IP adress
#define CM_RTR0  0x0017		// retry time register
#define CM_RTR1  0x0018		// retry time register
#define CM_RCR   0x0019		// Retry count 
#define CM_RMSR  0x001A		// RX mem size					-- full mem size for sock0 --> value=0x03
#define CM_TMSR  0x001B		// TX mem size					-- full mem size for sock0 --> value=0x03

// Socket 0 registers
#define S0_MR 		0x0400		// Socket 0 mode			-- for TCP value = 0x01 (set bit 5 for no delay ACK)
#define S0_CR		0x0401		// socket 0 command			-- for commands see below
#define S0_IR		0x0402		// socket 0 interrupt		-- see bit description below
#define S0_SR		0x0403		// socket 0 status			-- see below
#define S0_PORT0	0x0404		// socket 0 Port MSB			-- FSC might get port number 0x1F5C = 8028, since we have only one FSC in cam
#define S0_PORT1	0x0405		// socket 0 Port LSB			-- 0xF5C1= 62913 is okay as well ... 
#define S0_MSSR0	0x0412		// max segment size			--
#define S0_MSSR1	0x0413		// max segment size			-- reset value is 0xFFFF; is set to other party value, if in TCP passive mode
#define S0_TX_FSR0	0x0420		// socket 0 TX free size
#define S0_TX_FSR1	0x0421		// socket 0 TX free size
#define S0_TX_RD0	0x0422		// socket 0 TX read pointer	-- read only:
#define S0_TX_RD1	0x0423		// socket 0 TX read pointer
#define S0_TX_WR0	0x0424		// socket 0 TX write pointer
#define S0_TX_WR1	0x0425		// socket 0 TX write pointer
#define S0_RX_RSR0	0x0426		// socket 0 RX received size 
#define S0_RX_RSR1	0x0427		// socket 0 RX received size 
#define S0_RX_RD0	0x0428		// socket 0 RX read pointer
#define S0_RX_RD1	0x0429		// socket 0 RX read pointer 



// some register values:
#define MR_TCP			0
#define MR_NOACKDELAY	5

#define CR_OPEN			0x01
#define CR_LISTEN		0x02
#define CR_CONECT		0x04
#define CR_DISCON		0x08
#define CR_CLOSE		0x10
#define CR_SEND			0x20
#define CR_RECV			0x40
#define CR_SEND_MAC		0x21
#define CR_SEND_KEEP	0x22

#define IR_SEND_OK		4
#define IR_TIMEOUT		3
#define IR_RECV			2
#define IR_DISCON		1
#define IR_CON			0

#define SR_SOCK_CLOSED			0x00
#define SR_SOCK_INIT			0x13
#define SR_SOCK_LISTEN			0x14
#define SR_SOCK_ESTABLISHED		0x17
#define SR_SOCK_CLOSE_WAIT		0x1C
#define SR_SOCK_UDP				0x22
#define SR_SOCK_IPRAW			0x32
#define SR_SOCK_MACRAW			0x42
#define SR_SOCK_PPPOE			0x5F
//changing
#define SR_SOCK_SYNSENT			0x15
#define SR_SOCK_SYNRECV			0x16
#define SR_SOCK_FIN_WAIT		0x18
#define SR_SOCK_CLOSING			0x1A
#define SR_SOCK_TIME_WAIT		0x1B
#define SR_SOCK_LAST_ACK		0x1D
#define SR_SOCK_ARP0			0x11
#define SR_SOCK_ARP1			0x21
#define SR_SOCK_ARP2			0x31

// low level functions
void w5100_write( U16 addr, U08 data);
U08 w5100_read( U16 addr);

//------------------------------------------------------------------------------
// Description of w5100_init() function:
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
// BASIC SETTINGS:
//------------------------------------------------------------------------------
// leave common MR in default state: 0x00
// leave IMR in default state: 0x00 no interrupt will occur, since #INT line is not routed
// leave RTR in default state: 0x07D0  --> 200ms
// leave RCR in default state: 0x08, after 8 re-transmissionsm, the TIMEOUT bit in Sn_IR is set '1'
//------------------------------------------------------------------------------
// NETWORK SETTING:
// set GAR to FSC_GATEWAY_ADDRESS
//#define FSC_GATEWAY_ADDRESS0 0xC0		// 192.33.96.1
//#define FSC_GATEWAY_ADDRESS1 0x21
//#define FSC_GATEWAY_ADDRESS2 0x60
//#define FSC_GATEWAY_ADDRESS3 0x01

#define FSC_GATEWAY_ADDRESS0 0xC0		// 192.168.0.1
#define FSC_GATEWAY_ADDRESS1 0x21
#define FSC_GATEWAY_ADDRESS2 0x60
#define FSC_GATEWAY_ADDRESS3 0x01


// set SHAR to FSC_MAC_ADDRESS
#define FSC_MAC_ADDRESS0 0xFA	//FA:C7:0F:AD:22:01
#define FSC_MAC_ADDRESS1 0xC7
#define FSC_MAC_ADDRESS2 0x0F 
#define FSC_MAC_ADDRESS3 0xAD
#define FSC_MAC_ADDRESS4 0x22
#define FSC_MAC_ADDRESS5 0x01
// set SUBR to FSC_SUBNET_MASK
#define FSC_SUBNET_MASK0 255	//255.255.248.0
#define FSC_SUBNET_MASK1 255
#define FSC_SUBNET_MASK2 255
#define FSC_SUBNET_MASK3 0
// set SIPR to FSC_IP_ADDRESS
//#define FSC_IP_ADDRESS0 0xC0	// 192.33.99.247
//#define FSC_IP_ADDRESS1 0x21
//#define FSC_IP_ADDRESS2 0x63
//#define FSC_IP_ADDRESS3 0xF7
#define FSC_IP_ADDRESS0 192	// 192.168.0.2
#define FSC_IP_ADDRESS1 168
#define FSC_IP_ADDRESS2 0
#define FSC_IP_ADDRESS3 2
//------------------------------------------------------------------------------
// MEM SETTINGS:
// we plan the possible use of 2 Sockets:
// Socket 0 for command input and requested data output
// Socket 1 as a Webserver
//
// set RMSR=0x0A --> 4k memory for Socket 0 and 1 
#define RX_MEM_BASE 0x6000
#define S0_RX_BASE  0x6000 			// since it is the first socket its base adress is just the RX MEM SPACE BASE
#define S0_RX_MASK  0x0FFF			// this is 4k - 1
#define S1_RX_BASE  0x7000			// 2nd socket start, right after first socket
#define S1_RX_MASK  0x0FFF			// again: 4k - 1
// set TMSR=0x0A --> 4k mem for each socket
#define TX_MEM_BASE 0x4000
#define S0_TX_BASE  0x4000 			// since it is the first socket its base adress is just the RX MEM SPACE BASE
#define S0_TX_MASK  0x0FFF			// this is 4k - 1
#define S1_TX_BASE  0x5000			// 2nd socket start, right after first socket
#define S1_TX_MASK  0x0FFF			// again: 4k - 1
//------------------------------------------------------------------------------
U08 w5100_init();


// -- returns: TRUE if socket status is == SR_SOCK_ESTABLISHED
BOOL w5100_is_established();
// request socket status info:
U08 w5100_sock_status();


U16 w5100_get_received_size();
U16 w5100_get_start_address();
//-----------------------------------------------------------------------------
#endif //__W5100_SPI_INTERFACE_H

U16 get_S0_TX_FSR();
U16 get_S0_TX_RD();
U16 get_S0_TX_WR();
U16 get_S0_RX_RSR();
U16 get_S0_RX_RD();
U08 w5100_get_RX(U08 NumBytes, BOOL send_ACK);
void w5100_TX(U08 NumBytes, U08 *str);
U16 w5100_set_TX(U08* string, U16 NumBytes);

extern bool w5100_needs_reset;
extern bool w5100_needs_init;
extern bool w5100_ready;



U08 w5100_caretaker();
void w5100_reset() ;


void eth_write_str( U08* str );
void eth_writeln_str( U08* str );