#ifndef FACT_EventBuilderWrapper
#define FACT_EventBuilderWrapper

/*
#if BOOST_VERSION < 104400
#if (__GNUC__ > 4 || (__GNUC__ == 4 && __GNUC_MINOR__ > 4))
#undef BOOST_HAS_RVALUE_REFS
#endif
#endif
#include <boost/thread.hpp>

using namespace std;
*/

#include <boost/date_time/posix_time/posix_time_types.hpp>

#include <CCfits/CCfits>

#include "EventBuilder.h"

extern "C" {
    extern void StartEvtBuild();
    extern int CloseRunFile(uint32_t runId, uint32_t closeTime);
}

class DataFileImp
{
    uint32_t fRunId;

public:
    DataFileImp(uint32_t id) : fRunId(id) { }

    virtual bool OpenFile(RUN_HEAD* h) = 0;
    virtual bool Write(EVENT *) = 0;
    virtual bool Close(RUN_TAIL * = 0) = 0;

    uint32_t GetRunId() const { return fRunId; }

    // --------------------------------------------------------------------------
    //
    //! This creates an appropriate file name for a particular run number and type
    //! @param runNumber the run number for which a filename is to be created
    //! @param runType an int describing the kind of run. 0=Data, 1=Pedestal, 2=Calibration, 3=Calibrated data
    //! @param extension a string containing the extension to be appened to the file name
    //
    string FormFileName(uint32_t runType, string extension)
    {
        //TODO where am I supposed to get the base directory from ?
        //TODO also, for creating subsequent directories, should I use the functions from the dataLogger ?
        string baseDirectory = "./Run";

        ostringstream result;
//        result << baseDirectory;
//        result << Time::fmt("/%Y/%m/%d/") << (Time() - boost::posix_time::time_duration(12,0,0));
        result << setfill('0') << setw(8) << fRunId;
        result << ".001_";
        switch (runType)
        {
        case -1:
            result << 'T';
            break;
        case 0:
            result << 'D';
            break;
        case 1:
            result << 'P';
            break;
        case 2:
            result << 'C';
            break;
        case 3:
            result << 'N';
            break;
        default:
            result << runType;
        };
        result << "." << extension;

        return result.str();
    }
};

class DataFileNone : public DataFileImp
{
public:
    DataFileNone(uint32_t id) : DataFileImp(id) { }

    bool OpenFile(RUN_HEAD* h)
    {
        cout << "OPEN_FILE #" << GetRunId() << " (" << this << ")" <<  endl;
        cout << " Ver= " << h->Version << endl;
        cout << " Typ= " << h->RunType << endl;
        cout << " Nb = " << h->NBoard << endl;
        cout << " Np = " << h->NPix << endl;
        cout << " NTm= " << h->NTm << endl;
        cout << " roi= " << h->Nroi << endl;

        return true;
    }
    bool Write(EVENT *)
    {
        return true;
    }
    bool Close(RUN_TAIL * = 0)
    {
        cout << "CLOSE FILE #" << GetRunId() << " (" << this << ")" << endl;
        return true;
    }
};

class DataFileDebug : public DataFileNone
{
public:
    DataFileDebug(uint32_t id) : DataFileNone(id) { }

    bool Write(EVENT *e)
    {
        cout << "WRITE_EVENT #" << GetRunId() << " (" << e->EventNum << ")" << endl;
        cout << " Typ=" << e->TriggerType << endl;
        cout << " roi=" << e->Roi << endl;
        cout << " trg=" << e->SoftTrig << endl;
        cout << " tim=" << e->PCTime << endl;

        return true;
    }
};

#include "FAD.h"

class DataFileRaw : public DataFileImp
{
    ofstream fOut;

    off_t fPosTail;

    uint32_t fCounter;


    // WRITE uint32_t 0xFAC77e1e  (FACT Tele)
    // ===
    // WRITE uint32_t TYPE(>0)          == 1
    // WRITE uint32_t ID(>0)            == 0
    // WRITE uint32_t VERSION(>0)       == 1
    // WRITE uint32_t LENGTH
    // -
    // WRITE uint32_t TELESCOPE ID
    // WRITE uint32_t RUNID
    // ===
    // WRITE uint32_t TYPE(>0)          == 2
    // WRITE uint32_t ID(>0)            == 0
    // WRITE uint32_t VERSION(>0)       == 1
    // WRITE uint32_t LENGTH
    // -
    // WRITE          HEADER
    // ===
    // [ 40 TIMES
    //    WRITE uint32_t TYPE(>0)       == 3
    //    WRITE uint32_t ID(>0)         == 0..39
    //    WRITE uint32_t VERSION(>0)    == 1
    //    WRITE uint32_t LENGTH
    //    -
    //    WRITE          BOARD-HEADER
    // ]
    // ===
    // WRITE uint32_t TYPE(>0)          == 4
    // WRITE uint32_t ID(>0)            == 0
    // WRITE uint32_t VERSION(>0)       == 1
    // WRITE uint32_t LENGTH
    // -
    // WRITE          FOOTER (empty)
    // ===
    // [ N times
    //    WRITE uint32_t TYPE(>0)       == 10
    //    WRITE uint32_t ID(>0)         == counter
    //    WRITE uint32_t VERSION(>0)    == 1
    //    WRITE uint32_t LENGTH HEADER
    //    -
    //    WRITE          HEADER+DATA
    // ]
    // ===
    // WRITE uint32_t TYPE   ==0
    // WRITE uint32_t VERSION==0
    // WRITE uint32_t LENGTH ==0
    // ===
    // Go back and write footer

public:
    DataFileRaw(uint32_t id) : DataFileImp(id)  { }
    ~DataFileRaw() { Close(); }

    void WriteBlockHeader(uint32_t type, uint32_t ver, uint32_t cnt, uint32_t len)
    {
        const uint32_t val[4] = { type, ver, cnt, len };

        fOut.write(reinterpret_cast<const char*>(val), sizeof(val));
    }

    template<typename T>
        void WriteValue(const T &t)
    {
        fOut.write(reinterpret_cast<const char*>(&t), sizeof(T));
    }

    enum
    {
        kEndOfFile = 0,
        kIdentifier = 1,
        kRunHeader,
        kBoardHeader,
        kRunSummary,
        kEvent,
    };

    virtual bool OpenFile(RUN_HEAD *h)
    {
        const string name = FormFileName(h->RunType, "bin");

        errno = 0;
        fOut.open(name.c_str(), ios_base::out);
        if (!fOut)
        {
            //ostringstream str;
            //str << "Open file " << name << ": " << strerror(errno) << " (errno=" << errno << ")";
            //Error(str);

            return false;
        }

        fCounter = 0;

        static uint32_t FACT = 0xFAC77e1e;

        fOut.write(reinterpret_cast<char*>(&FACT), 4);

        WriteBlockHeader(kIdentifier, 1, 0, 8);
        WriteValue(uint32_t(0));
        WriteValue(GetRunId());

        WriteBlockHeader(kRunHeader, 1, 0, sizeof(RUN_HEAD)-sizeof(PEVNT_HEADER*));
        fOut.write(reinterpret_cast<char*>(h), sizeof(RUN_HEAD)-sizeof(PEVNT_HEADER*));

        for (int i=0; i<40; i++)
        {
            WriteBlockHeader(kBoardHeader, 1, i, sizeof(PEVNT_HEADER));
            fOut.write(reinterpret_cast<char*>(h->FADhead+i), sizeof(PEVNT_HEADER));
        }

        // FIXME: Split this
        const vector<char> block(sizeof(uint32_t)+sizeof(RUN_TAIL));
        WriteBlockHeader(kRunSummary, 1, 0, block.size());

        fPosTail = fOut.tellp();
        fOut.write(block.data(), block.size());

        if (!fOut)
        {
            //ostringstream str;
            //str << "Open file " << name << ": " << strerror(errno) << " (errno=" << errno << ")";
            //Error(str);

            return false;
        }

        return true;
    }
    virtual bool Write(EVENT *evt)
    {
        const int sh = sizeof(EVENT)-2 + NPIX*evt->Roi*2;

        WriteBlockHeader(kEvent, 1, fCounter++, sh);
        fOut.write(reinterpret_cast<char*>(evt)+2, sh);
        return true;
    }
    virtual bool Close(RUN_TAIL *tail= 0)
    {
        WriteBlockHeader(kEndOfFile, 0, 0, 0);

        if (tail)
        {
            fOut.seekp(fPosTail);

            WriteValue(uint32_t(1));
            fOut.write(reinterpret_cast<char*>(tail), sizeof(RUN_TAIL));
        }

        if (!fOut)
        {
            //ostringstream str;
            //str << "Open file " << name << ": " << strerror(errno) << " (errno=" << errno << ")";
            //Error(str);

            return false;
        }

        fOut.close();

        if (!fOut)
        {
            //ostringstream str;
            //str << "Open file " << name << ": " << strerror(errno) << " (errno=" << errno << ")";
            //Error(str);

            return false;
        }

        return true;
    }
};

#ifdef HAS_FITS
class DataFileFits : public DataFileImp
{
    CCfits::FITS*  fFile;        /// The pointer to the CCfits FITS file
    CCfits::Table* fTable;       /// The pointer to the CCfits binary table

    uint64_t fNumRows;                ///the number of rows that have been written already to the FITS file.

public:
    DataFileFits(uint32_t runid) : DataFileImp(runid), fFile(0)
    {
    }

    // --------------------------------------------------------------------------
    //
    //! Default destructor
    //! The Fits file SHOULD have been closed already, otherwise the informations
    //! related to the RUN_TAIL will NOT be written to the file.
    //
    ~DataFileFits() { Close(); }

    // --------------------------------------------------------------------------
    //
    //! Add a new column to the vectors storing the column data.
    //! @param names the vector of string storing the columns names
    //! @param types the vector of string storing the FITS data format
    //! @param numElems the number of elements in this column
    //! @param type the char describing the FITS data format
    //! @param name the name of the particular column to be added.
    //
    inline void AddColumnEntry(vector<string>& names, vector<string>& types, int numElems, char type, string name)
    {
        names.push_back(name);

        ostringstream str;
        if (numElems != 1)
            str << numElems;
        str << type;
        types.push_back(str.str());
    }

    // --------------------------------------------------------------------------
    //
    //! Writes a single header key entry
    //! @param name the name of the key
    //! @param value its value
    //! @param comment the comment associated to that key
    //
    //FIXME this function is a duplicate from the class Fits. should we try to merge it ?
    template <typename T>
    void WriteKey(const string &name, const T &value, const string &comment)
    {
        try
        {
            fTable->addKey(name, value, comment);
        }
        catch (CCfits::FitsException e)
        {
            ostringstream str;
            str << "Could not add header key ";
            //TODO pipe the error message somewhere
        }
    }

    template <typename T>
    void WriteKey(const string &name, const int idx, const T &value, const string &comment)
    {
        ostringstream str;
        str << name << idx;

        WriteKey(str.str(), value, comment);
    }

    // --------------------------------------------------------------------------
    //
    //! DataFileFits constructor. This is the one that should be used, not the default one (parameter-less)
    //! @param runid This parameter should probably be removed. I first thought it was the run number, but apparently it is not
    //! @param h a pointer to the RUN_HEAD structure that contains the informations relative to this run
    //
    bool OpenFile(RUN_HEAD* h)
    {
        //Form filename, based on runid and run-type
        const string fileName = FormFileName(h->RunType, "fits");

        //create the FITS object
        try
        {
            fFile = new CCfits::FITS(fileName, CCfits::RWmode::Write);
        }
        catch (CCfits::FitsException e)
        {
            ostringstream str;
            str << "Could not open FITS file " << fileName << " reason: " << e.message();
            //TODO display the message somewhere

            return false;
        }

        //create columns according to header
        ostringstream arrayTypes;

        // uint32_t EventNum ;       // EventNumber as from FTM
        // uint16_t TriggerType ;    // Trigger Type from FTM
        // uint32_t SoftTrig ;       // SoftTrigger Info (TBD)
        // uint32_t PCTime ;         // when did event start to arrive at PC
        // uint32_t BoardTime[NBOARDS];//
        // int16_t  StartPix[NPIX];  // First Channel per Pixel (Pixels sorted according Software ID)  ; -1 if not filled
        // int16_t  StartTM[NTMARK]; // First Channel for TimeMark (sorted Hardware ID) ; -1 if not filled
        // uint16_t Adc_Data[];      // final length defined by malloc ....

        vector<string> colNames;
        vector<string> dataTypes;
        AddColumnEntry(colNames, dataTypes, 1,              'V', "EventNum");
        AddColumnEntry(colNames, dataTypes, 1,              'U', "TriggerType");
        AddColumnEntry(colNames, dataTypes, 1,              'V', "SoftTrig");
        AddColumnEntry(colNames, dataTypes, 1,              'V', "PCTime");
        AddColumnEntry(colNames, dataTypes, NBOARDS,        'V', "BoardTime");
        AddColumnEntry(colNames, dataTypes, NPIX,           'I', "StartPix");
        AddColumnEntry(colNames, dataTypes, NTMARK,         'I', "StartTM");
        AddColumnEntry(colNames, dataTypes, NPIX*h->Nroi,   'U', "Data");

        //actually create the table
        try
        {
            fTable = fFile->addTable("Events", 0, colNames, dataTypes);
            if (fTable->rows() != 0)
            {
                ostringstream str;
                str << "Error: table created on the fly looks non-empty.";
                //TODO giev the error text to some error handler
                //FIXME I guess that this error checking is useless. remove it for performances.
            }
        }
        catch (const CCfits::FitsException &e)
        {
            ostringstream str;
            str << "Could not create FITS table " << "Events" << " in file " << fileName << " reason: " << e.message();
            //TODO give the error text to some error handler

            Close();
            return false;
        }

        //write header data
        //first the "standard" keys
        string stringValue;
        WriteKey("EXTREL",   1.0f,   "Release Number");
        WriteKey("TELESCOP", "FACT", "Telescope that acquired this data");
        WriteKey("ORIGIN",   "ISDC", "Institution that wrote the file");
        WriteKey("CREATOR",  "FACT++ Event Builder", "Program that wrote this file");

        stringValue = Time().GetAsStr();
        stringValue[10]= 'T';
        WriteKey("DATE",     stringValue, "File creation data");
        WriteKey("TIMESYS",  "TT",        "Time frame system");
        WriteKey("TIMEUNIT", "d",         "Time unit");
        WriteKey("TIMEREF",  "UTC",       "Time reference frame");
        //FIXME should we also put the start and stop time of the received data ?
        //now the events header related variables
        WriteKey("VERSION",  h->Version,  "Builder version");
        WriteKey("RUNTYPE",  h->RunType,  "Type of run");
        WriteKey("NBOARD",   h->NBoard,   "Number of acquisition boards");
        WriteKey("NPIX",     h->NPix,     "Number of pixels");
        WriteKey("NTM",      h->NTm,      "Number of Time marks");
        WriteKey("NROI",     h->Nroi,     "Number of slices per pixels");

        //now the boards related keywords
        for (int i=0; i<h->NBoard; i++)
        {
            const PEVNT_HEADER &hh = h->FADhead[i];

            WriteKey("STPKGFG", i, hh.start_package_flag,
                     "Start package flag");

            WriteKey("PKGLEN",  i, hh.package_length,
                     "Package length");

            WriteKey("VERNO",   i, hh.version_no,
                     "Version number");

            WriteKey("PLLLCK",  i, hh.PLLLCK,
                     "");
/*
            WriteKey("TRIGCRC", i, hh.trigger_crc,
                     "Trigger CRC");

            WriteKey("TRIGTYP", i, hh.trigger_type,
                     "Trigger type");

            WriteKey("TRIGID",  i, hh.trigger_id,
                     "Trigger ID");

            WriteKey("EVTCNTR", i, hh.fad_evt_counter,
                     "FAD Event Counter");
*/
            WriteKey("REFCLK",  i, hh.REFCLK_frequency,
                     "Reference Clock Frequency");

            WriteKey("BOARDID", i, hh.board_id,
                     "Board ID");

            WriteKey("PHASESH", i, hh.adc_clock_phase_shift,
                     "ADC clock phase shift");

            //WriteKey("TRGGEN",  i, hh.number_of_triggers_to_generate,
            //         "Number of triggers to generate");

            WriteKey("PRESC",   i, hh.trigger_generator_prescaler,
                    "Trigger generator prescaler");

            WriteKey("DNA",     i, hh.DNA,       "DNA");
            WriteKey("TIME",    i, hh.time,      "Time");
            WriteKey("RUNNB",   i, hh.runnumber, "Run number");
/*
            for (int j=0;j<NTemp;j++)
            {
                str.str(""); str2.str("");
                str << "DRS_T" << i << j;
                str2 << "DRS temperature #" << i << " " << j;
                WriteKey(str.str(), h->FADhead[i].drs_temperature[j], str2.str());
            }
*/
            for (int j=0;j<NDAC;j++)
                WriteKey("DAC", i*NDAC+j, hh.dac[j], "DAC");

            //Last but not least, add header keys that will be updated when closing the file
            WriteFooter(NULL);
        }

        return true;
    }


    int WriteColumns(size_t &start, size_t size, void *e)
    {
        int status = 0;
        fits_write_tblbytes(fFile->fitsPointer(), fNumRows, start, size, reinterpret_cast<unsigned char*>(e), &status);
        if (status)
        {
            char text[30];//max length of cfitsio error strings (from doc)
            fits_get_errstatus(status, text);
            //ostringstream str;
            //str << "Writing FITS row " << i << " in " << groupName << ": " << text << " (file_write_tblbytes, rc=" << status << ")";
            //Error(str);
        }

        start += size;
        return status;
    }

    // --------------------------------------------------------------------------
    //
    //! This writes one event to the file
    //! @param e the pointer to the EVENT
    //
    virtual bool Write(EVENT *e)
    {
        //FIXME As discussed earlier, we do not swap the bytes yet.
        fTable->makeThisCurrent();

        //insert a new row
        int status(0);
        if (fits_insert_rows(fTable->fitsPointer(), fNumRows, 1, &status))
        {
            //ostringstream str;
            //str << "Inserting row into " << fFileName << " failed (fits_insert_rows, rc=" << status << ")";
            //fMess->Error(str);
            //TODO pipe this error message to the appropriate error stream
        }
        fNumRows++;

        const int sh = sizeof(EVENT)+NPIX*e->Roi*2;

        // column size pointer
        size_t col = 1;
        if (!WriteColumns(col, sh, e))
            return true;

        //TODO output an error
        return false;

        /*
        //write the data, chunk by chunk
        //FIXME hard-coded size corresponds to current variables of the event, in bytes.
        //FIXME no padding was taken into account. Because smallest member is 2 bytes, I don't think that this should be a problem.
        const long sizeInBytesOfEventBeforePointers = 16;

        long col = 1;
        if (FitsWriteTblBytes(col, sizeInBytesOfEventBeforePointers, e))
        {
            //TODO output an error
            return false;
        }
        if (FitsWriteTblBytes(col, NBOARDS*2, e->BoardTime))
        {
            //TODO output an error
            return false;
        }
        if (FitsWriteTblBytes(col, NPIX*2, e->StartPix))
        {
            //TODO output an error
            return false;
        }
        if (FitsWriteTblBytes(col, NTMARK*2, e->StartTM))
        {
            //TODO output an error
            return false;
        }
        if (FitsWriteTblBytes(col, NPIX*fRoi*2, e->Adc_Data))
        {
            //TODO output an error
            return false;
        }
        return true;*/
    }

    void WriteFooter(RUN_TAIL *rt)
    {
        //write final header keys
        fTable->makeThisCurrent();

        WriteKey("NBEVTOK",  rt ? rt->nEventsOk  : uint32_t(0),
                 "How many events were written");

        WriteKey("NBEVTREJ", rt ? rt->nEventsRej : uint32_t(0),
                 "How many events were rejected by SW-trig");

        WriteKey("NBEVTBAD", rt ? rt->nEventsBad : uint32_t(0),
                 "How many events were rejected by Error");

        //FIXME shouldn't we convert start and stop time to MjD first ?
        //FIXME shouldn't we also add an MjD reference ?

        WriteKey("TSTART",   rt ? rt->PCtime0    : uint32_t(0),
                 "Time when first event received");

        WriteKey("TSTOP",    rt ? rt->PCtimeX    : uint32_t(0),
                 "Time when last event received");
    }

    // --------------------------------------------------------------------------
    //
    //! Closes the file, and before this it write the TAIL data
    //! @param rt the pointer to the RUN_TAIL data structure
    //
    virtual bool Close(RUN_TAIL *rt = 0)
    {
        if (!fFile)
            return false;

        WriteFooter(rt);

        delete fFile;
        fFile = NULL;

        return true;
    }

};
#else
#define DataFileFits DataFileRaw
#endif

class EventBuilderWrapper
{
public:
    // FIXME
    static EventBuilderWrapper *This;

    MessageImp &fMsg;

private:
    boost::thread fThread;

    enum CommandStates_t // g_runStat
    {
        kAbort      = -2,  // quit as soon as possible ('abort')
        kExit       = -1,  // stop reading, quit when buffered events done ('exit')
        kInitialize =  0,  // 'initialize' (e.g. dim not yet started)
        kHybernate  =  1,  // do nothing for long time ('hybernate') [wakeup within ~1sec]
        kSleep      =  2,  // do nothing ('sleep')                   [wakeup within ~10msec]
        kModeFlush  = 10,  // read data from camera, but skip them ('flush')
        kModeTest   = 20,  // read data and process them, but do not write to disk ('test')
        kModeFlag   = 30,  // read data, process and write all to disk ('flag')
        kModeRun    = 40,  // read data, process and write selected to disk ('run')
    };

    enum
    {
        kCurrent = 0,
        kTotal   = 1
    };

    enum FileFormat_t
    {
        kNone,
        kDebug,
        kFits,
        kRaw
    };

    FileFormat_t fFileFormat;


    uint32_t fMaxRun;
    uint32_t fNumEvts[2];

    DimDescribedService fDimFiles;
    DimDescribedService fDimRuns;
    DimDescribedService fDimEvents;
    DimDescribedService fDimCurrentEvent;
    DimDescribedService fDimEventData;
    DimDescribedService fDimFwVersion;
    DimDescribedService fDimStatus;
    DimDescribedService fDimStatistics;

    bool fDebugStream;
    bool fDebugRead;

    int Write(const Time &time, const std::string &txt, int qos)
    {
        return fMsg.Write(time, txt, qos);
    }

public:
    EventBuilderWrapper(MessageImp &imp) : fMsg(imp),
        fFileFormat(kRaw), fMaxRun(0),
        fDimFiles       ("FAD_CONTROL/FILES",            "X:1", ""),
        fDimRuns        ("FAD_CONTROL/RUNS",             "I:1", ""),
        fDimEvents      ("FAD_CONTROL/EVENTS",           "I:2", ""),
        fDimCurrentEvent("FAD_CONTROL/CURRENT_EVENT",    "I:1", ""),
	fDimEventData   ("FAD_CONTROL/EVENT_DATA",       "S:1;I:1;S:1;I:2;S:1;S", ""),
        fDimFwVersion   ("FAD_CONTROL/FIRMWARE_VERSION", "F:43", ""),
        fDimStatus      ("FAD_CONTROL/STATUS",           "S:42", ""),
        fDimStatistics  ("FAD_CONTROL/STATISTICS",       "X:8",  ""),
        fDebugStream(false), fDebugRead(false)
   {
        if (This)
            throw logic_error("EventBuilderWrapper cannot be instantiated twice.");

        This = this;

        memset(fNumEvts, 0, sizeof(fNumEvts));

        fDimRuns.Update(uint32_t(0));
        fDimCurrentEvent.Update(uint32_t(0));
        fDimEvents.Update(fNumEvts);

        for (size_t i=0; i<40; i++)
            ConnectSlot(i, tcp::endpoint());
    }
    ~EventBuilderWrapper()
    {
        Abort();

        // FIXME: Used timed_join and abort afterwards
        //        What's the maximum time the eb need to abort?
        fThread.join();
        //ffMsg.Info("EventBuilder stopped.");

        for (vector<DataFileImp*>::iterator it=fFiles.begin(); it!=fFiles.end(); it++)
            delete *it;
    }

    bool IsThreadRunning()
    {
        return !fThread.timed_join(boost::posix_time::microseconds(0));
    }

    void SetMaxMemory(unsigned int mb) const
    {
        if (mb*1000000<GetUsedMemory())
        {
            // ffMsg.Warn("...");
            return;
        }

        g_maxMem = mb*1000000;
    }

    void StartThread(const vector<tcp::endpoint> &addr)
    {
        if (IsThreadRunning())
        {
            fMsg.Warn("Start - EventBuilder still running");
            return;
        }

        for (size_t i=0; i<40; i++)
            ConnectSlot(i, addr[i]);

        g_runStat = kModeRun;

        fMsg.Message("Starting EventBuilder thread");

        fThread = boost::thread(StartEvtBuild);
    }
    void ConnectSlot(unsigned int i, const tcp::endpoint &addr)
    {
        if (i>39)
            return;

        if (addr==tcp::endpoint())
        {
            DisconnectSlot(i);
            return;
        }

        g_port[i].sockAddr.sin_family      = AF_INET;
        g_port[i].sockAddr.sin_addr.s_addr = htonl(addr.address().to_v4().to_ulong());
        g_port[i].sockAddr.sin_port        = htons(addr.port());
        // In this order
        g_port[i].sockDef = 1;
    }
    void DisconnectSlot(unsigned int i)
    {
        if (i>39)
            return;

        g_port[i].sockDef = 0;
        // In this order
        g_port[i].sockAddr.sin_family      = AF_INET;
        g_port[i].sockAddr.sin_addr.s_addr = 0;
        g_port[i].sockAddr.sin_port        = 0;
    }
    void IgnoreSlot(unsigned int i)
    {
        if (i>39)
            return;
        if (g_port[i].sockAddr.sin_port==0)
            return;

        g_port[i].sockDef = -1;
    }


    void Abort()
    {
        fMsg.Message("Signal abort to EventBuilder thread...");
        g_runStat = kAbort;
    }

    void Exit()
    {
        fMsg.Message("Signal exit to EventBuilder thread...");
        g_runStat = kExit;
    }

    /*
    void Wait()
    {
        fThread.join();
        ffMsg.Message("EventBuilder stopped.");
    }*/

    void Hybernate() const { g_runStat = kHybernate; }
    void Sleep()     const { g_runStat = kSleep;     }
    void FlushMode() const { g_runStat = kModeFlush; }
    void TestMode()  const { g_runStat = kModeTest;  }
    void FlagMode()  const { g_runStat = kModeFlag;  }
    void RunMode()   const { g_runStat = kModeRun;   }

    // FIXME: To be removed
    void SetMode(int mode) const { g_runStat = mode; }

    bool IsConnected(int i) const     { return gi_NumConnect[i]==7; }
    bool IsDisconnected(int i) const  { return gi_NumConnect[i]<=0; }
    int  GetNumConnected(int i) const { return gi_NumConnect[i]; }

    void SetIgnore(int i, bool b) const { if (g_port[i].sockDef!=0) g_port[i].sockDef=b?-1:1; }
    bool IsIgnored(int i) const { return g_port[i].sockDef==-1; }

    void SetDebugStream(bool b)
    {
        fDebugStream = b;
        if (b)
            return;

        for (int i=0; i<40; i++)
        {
            if (!fDumpStream[i].is_open())
                continue;

            fDumpStream[i].close();

            ostringstream name;
            name << "socket_dump-" << setfill('0') << setw(2) << i << ".bin";
            fMsg.Message("Closed file '"+name.str()+"'");
        }
    }

    void SetDebugRead(bool b)
    {
        fDebugRead = b;
        if (b || !fDumpRead.is_open())
            return;

        fDumpRead.close();
        fMsg.Message("Closed file 'socket_events.txt'");
    }

    size_t GetUsedMemory() const { return gi_usedMem; }

    virtual int CloseOpenFiles() { CloseRunFile(0, 0); return 0; }


    /*
     struct OpenFileToDim
     {
        int code;
        char fileName[FILENAME_MAX];
     };

     SignalRunOpened(runid, filename);
     // Send num open files
     // Send runid, (more info about the run?), filename via dim

     SignalEvtWritten(runid);
     // Send num events written of newest file

     SignalRunClose(runid);
     // Send new num open files
     // Send empty file-name if no file is open

     */

    // -------------- Mapped event builder callbacks ------------------

    vector<DataFileImp*> fFiles;

    FileHandle_t runOpen(uint32_t runid, RUN_HEAD *h, size_t)
    {
        // Check if file already exists...
        DataFileImp *file = 0;
        switch (fFileFormat)
        {
        case kNone:  file = new DataFileNone(runid);  break;
        case kDebug: file = new DataFileDebug(runid); break;
        case kFits:  file = new DataFileFits(runid);  break;
        case kRaw:   file = new DataFileRaw(runid);   break;
        }

        try
        {
            if (!file->OpenFile(h))
                return 0;
        }
        catch (const exception &e)
        {
            return 0;
        }

        fFiles.push_back(file);

        if (runid>fMaxRun)
        {
            fMaxRun = runid;
            fNumEvts[kCurrent] = 0;

            fDimRuns.Update(fMaxRun);
            fDimEvents.Update(fNumEvts);
            fDimCurrentEvent.Update(uint32_t(0));
        }

        fDimFiles.Update(fFiles.size());

        return reinterpret_cast<FileHandle_t>(file);
    }

    int runWrite(FileHandle_t handler, EVENT *e, size_t)
    {
        DataFileImp *file = reinterpret_cast<DataFileImp*>(handler);

        if (!file->Write(e))
            return -1;

        if (file->GetRunId()==fMaxRun)
        {
            fDimCurrentEvent.Update(e->EventNum);
            fNumEvts[kCurrent]++;
        }

        fNumEvts[kTotal]++;
        fDimEvents.Update(fNumEvts);

        // ===> SignalEvtWritten(runid);
        // Send num events written of newest file

        /* close run runId (all all runs if runId=0) */
        /* return: 0=close scheduled / >0 already closed / <0 does not exist */
        //CloseRunFile(file->GetRunId(), time(NULL)+2) ;

        return 0;
    }

    int runClose(FileHandle_t handler, RUN_TAIL *tail, size_t)
    {
        DataFileImp *file = reinterpret_cast<DataFileImp*>(handler);

        const vector<DataFileImp*>::iterator it = find(fFiles.begin(), fFiles.end(), file);
        if (it==fFiles.end())
        {
            ostringstream str;
            str << "File handler (" << handler << ") requested to close by event builder doesn't exist.";
            fMsg.Fatal(str);
            return -1;
        }

        ostringstream str;
        str << "Closing file for run " << file->GetRunId() << " (" << file << ")" << endl;
        fMsg.Info(str);

        fFiles.erase(it);

        fDimFiles.Update(fFiles.size());

        const bool rc = file->Close(tail);
        if (!rc)
        {
            // Error message
        }

        delete file;

        // ==> SignalRunClose(runid);
        // Send new num open files
        // Send empty file-name if no file is open

        return rc ? 0 : -1;
    }

    ofstream fDumpStream[40];

    void debugStream(int isock, void *buf, int len)
    {
        if (!fDebugStream)
            return;

        const int slot = isock/7;
        if (slot<0 || slot>39)
            return;

        if (!fDumpStream[slot].is_open())
        {
            ostringstream name;
            name << "socket_dump-" << setfill('0') << setw(2) << slot << ".bin";

            fDumpStream[slot].open(name.str().c_str(), ios::app);
            if (!fDumpStream[slot])
            {
                ostringstream str;
                str << "Open file '" << name << "': " << strerror(errno) << " (errno=" << errno << ")";
                fMsg.Error(str);

                return;
            }

            fMsg.Message("Opened file '"+name.str()+"' for writing.");
        }

        fDumpStream[slot].write(reinterpret_cast<const char*>(buf), len);
    }

    ofstream fDumpRead; // Stream to possibly dump docket events

    void debugRead(int isock, int ibyte, int32_t event, int state, uint32_t tsec, uint32_t tusec)
    {
        //   isock = socketID (0-279)
        //   ibyte = #bytes gelesen
        //   event = eventId (oder 0 wenn noch nicht bekannt)
        //   state : 1=finished reading data
        //           0=reading data
        //          -1=start reading data (header)
        //          -2=start reading data,
        //             eventId not known yet (too little data)
        //   tsec, tusec = time when reading seconds, microseconds
        //
        if (!fDebugRead || ibyte==0)
            return;

        if (!fDumpRead.is_open())
        {
            fDumpRead.open("socket_events.txt", ios::app);
            if (!fDumpRead)
            {
                ostringstream str;
                str << "Open file 'socket_events.txt': " << strerror(errno) << " (errno=" << errno << ")";
                fMsg.Error(str);

                return;
            }

            fMsg.Message("Opened file 'socket_events.txt' for writing.");
        }

        fDumpRead
            << setw(2) << state   << " "
            << setw(8) << tsec    << " "
            << setw(9) << tusec   << " "
            << setw(3) << isock   << " "
            << setw(2) << isock/7 << " "
            << event << " "
            << ibyte << endl;
    }

    	struct DimEventData
	{
	    uint16_t Roi ;            // #slices per pixel (same for all pixels and tmarks)
	    uint32_t EventNum ;       // EventNumber as from FTM
	    uint16_t TriggerType ;    // Trigger Type from FTM

	    uint32_t PCTime ;         // when did event start to arrive at PC
	    uint32_t BoardTime;       //

	    int16_t StartPix;         // First Channel per Pixel (Pixels sorted according Software ID)  ; -1 if not filled
	    int16_t StartTM;          // First Channel for TimeMark (sorted Hardware ID) ; -1 if not filled

	    uint16_t Adc_Data[];     // final length defined by malloc ....

	} __attribute__((__packed__));;

        template<typename T>
        vector<T> CheckVals(const PEVNT_HEADER *fadhd, const T *val, bool &rc)
        {
            const size_t offset = reinterpret_cast<const char*>(val)-reinterpret_cast<const char*>(fadhd);

            vector<T> vec(40);

            vec[0] = *val;

            rc = true;
            for (int i=1; i<40; i++)
            {

                const T &t = *reinterpret_cast<const T*>(reinterpret_cast<const char*>(fadhd+i)+offset);
                if (t!=*val)
                    rc = false;

                vec[i] = t;
            }

            return vec;
        }
       /*
        template<typename T>
        vector<uint8_t> CheckBits(const PEVNT_HEADER *fadhd, const T &val, T &rc)
        {
            const size_t offset = reinterpret_cast<const char*>(&val)-reinterpret_cast<const char*>(fadhd);

            vector<uint8_t> vec(40);

            rc = 0;
            for (int i=0; i<40; i++)
            {
                const T &t = *reinterpret_cast<const T*>(reinterpret_cast<const char*>(fadhd+i)+offset);

                rc |= val^t;

                vec[i] = t;
            }

            // Return 1 for all bits which are identical
            //        0 for all other bits
            rc = ~rc;
            return vec;
        }*/


    int eventCheck(PEVNT_HEADER *fadhd, EVENT *event)
    {
        /*
         fadhd[i] ist ein array mit den 40 fad-headers
         (falls ein board nicht gelesen wurde, ist start_package_flag =0 )

         event  ist die Struktur, die auch die write routine erhaelt;
         darin sind im header die 'soll-werte' fuer z.B. eventID
         als auch die ADC-Werte (falls Du die brauchst)

         Wenn die routine einen negativen Wert liefert, wird das event
         geloescht (nicht an die write-routine weitergeleitet [mind. im Prinzip]
         */

        bool     ok_verno;
        bool     ok_runno;
       // uint16_t ok_bitmask;
/*
        const vector<uint16_t> verno = CheckVals(fadhd, &fadhd->version_no, ok_verno);
        const vector<uint32_t> runno = CheckVals(fadhd, &fadhd->runnumber,  ok_runno);
       // const vector<uint8_t>  bitmask = CheckBits(fadhd, &fadhd->PLLLCK,     ok_bitmask);

        static vector<uint16_t> fStoreVersion;

        if (verno!=fStoreVersion)
        {
            vector<float> data(40);
            for (int i=0; i<40; i++)
            {
                ostringstream ver;
                ver << (verno[i]&0xff) << '.' << (verno[i]>>8); // WARNING: No byte-swap yet!
                data[i] = atof(ver.str().c_str());
            }
            fDimFwVersion.Update(data);
            fStoreVersion=verno;
        }
*/
        /*
         uint16_t start_package_flag;
         uint16_t package_length;
         uint16_t version_no;
         uint16_t PLLLCK;

         uint16_t trigger_crc;
         uint16_t trigger_type;
         uint32_t trigger_id;

         uint32_t fad_evt_counter;
         uint32_t REFCLK_frequency;

         uint16_t board_id;
         uint8_t  zeroes;
         int8_t   adc_clock_phase_shift;
         uint16_t number_of_triggers_to_generate;
         uint16_t trigger_generator_prescaler;

         uint64_t DNA;

         uint32_t time;
         uint32_t runnumber;

         int16_t  drs_temperature[NTemp];

         uint16_t dac[NDAC];
         */

	static DimEventData *data = 0;

	const size_t sz = sizeof(DimEventData)+event->Roi*2;

	if (data && data->Roi != event->Roi)
	{
	    delete data;
	    data = 0;
	}

        if (!data)
	    data = reinterpret_cast<DimEventData*>(new char[sz]);

//        cout << sizeof(DimEventData) << " " << event->Roi << " " << sz << " " << sizeof(*data) << endl;

	data->Roi         = event->Roi;
	data->EventNum    = event->EventNum;
	data->TriggerType = event->TriggerType;
	data->PCTime      = event->PCTime;
	data->BoardTime   = event->BoardTime[0];
	data->StartPix    = event->StartPix[0];
	data->StartTM     = event->StartTM[0];

        memcpy(data->Adc_Data, event->Adc_Data, event->Roi*2);

        fDimEventData.setData(data, sz);
	fDimEventData.updateService();

        //delete data;

	return 0;
    }

    void factOut(int severity, int err, const char *message)
    {
        // FIXME: Make the output to the console stream thread-safe
        ostringstream str;
        str << "EventBuilder(";
        if (err<0)
            str << "---";
        else
            str << err;
        str << "): " << message;

        fMsg.Update(str, severity);
    }

    void factStat(int64_t *stat, int len)
    {
        if (len!=7)
        {
            fMsg.Warn("factStat received unknown number of values.");
            return;
        }

        vector<int64_t> data(1, g_maxMem);
        data.insert(data.end(), stat, stat+len);

        static vector<int64_t> last(8);
        if (data==last)
            return;
        last = data;

        fDimStatistics.Update(data);

        //   len ist die Laenge des arrays.
        //   array[4] enthaelt wieviele bytes im Buffer aktuell belegt sind; daran
        //   kannst Du pruefen, ob die 100MB voll sind ....

        /*
         stat[0]= qwait;
         stat[1]= qskip;
         stat[2]= qdel ;
         stat[3]= qtot ;
         stat[4]= gi_usedMem ;
         stat[5]= qread;
         stat[6]= qconn;
         */

        ostringstream str;
        str
            << "Wait=" << stat[0] << " "
            << "Skip=" << stat[1] << " "
            << "Del="  << stat[2] << " "
            << "Tot="  << stat[3] << " "
            << "Mem="  << stat[4] << "/" << g_maxMem << " "
            << "Read=" << stat[5] << " "
            << "Conn=" << stat[6];

        fMsg.Info(str);
    }

    boost::array<FAD::EventHeader, 40> fVecHeader;

    template<typename T>
    pair<bool,boost::array<T, 43>> Compare(const FAD::EventHeader *h, const T *t, const uint64_t mask=~0, const uint8_t shift=0)
    {
        const int offset = reinterpret_cast<const char *>(t) - reinterpret_cast<const char *>(h);

        const T *min = NULL;
        const T *val = NULL;
        const T *max = NULL;

        boost::array<T, 43> vec;

        bool rc = true;
        for (int i=0; i<40; i++)
        {
            const char *base = reinterpret_cast<const char*>(&fVecHeader[i]);
            const T *ref = reinterpret_cast<const T*>(base+offset);

            vec[i+3] = (*ref&mask)>>shift;

//            if (gi_NumConnect[i/7]!=7)
//                continue;

            if (!val)
            {
                min = ref;
                val = ref;
                max = ref;
            }

            if (*val<*min)
                min = val;

            if (*val>*max)
                max = val;

            if ((*val&mask)!=(*ref&mask))
                rc = false;
        }

        if (!val)
            return make_pair(false, vec);

        vec[0] = (*min&mask)>>shift;
        vec[1] = (*val&mask)>>shift;
        vec[2] = (*max&mask)>>shift;

        return make_pair(rc, vec);
    }

    template<typename T>
    boost::array<T, 42> CompareBits(const FAD::EventHeader *h, const T *t)
    {
        const int offset = reinterpret_cast<const char *>(t) - reinterpret_cast<const char *>(h);

        T val;  // All bits 0
        T rc;

        boost::array<T, 42> vec;

        bool first = true;

        for (int i=0; i<40; i++)
        {
            const char *base = reinterpret_cast<const char*>(&fVecHeader[i]);
            const T *ref = reinterpret_cast<const T*>(base+offset);

            vec[i+2] = *ref;

            if (gi_NumConnect[i/7]!=7)
                continue;

            if (first)
            {
                first = false;
                val = *ref;
                rc = ~0;
            }

            rc |= val^*ref;
        }

        vec[0] = rc;
        vec[1] = val;

        return vec;
    }

    template<typename T, size_t N>
    void Update(DimDescribedService &svc, const pair<bool,boost::array<T, N>> &data)
    {
        svc.setQuality(data.first);
        svc.setData(const_cast<T*>(data.second.data()), sizeof(T)*N);
        svc.updateService();
    }

    template<typename T>
        void Print(const char *name, const pair<bool,boost::array<T, 43>> &data)
    {
        cout << name << "|" << data.first << "|" << data.second[1] << "|" << data.second[0] << "<x<" << data.second[1] << ":";
        for (int i=0; i<40;i++)
            cout << " " << data.second[i+3];
        cout << endl;
    }

    void debugHead(int socket, const FAD::EventHeader &h)
    {
        const uint16_t id = h.Id();
        if (id>39) 
            return;

        const FAD::EventHeader old = fVecHeader[id];

        fVecHeader[id] = h;

        if (old.fVersion != h.fVersion)
        {
            const pair<bool, boost::array<uint16_t,43>> ver = Compare(&h, &h.fVersion);
            //Print("Ver", ver);

            pair<bool, boost::array<float,43>> data;
            data.first = ver.first;

            for (int i=0; i<43; i++)
            {
                ostringstream str;
                str << (ver.second[i]>>8) << '.' << (ver.second[i]&0xff);
                data.second[i] = atof(str.str().c_str());
            }
            Update(fDimFwVersion, data);
        }

        if (old.fTriggerType != h.fTriggerType)
        {
            const pair<bool, boost::array<uint16_t,43>> typ = Compare(&h, &h.fTriggerType);
            Print("Typ", typ);
        }

        if (old.fRunNumber != h.fRunNumber)
        {
            const pair<bool, boost::array<uint32_t,43>> run = Compare(&h, &h.fRunNumber);
            Print("Run", run);
        }

        if (old.fStatus != h.fStatus)
        {
            const boost::array<uint16_t,42> sts = CompareBits(&h, &h.fStatus);
            fDimStatus.setData(const_cast<uint16_t*>(sts.data()), 42*sizeof(uint16_t));
            fDimStatus.updateService();
        }
    }
};

EventBuilderWrapper *EventBuilderWrapper::This = 0;

// ----------- Event builder callbacks implementation ---------------
extern "C"
{
    FileHandle_t runOpen(uint32_t irun, RUN_HEAD *runhd, size_t len)
    {
        return EventBuilderWrapper::This->runOpen(irun, runhd, len);
    }

    int runWrite(FileHandle_t fileId, EVENT *event, size_t len)
    {
        return EventBuilderWrapper::This->runWrite(fileId, event, len);
    }

    int runClose(FileHandle_t fileId, RUN_TAIL *runth, size_t len)
    {
        return EventBuilderWrapper::This->runClose(fileId, runth, len);
    }

    void factOut(int severity, int err, const char *message)
    {
        EventBuilderWrapper::This->factOut(severity, err, message);
    }

    void factStat(int64_t *array, int len)
    {
        EventBuilderWrapper::This->factStat(array, len);
    }

    void debugHead(int socket, void *buf)
    {
        EventBuilderWrapper::This->debugHead(socket, FAD::EventHeader(vector<uint16_t>(reinterpret_cast<uint16_t*>(buf),
                                                                                       reinterpret_cast<uint16_t*>(buf)+sizeof(FAD::EventHeader)/2)));
    }

    void debugStream(int isock, void *buf, int len)
    {
        return EventBuilderWrapper::This->debugStream(isock, buf, len);
    }

    void debugRead(int isock, int ibyte, int32_t event, int state, uint32_t tsec, uint32_t tusec)
    {
        return EventBuilderWrapper::This->debugRead(isock, ibyte, event, state, tsec, tusec);
    }

    int eventCheck(PEVNT_HEADER *fadhd, EVENT *event)
    {
        return EventBuilderWrapper::This->eventCheck(fadhd, event);
    }
}

#endif
