/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Oscar Blanch  11/2001 < mailto:blanch@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  MMcPedestalNSBAdd                                                      //
//                                                                         //
//  This Task adds the contribution of the diffuse NSB to the FADC         //
//  pedestals. We assume that NSB introduces larger fluctuation but does   //
//  not change the mean value.                                             //
//  To be precise we add quadratically to the rms that is already in       //
//  MPedestalCam the NSB contribution.                                     //
//  The number of photons from the diffuse NSB follows a poisson           // 
//  distribution with expected mean value X, then its standard desviation  //
//  is sqrt(X).                                                            //
//  X is the number of phe per ns so we have to convert in FADC counts per // 
//  slice:                                                                 //
//                                                                         //
//  Y=sqrt(X*FADC_time/Number_of_slice*pixel_size)*Amp_single_phe_response //
//                                                                         //
//  Input Containers:                                                      //
//   MMcFadcHeader                                                         //
//   MMcRunHeader                                                          //
//   MRawRunHeader                                                         //
//                                                                         //
//  Output Containers:                                                     //
//   MPedestalCam                                                          //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MMcPedestalNSBAdd.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MPedestalCam.h"
#include "MRawRunHeader.h"
#include "MMcRunHeader.hxx"
#include "MMcFadcHeader.hxx"
#include "MGeomCam.h"
#include "MGeomPix.h"

ClassImp(MMcPedestalNSBAdd);

// --------------------------------------------------------------------------
//
// Default constructor.
//
MMcPedestalNSBAdd::MMcPedestalNSBAdd(const Float_t difnsb,
			       const char *name, const char *title)
{
    fName  = name  ? name  : "MMcPedestalNSBAdd";
    fTitle = title ? title : "Task to copy monte carlo pedestals into MPedestal Container";

    //
    // This is not needed here using MReadMarsFile because for the
    // RunHeader tree the auto scheme is disabled by default
    //
    AddToBranchList("fPedesMean");
    AddToBranchList("fElecNoise");

    fDnsbPixel=difnsb;
}

// --------------------------------------------------------------------------
//
// The PreProcess does nothing since the process does not exist and
// therefore it does not need any pointer or files already initialysed

Bool_t MMcPedestalNSBAdd::PreProcess(MParList *pList)
{

   // FIX ME , ReInit should be called automatically. When it is 
   // implemented then this line should be removed.

    ReInit(pList);    

    return kTRUE;
}

// --------------------------------------------------------------------------
//
//  This function is called each time MReadTree::Notify is called, which 
//  happens when it  changes the file to read from.
//  Here we add the contribution from NSB to the pedestals.
//  The ReInit searches for the following input containers:
//   - MRawRunHeader
//   - MMcRunHeader
//   - MMcFacdHeader
//   - MGeomCam
//
//   The following output containers are also searched and created if
//   they were not found:
//   - MPedestalCam
//
Bool_t MMcPedestalNSBAdd::ReInit(MParList *pList)
{

    MRawRunHeader *runheader = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (runheader)
    {
        if (runheader->GetRunType() != kRTMonteCarlo)
        {
            *fLog << warn << dbginf << "Warning - MMcPedestalNSB is for Monte Carlo files only... removing this task from list." << endl;
            return kSKIP;
        }
    }	
    else
        *fLog << warn << dbginf << "Warning - cannot check file type, MRawRunHeader not found." << endl;


    MMcFadcHeader *fadc = (MMcFadcHeader*)pList->FindObject("MMcFadcHeader");
    if (!fadc)
    {
        *fLog << err << dbginf << "MMcFadcHeader not found... aborting." << endl;
	return kFALSE;
    }


    MMcRunHeader *mcrunheader = (MMcRunHeader*)pList->FindObject("MMcRunHeader");
    if (!mcrunheader && fDnsbPixel < 0 )
    {
        *fLog << err << dbginf << "Using the default argument of MMcPedestalNSB::MMcPedestalNSB() ";
        *fLog << "only allowed if MMcRunHeader is available... aborting." << endl;
        return kFALSE;
    }

    if (mcrunheader)
    {
        if (fDnsbPixel >= 0 && fDnsbPixel != mcrunheader->GetNumPheFromDNSB())
        {
            *fLog << warn << dbginf << "The MC file has been generated with diffuse nsb " << mcrunheader->GetNumPheFromDNSB();
            *fLog <<" but you set up the diffuse NSB to " << fDnsbPixel << endl;
        }
	else
	    fDnsbPixel = mcrunheader->GetNumPheFromDNSB();
    }

    fDnsbPixel *= 50.0/15.0;


    MGeomCam *geometry = (MGeomCam*)pList->FindObject("MGeomCam");
    if (!geometry)
    {
        *fLog << err << dbginf << "MGeomCam not found... aborting." << endl;
	return kFALSE;
    }


    MPedestalCam *pedestals = (MPedestalCam*)pList->FindCreateObj("MPedestalCam");
    if (!pedestals)
	return kFALSE;


    const int num = fadc->GetNumPixel();
    
    pedestals->InitSize(num);

    MGeomPix &pixgeom0 = (*geometry)[0];
    const Float_t size0   = pixgeom0.GetR()*pixgeom0.GetR();

    for (int i=0; i<num; i++)
    {
        MPedestalPix &pix     = (*pedestals)[i];
	MGeomPix     &pixgeom = (*geometry)[i];

        const Float_t pedrms = pix.GetSigma();
	const Float_t size   = pixgeom.GetR()*pixgeom.GetR()/size0;

	pix.SetSigma(sqrt(pedrms*pedrms+fDnsbPixel*fadc->GetAmplitud()*fadc->GetAmplitud()*size));
    }

    return kTRUE;
}

