/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Oscar Blanch  11/2001 < mailto:blanch@ifae.es>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//                                                                         //
//  MMcPedestalNSB                                                        //
//                                                                         //
//  Input Containers:                                                      //
//   MMcFadcHeader                                                         //
//   MMcRunHeader                                                          //
//   MRawRunHeader                                                         //
//                                                                         //
//  Output Containers:                                                     //
//   MPedestalCam                                                          //
//                                                                         //
/////////////////////////////////////////////////////////////////////////////

#include "MMcPedestalNSB.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MPedestalCam.h"
#include "MRawRunHeader.h"
#include "MMcRunHeader.hxx"
#include "MMcFadcHeader.hxx"

ClassImp(MMcPedestalNSB);

MMcPedestalNSB::MMcPedestalNSB(const float difnsb,
			       const char *name, const char *title)
{
    fName  = name  ? name  : "MMcPedestalNSB";
    fTitle = title ? title : "Task to copy monte carlo pedestals into MPedestal Container";

    AddToBranchList("fPedestalMean");
    AddToBranchList("fElecNoise");

    fdnsb_pixel = difnsb;
}

Bool_t MMcPedestalNSB::ReInit(MParList *pList)
{
    MRawRunHeader *run = (MRawRunHeader*)pList->FindObject("MRawRunHeader");
    if (!run)
    {
        *fLog << err << dbginf << "MRawRunHeader not found... aborting." << endl;
	return kFALSE;
    }

    if (run->GetRunType() != kRTMonteCarlo)
    {
        *fLog << warn << dbginf << "Warning - MMcPedestalNSB is for Monte Carlo files only... removing this task from list." << endl;
        return kSKIP;
    }

    const MMcFadcHeader *mcped = (MMcFadcHeader*)pList->FindObject("MMcFadcHeader");
    if (!mcped)
    {
        *fLog << err << dbginf << "MMcFadcHeader not found... aborting." << endl;
        return kFALSE;
    }

    MMcRunHeader *mcrun = (MMcRunHeader*)pList->FindObject("MMcRunHeader");
    if (!mcrun && fdnsb_pixel < 0)
    {
        *fLog << err << dbginf << "Using the default argument of MMcPedestalNSB::MMcPedestalNSB() ";
        *fLog << "only allowed if MMcRunHeader is available... aborting." << endl;
        return kFALSE;
    }

    if (mcrun)
    {
        if (fdnsb_pixel >= 0 && fdnsb_pixel != mcrun->GetNumPheFromDNSB())
        {
            *fLog << warn << dbginf << "The MC file has been generated with diffuse nsb " << mcrun->GetNumPheFromDNSB();
            *fLog <<" but you set up the diffuse NSB to " << fdnsb_pixel << endl;
        }

        fdnsb_pixel = mcrun->GetNumPheFromDNSB();
    }

    fdnsb_pixel *= 50.0/15.0;

    MPedestalCam *pedcam = (MPedestalCam*)pList->FindCreateObj("MPedestalCam");
    if (!pedcam)
        return kFALSE;

    const int num = mcped->GetNumPixel();

    pedcam->InitSize(num);

    for (int i=0; i<num; i++)
    {
        MPedestalPix &pix = (*pedcam)[i];

        const Float_t pedrms = mcped->GetPedestalRms(i);
        const Float_t sigrms = pedrms/sqrt(run->GetNumSamplesHiGain()*2);
        const Float_t ampl   = mcped->GetAmplitud();

        //
        // check for central pixel (i<397)
        //

        const Double_t value = pedrms*pedrms + fdnsb_pixel*ampl*ampl;

        pix.SetPedestalRms(sqrt(i<397 ? value : value*4), sigrms);
    }

    return kTRUE;
}
