#ifndef GUI_H_SEEN
#define GUI_H_SEEN

#include <QtGui>
#include <QtConcurrentRun>

#include <qwt_plot.h>
#include <qwt_plot_curve.h>
#include <qwt_plot_grid.h>
#include <qwt_plot_zoomer.h>
#include <qwt_plot_magnifier.h>
#include <qwt_plot_panner.h>
#include <qwt_scale_engine.h>
#include <qwt_analog_clock.h>
#include <qwt_scale_widget.h>
#include <qwt_plot_layout.h>
#include <qwt_legend.h>
#include <qwt_legend_item.h>
#include <qwt_symbol.h>
#include <qwt_plot_marker.h>
#include <qwt_data.h>
#include <qwt_color_map.h>

#include <limits>
#include <float.h>
#include <limits>

#include "dic.hxx"
#include "Evidence.h"

const QColor EddPlotBackgroundColor(Qt::yellow);

void OpenHistory(char *, int);
bool SetStatus(QWidget *, QString, int, QString, int = -1);

// General Edd Widget: has Update() method called by DIM interface
class EddWidget {

  public:
  	virtual void Update(const QString &, int, const QByteArray &, const QString &, const QString &, int=-1) = 0;
};

// Base class for Edd plot
// DeleteCurve() is pure virtual and needs to be implemented iin the application class
class EddBasePlot: public QwtPlot {
  Q_OBJECT

  protected:    
    QMenu *Menu;
    QAction *StripAction;

  private:
    struct PlotItem {
      QwtPlotCurve *Signal;
	  QVector<double> x;
	  QVector<double> y;
      double Smallest;
      double Largest;
	  double Mean;
	  double Sigma;
    };
    QList<struct PlotItem> Items;

    QAction *YLogAction;
    QAction *NormAction;
    QAction *StyleAction;
    QAction *StatisticsAction;

    QwtPlotPanner *Panner;
    QwtPlotGrid *Grid;
    QwtPlotZoomer *Zoomer;
	QwtPlotMagnifier *Magnifier;
    QwtPicker *Picker;
	QwtDoubleRect BBox;
	QwtPlotMarker *Stats;
	 	
  public:
    EddBasePlot(QWidget * = NULL);
    ~EddBasePlot();

	QTimer *Timer;
	bool NewData;
	QwtPlotCurve *NewCurve(QwtText);
	void ClearCurve(unsigned int);
	void AddPoint(unsigned int, double, double);
	virtual void DeleteCurve(QwtPlotCurve *) = 0;

  protected slots:
    void UpdatePlot();

  private slots:
	void ReDoStats();
    void HandleZoom(const QwtDoubleRect &);
	void MouseSelection(const QwtPolygon &);
	void contextMenuEvent(QContextMenuEvent *);
    void MenuSingleTrace();        
    void MenuSetUpdateRate();        
    void MenuZoomOut();
    void MenuSaveASCII();
    void MenuSave();
    void MenuPrint();
	void MenuPlotHelp();
};
		  
// General indicator for DIM service
class EddLineDisplay: public QLineEdit, public EddWidget {
    Q_OBJECT

    QMenu *Menu;
    QPoint dragStart;
    QWidget *LastHist;
	
	QString ServiceName;
	int Index;
	
    void mousePressEvent(QMouseEvent *); 
    void mouseReleaseEvent(QMouseEvent *); 
    void mouseMoveEvent(QMouseEvent *); 
	
  public:
    EddLineDisplay(QString, int=-1, QWidget * = NULL);
    ~EddLineDisplay();
	void Update(const QString &, int, const QByteArray &, const QString &, const QString &, int = -1);

	bool ShowAsTime;
	
  private slots:
    void contextMenuEvent(QContextMenuEvent *);    
    void MenuOpenHistory();
    void MenuCopyService();
    void MenuCopyData();
};

// Sending command to DIM server
class EddCommand: public QLineEdit {
    Q_OBJECT

	QString Name;

	QString GetFormat();

  public:
    EddCommand(QString, QWidget * = NULL);

  private slots:
	void SendCommand();	
    void contextMenuEvent(QContextMenuEvent *);
	void MenuCommandHelp();   
};

// Graph class for history display 
class EddPlot: public EddBasePlot, public EddWidget {
    Q_OBJECT

	// Time scale for axis
	class EddTimeScale: public QwtScaleDraw {

	  public:
		EddTimeScale() {}

		virtual QwtText label(double v) const {
		  // Adapt text format to time span 
		  QString Format;
		  if (scaleDiv().range() < 60*60) Format = "hh' h\n'mm:ss";
		  else if (scaleDiv().range() < 24*60*60) Format = "hh:mm";
		  else if (scaleDiv().range() < 30*24*60*60) Format = "h' h\n'd-MMM";
		  else Format = "d-MMM'\n'yyyy";

		  // Generate text
		  QwtText Text = QDateTime::fromTime_t((int) v).toString(Format);
		  QFont Font = Text.font();
		  Font.setPointSize(7);
		  Text.setFont(Font);

    	  return Text;
		}
	};

    struct ItemDetails {
      QString Name;
	  int Index;
      QwtPlotCurve *Signal;	  
    };
    QList<struct ItemDetails> List;

  private:
	QwtLegend *Legend;
	QTimer *SingleShot;

    void dragEnterEvent(QDragEnterEvent *);
    void dropEvent(QDropEvent *);
	void paintEvent(QPaintEvent *);
	 	
  public:
    EddPlot(QString = QString(), int = 0, QWidget * = NULL);
    ~EddPlot();
    void AddService(QString, int = 0);
	void DeleteCurve(QwtPlotCurve *);
	void Update(const QString &, int, const QByteArray &, const QString &, const QString &, int = -1);

  private slots:
	void LegendClicked(QwtPlotItem *);
    void MenuPasteService();
	void MenuShowLastHour();
	void MenuShowLastDay();
};


// Text history and output class
class EddText: public QTextEdit, public EddWidget {
  Q_OBJECT

  private:
	QString Name;
	bool Pure;
	
  public:
    EddText(QString, bool = false, QWidget * = NULL);
    ~EddText();
	void Update(const QString &, int, const QByteArray &, const QString &, const QString &, int = -1);

	bool Accumulate;
};


// Interface to DIM system
class EddDim: public QObject, public DimInfo {
  Q_OBJECT

  private:
	struct Item {
	  DimStampedInfo *DIMService;
	  QMap<class EddWidget *, int> Subscribers;
	  int TimeStamp;
	  QByteArray ByteArray;
	  QString Format;
	  QString Text;
	  QStringList Items;
	};
    QMap<QString, struct Item> ServiceList;
	QMap<QString, bool> IgnoreMap;
    QMutex *Mutex, IgnoreMutex;

	struct HistItem {
	  int Count;
	  int LastUpdate;
	  class EvidenceHistory *HistClass;
	};
    QMap<QString, struct HistItem> HistoryList;

	unsigned int Period;
	long long Volume;

	void infoHandler();

  private slots:
	void Update(QString, int, QByteArray, QString);
	void UpdateStatistics();

  public:
    EddDim();
    ~EddDim();

	void Subscribe(QString, class EddWidget *, int = -1);
	void Unsubscribe (QString, class EddWidget *);
	void Ignore (QString, bool);
	class EvidenceHistory *GetHistory(QString);
	void DropHistory(QString);

  signals:
    void INT(QString, int, QByteArray = QByteArray(), QString = QString());
};

// Open new window
class EddWindow: public QPushButton {
  Q_OBJECT

  QMainWindow *M;
  QGridLayout *L;

  public:
    EddWindow(QString, QString);
	QGridLayout *Layout();
	QMainWindow *Window();
};

#endif
