----------------------------------------------------------------------------------
-- Company:        ETH Zurich, Institute for Particle Physics
-- Engineer:       Patrick Vogler
-- 
-- Create Date:    24 February 2010
-- Design Name:    
-- Module Name:    FTM Lightpulser interface Basic 
-- Project Name: 
-- Target Devices: 
-- Tool versions: 
-- Description:    Interface to the lightpulsers LP1 (in the mirror dish)
--                 and LP2 (inside the shutter)
--
-- Dependencies: 
--
-- Revision: 
-- Revision 0.01 - File Created
-- Additional Comments: 
--
--
-- modifications:  May 13 2011
--
-- Version 2
--
--        
-- modified:     May 26 2011
--               by Patrick Vogler
--               "Lightpulser Basic Version"
--
-- modified:     May 27 2011
--               by Patrick Vogler, Quirin Weitzel
--               -> clean up
--
-- modified:     July 20 2011
--               by Patrick Vogler
--               reduce minimal LED light output and increase dynamic range
--
----------------------------------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

---- Uncomment the following library declaration if instantiating
---- any Xilinx primitives in this code.
library UNISIM;
use UNISIM.VComponents.all;

library ftm_definitions;
USE ftm_definitions.ftm_array_types.all;
USE ftm_definitions.ftm_constants.all;


entity Lightpulser_interface_Basic is
  port(
    
-- Clock
-------------------------------------------------------------------------------
   clk_50    : IN  STD_LOGIC;              --  50 MHz system clock
--   clk_250   : IN  STD_LOGIC;              -- 250 MHz system clock
   
-- Lightpulser
-- RJ-45 connectors J13 or J12 on the FTM board
-- LVDS calibration outputs
-- on IO-Bank 0
-------------------------------------------------------------------------------
-- connector J13   => Light Pulser 1 in the mirror dish 
   Cal_0_p    : out STD_LOGIC := '0';   -- Feedback / pulse width modulation
   Cal_0_n    : out STD_LOGIC := '1';
   Cal_1_p    : out STD_LOGIC := '0';   -- Pulse
   Cal_1_n    : out STD_LOGIC := '1';
   Cal_2_p    : out STD_LOGIC := '0';   -- Gate_1_4_7
   Cal_2_n    : out STD_LOGIC := '1';
   Cal_3_p    : out STD_LOGIC := '0';   -- Gate_3_5_8
   Cal_3_n    : out STD_LOGIC := '1';

-- connector J12   => Light Pulser 2 in the shutter
   Cal_4_p    : out STD_LOGIC := '0';   -- Feedback / pulse width modulation
   Cal_4_n    : out STD_LOGIC := '1';
   Cal_5_p    : out STD_LOGIC := '0';   -- Pulse
   Cal_5_n    : out STD_LOGIC := '1';
   Cal_6_p    : out STD_LOGIC := '0';   -- Gate_1_4_7
   Cal_6_n    : out STD_LOGIC := '1'; 
   Cal_7_p    : out STD_LOGIC := '0';   -- Gate_3_5_8
   Cal_7_n    : out STD_LOGIC := '1';  
   
      
-- FPGA intern signals: Lightpulser brightness
-------------------------------------------------------------------------------
  
   LP1_ampl            : in std_logic_vector (15 downto 0);
   LP2_ampl            : in std_logic_vector (15 downto 0);

--   LP1_delay           : in std_logic_vector (15 downto 0);
--   LP2_delay           : in std_logic_vector (15 downto 0);
   
   
   LP1_pulse           : in std_logic;  -- trigger lightpulse in the mirror dish
   LP2_pulse           : in std_logic;  -- trigger lightpulse in the shutter


   start_config        : in std_logic;  -- handshaking
   config_started      : out std_logic := '0';
   config_done         : out std_logic := '0'
   
  );
end Lightpulser_interface_Basic;


architecture Behavioral of Lightpulser_interface_Basic is


component FM_pulse_generator_Basic is
  port( 
    clk            : in  std_logic;    -- 50 MHz
    pulse_freq     : in  std_logic_vector (6 downto 0);
    FM_out         : out std_logic  := '0'
  );
end component;


component single_LP_Basic is
  port(  
    clk_50         : in  STD_LOGIC;       
    LP_Pulse_out   : out STD_LOGIC;                                                 
    LP_pulse_in    : in  std_logic  
  );
end component;


-- LP1: mirror dish
signal Cal_0_1 : STD_LOGIC := '0';  
--  signal Cal_1_1 : STD_LOGIC;

-- LP2: shutter
signal Cal_0_2 : STD_LOGIC := '0';
--  signal Cal_1_2 : STD_LOGIC;

-- PWM for amplitude stabilization
signal PWM_sig_1 : std_logic := '0';  -- LP1: mirror dish
signal PWM_sig_2 : std_logic := '0';  -- LP2: shutter

-- control data latch
signal LP1_ampl_sig   :  std_logic_vector (15 downto 0) := (others => '0');
signal LP2_ampl_sig   :  std_logic_vector (15 downto 0) := (others => '0'); 
 

type type_latch_state is (IDLE, COPY, CONFIGURED);   
signal latch_state : type_latch_state  := IDLE;   
  

begin
  

  -- input latch
  input_latch : process (clk_50)
  begin
    if rising_edge(clk_50) then
       case latch_state is

         when IDLE =>
              if start_config = '1' then 
                 config_done <= '0';
                 config_started <= '1'; 
                 latch_state <= COPY;
              end if;
                       
         when COPY =>
              LP1_ampl_sig   <= LP1_ampl;
              LP2_ampl_sig   <= LP2_ampl;
      --        LP1_delay_sig  <= LP1_delay;
      --        LP2_delay_sig  <= LP2_delay;         
              latch_state    <= CONFIGURED;

         when CONFIGURED =>
              config_started <= '0'; 
              config_done <= '1';
              latch_state <= IDLE;  
                
       end case;           
    end if;  
  end process input_latch; 
	  	 
  
  Inst_LP1_mirror_dish:single_LP_Basic
    port map (
      clk_50        => clk_50,    
      LP_Pulse_out  => Cal_0_1,                                     
      LP_pulse_in   => LP1_pulse               
      );

  
  Inst_LP2_shutter:single_LP_Basic
    port map (
      clk_50        => clk_50,    
      LP_Pulse_out  => Cal_0_2,                                     
      LP_pulse_in   => LP2_pulse               
    );

  Inst_LP1_FM_pulse_generator:FM_pulse_generator_Basic    -- LP1: mirror dish
    port map( 
      clk            => clk_50,  
      pulse_freq     => LP1_ampl_sig(6 downto 0),
      FM_out         => PWM_sig_1   
    );


  Inst_LP2_FM_pulse_generator:FM_pulse_generator_Basic    -- LP2: shutter
    port map( 
      clk            => clk_50,  
      pulse_freq     => LP2_ampl_sig(6 downto 0),
      FM_out         => PWM_sig_2   
    );


  -- Light Pulser 1 (in the mirror dish): differential output buffers 

  OBUFDS_inst_Cal_0 : OBUFDS
    generic map (
      IOSTANDARD => "DEFAULT")
    port map (  O  => Cal_0_p ,    -- Diff_p output (connect directly to top-level port)
                OB =>  Cal_0_n ,   -- Diff_n output (connect directly to top-level port)
                I  =>  Cal_0_1     -- Buffer input 
                ); 

  OBUFDS_inst_Cal_1 : OBUFDS
    generic map (
      IOSTANDARD => "DEFAULT")
    port map  (  O  => Cal_1_p ,    -- Diff_p output (connect directly to top-level port)
                 OB =>  Cal_1_n ,   -- Diff_n output (connect directly to top-level port)
                 I  =>  PWM_sig_1   -- Buffer input 
                 ); 

  OBUFDS_inst_Cal_2 : OBUFDS
    generic map (
      IOSTANDARD => "DEFAULT")
    port map  (  O  => Cal_2_p ,     -- Diff_p output (connect directly to top-level port)
                 OB =>  Cal_2_n ,    -- Diff_n output (connect directly to top-level port)
                 I  =>  LP1_ampl_sig(14)       -- Buffer input 
                 ); 

  OBUFDS_inst_Cal_3 : OBUFDS
    generic map (
      IOSTANDARD => "DEFAULT")
    port map (   O  => Cal_3_p ,     -- Diff_p output (connect directly to top-level port)
                 OB =>  Cal_3_n ,    -- Diff_n output (connect directly to top-level port)
                 I  =>  LP1_ampl_sig(15)      -- Buffer input 
                 );     


     
  --  Light Pulser 2 (in the shutter): differential output buffers 
     
  OBUFDS_inst_Cal_4 : OBUFDS
    generic map (
      IOSTANDARD => "DEFAULT")
    port map (   O  => Cal_4_p ,    -- Diff_p output (connect directly to top-level port)
                 OB =>  Cal_4_n ,   -- Diff_n output (connect directly to top-level port)
                 I  =>  Cal_0_2     -- Buffer input 
                 ); 

  OBUFDS_inst_Cal_5 : OBUFDS
    generic map (
      IOSTANDARD => "DEFAULT")
    port map  (  O  => Cal_5_p ,    -- Diff_p output (connect directly to top-level port)
                 OB =>  Cal_5_n ,   -- Diff_n output (connect directly to top-level port)
                 I  =>  PWM_sig_2   -- Buffer input 
                 ); 

  OBUFDS_inst_Cal_6 : OBUFDS
    generic map (
      IOSTANDARD => "DEFAULT")
    port map  (  O  => Cal_6_p ,     -- Diff_p output (connect directly to top-level port)
                 OB =>  Cal_6_n ,    -- Diff_n output (connect directly to top-level port)
                 I  =>  LP2_ampl_sig(14)
                 );             

  OBUFDS_inst_Cal_7 : OBUFDS
    generic map (
      IOSTANDARD => "DEFAULT")
    port map  (  O  => Cal_7_p ,     -- Diff_p output (connect directly to top-level port)
                 OB =>  Cal_7_n ,    -- Diff_n output (connect directly to top-level port)
                 I  =>  LP2_ampl_sig(15)   -- Buffer input 
                 );     
    

end Behavioral;
