#include "Dim.h"
#include "Configuration.h"
#include "RemoteControl.h"

using namespace std;

template <class T>
void RunShell(Configuration &conf)
{
    // A normal kill will call its destructor! (Very nice feature ;) )
    static T shell(conf.GetName().c_str(), conf.Get<int>("console")!=1);

    WindowLog &win  = shell.GetStreamIn();
    WindowLog &wout = shell.GetStreamOut();

    if (conf.Has("log"))
        if (!wout.OpenLogFile(conf.Get<string>("log")))
            win << kRed << "ERROR - Couldn't open log-file " << conf.Get<string>("log") << ": " << strerror(errno) << endl;

    const vector<string> v1 = conf.Vec<string>("cmd");
    for (vector<string>::const_iterator it=v1.begin(); it!=v1.end(); it++)
        shell.ProcessLine(*it);

    const vector<string> v2 = conf.Vec<string>("exec");
    for (vector<string>::const_iterator it=v2.begin(); it!=v2.end(); it++)
        shell.Execute(*it);

    if (conf.Get<bool>("quit"))
        shell.Stop();

    shell.Run();
}


// ========================================================================
void SetupConfiguration(Configuration &conf)
{
    const string n = conf.GetName()+".log";

    po::options_description config("Program options");
    config.add_options()
        ("dns",       var<string>("localhost"),       "Dim nameserver (overwites DIM_DNS_NODE environment variable)")
        ("host",      var<string>(""),                "Address with which the Dim nameserver can connect to this host (overwites DIM_HOST_NODE environment variable)")
        ("log,l",     var<string>(n), "Write log-file")
        ("console,c", var<int>(0),    "Use console (0=shell, 1=simple buffered, X=simple unbuffered)")
        ("cmd",       vars<string>(), "Execute one or more commands at startup")
        ("exec,e",    vars<string>(), "Execute one or more scrips at startup")
        ("quit",      po_switch(),    "Quit after startup");
        ;

    conf.AddEnv("dns",  "DIM_DNS_NODE");
    conf.AddEnv("host", "DIM_HOST_NODE");

    conf.AddOptions(config);
}

/*
 Extract usage clause(s) [if any] for SYNOPSIS.
 Translators: "Usage" and "or" here are patterns (regular expressions) which
 are used to match the usage synopsis in program output.  An example from cp
 (GNU coreutils) which contains both strings:
  Usage: cp [OPTION]... [-T] SOURCE DEST
    or:  cp [OPTION]... SOURCE... DIRECTORY
    or:  cp [OPTION]... -t DIRECTORY SOURCE...
 */
void PrintUsage()
{
    cout << "\n"
        "The console connects to all available Dim Servers and allows to "
        "easily access all of their commands.\n"
        "\n"
        "Usage: dimctrl [-c type] [OPTIONS]\n"
        "  or:  dimctrl [OPTIONS]\n";
    cout << endl;

}

void PrintHelp()
{
    /* Additional help text which is printed after the configuration
     options goes here */
}

int main(int argc, const char *argv[])
{
    Configuration conf(argv[0]);
    conf.SetPrintUsage(PrintUsage);
    SetupConfiguration(conf);

    po::variables_map vm;
    try
    {
        vm = conf.Parse(argc, argv);
    }
#if BOOST_VERSION > 104000
    catch (po::multiple_occurrences &e)
    {
        cerr << "Program options invalid due to: " << e.what() << " of '" << e.get_option_name() << "'." << endl;
        return -1;
    }
#endif
    catch (exception& e)
    {
        cerr << "Program options invalid due to: " << e.what() << endl;
        return -1;
    }

    if (conf.HasVersion() || conf.HasPrint())
        return -1;

    if (conf.HasHelp())
    {
        PrintHelp();
        return -1;
    }

    Dim::Setup(conf.Get<string>("dns"), conf.Get<string>("host"));

    if (conf.Get<int>("console")==0)
        //Main<RemoteShell, DummyService>(conf);
        RunShell<RemoteShell>(conf);
    else
        //Main<RemoteConsole, DummyService>(conf);
        RunShell<RemoteConsole>(conf);


    return 0;
}
