----------------------------------------------------------------------------------
-- Company:        ETH Zurich, Institute for Particle Physics
-- Engineer:       Patrick Vogler
-- 
-- Create Date:    14 February 2010
-- Design Name:    
-- Module Name:    FTM Clock conditioner Interface
-- Project Name: 
-- Target Devices: 
-- Tool versions: 
-- Description:    Interface to the LMK03000 Clock conditioner
--
-- Dependencies: 
--
-- Revision: 
-- Revision 0.01 - File Created
-- Additional Comments: 
--
--
-- modifications:  February 21  2011 by Patrick Vogler
--                 March    23  2011 by Patrick Vogler
--                 May      03  2011 by Patrick Vogler and Quirin Weitzel
--                 July     19  2011 by Patrick Vogler
--                 July     27  2011 by Patrick Vogler
----------------------------------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

---- Uncomment the following library declaration if instantiating
---- any Xilinx primitives in this code.
--library UNISIM;
--use UNISIM.VComponents.all;

library ftm_definitions;
USE ftm_definitions.ftm_array_types.all;
USE ftm_definitions.ftm_constants.all;


entity Clock_cond_interface is
  port(
    
    -- Clock
    -------------------------------------------------------------------------------
    clk   : IN  STD_LOGIC;               -- 50 MHz system clock

    -- Clock conditioner LMK03000
    -------------------------------------------------------------------------------
    CLK_Clk_Cond  : out STD_LOGIC;  -- clock conditioner MICROWIRE interface clock
    LE_Clk_Cond   : out STD_LOGIC;  -- clock conditioner MICROWIRE interface latch enable   
    DATA_Clk_Cond : out STD_LOGIC;  -- clock conditioner MICROWIRE interface data
   
    SYNC_Clk_Cond : out STD_LOGIC;  -- clock conditioner global clock synchronization
    LD_Clk_Cond   : in STD_LOGIC;   -- clock conditioner lock detect                  

    -- Time Marker
    -------------------------------------------------------------------------------
    TIM_Sel    : out STD_LOGIC;  -- Time Marker selector 
                                 -- 1 = time marker from Clock conditioner
                                 --     for DRS timing calibration
                                 --
                                 -- 0 = time marker from FPGA for normal
                                 --     operation / physics run

    -- FPGA intern clock conditioner configuration data
    -------------------------------------------------------------------------------
    cc_R0             : in std_logic_vector (31 downto 0) := (others => '0');
    cc_R1             : in std_logic_vector (31 downto 0) := (others => '0');
    cc_R8             : in std_logic_vector (31 downto 0) := (others => '0');
    cc_R9             : in std_logic_vector (31 downto 0) := (others => '0');
    cc_R11            : in std_logic_vector (31 downto 0) := (others => '0');
    cc_R13            : in std_logic_vector (31 downto 0) := (others => '0');
    cc_R14            : in std_logic_vector (31 downto 0) := (others => '0');
    cc_R15            : in std_logic_vector (31 downto 0) := (others => '0');

    -- FPGA intern control signals
    -------------------------------------------------------------------------------
    start_config     : in STD_LOGIC;   -- load new configuration into the clock
                                       -- conditioner
   
    config_started   : out STD_LOGIC;  -- indicates that the new configuration
                                       -- is currently loaded into the clock conditioner

    config_done      : out STD_LOGIC;  -- indicates that the configuration has
                                       -- been loaded 

    locked           : out STD_LOGIC;  -- PLL in the Clock Conditioner locked
    
    timemarker_select: in STD_LOGIC    -- selects time marker source
                                       --
                                       -- 1 = time marker from Clock conditioner
                                       --     for DRS timing calibration
                                       --
                                       -- 0 = time marker from FPGA for normal
                                       --     operation / physics run
    
  );
end Clock_cond_interface;



architecture Behavioral of Clock_cond_interface is
   
  component microwire_interface IS
    PORT(
      clk               : IN     std_logic;
      clk_uwire         : OUT    std_logic;
      data_uwire        : OUT    std_logic;       
      le_uwire          : OUT    std_logic;
      clk_cond_array    : IN     clk_cond_array_type;    
      config_start      : IN     std_logic;
      config_ready      : OUT    std_logic; 
      config_started    : OUT    std_logic      
    );
  end component;

  signal clk_50M_sig   : STD_LOGIC;  -- system clock (50MHz)
  signal clk_uwire_sig : STD_LOGIC;  -- 2 MHz
  
  signal config_ready_sig : STD_LOGIC;
  signal config_started_sig : STD_LOGIC;

  signal clk_cond_array_sig    : clk_cond_array_type;

  signal timemarker_select_sig : std_logic := '0';

  type TIM_SEL_STATE_TYPE is (IDLE, CONFIG);
  signal tim_sel_state : TIM_SEL_STATE_TYPE := IDLE;

  signal load_detect_sr : std_logic_vector (1 downto 0) := "00";

begin
  
  Inst_microwire_interface:microwire_interface
    port map (
        clk                 => clk_50M_sig,         
        clk_uwire           => clk_uwire_sig,  
        data_uwire          => DATA_Clk_Cond,
        le_uwire            => LE_Clk_Cond,        
        clk_cond_array      => clk_cond_array_sig,  
        config_start        => start_config,
        config_ready        => config_ready_sig,
        config_started      => config_started_sig
    );

  sync_ld_proc : process (clk_uwire_sig)
  begin
    if rising_edge(clk_uwire_sig) then
      load_detect_sr <= load_detect_sr(0) & LD_Clk_Cond;
    end if;
  end process sync_ld_proc;
  
-- config_done <= config_ready_sig and (load_detect_sr(1) and load_detect_sr(0));

  config_done <= config_ready_sig;
  locked      <= load_detect_sr(1) and load_detect_sr(0);
  
  
  
  
  TIM_Sel <= timemarker_select_sig; 
  
  tim_sel_proc : process (clk_uwire_sig)
  begin
    if rising_edge(clk_uwire_sig) then
      case tim_sel_state is
        when IDLE =>
          if start_config = '1' then
            timemarker_select_sig <= '0';
            tim_sel_state <= CONFIG;
          end if;
        when CONFIG =>
          if config_ready_sig = '1' then
            timemarker_select_sig <= timemarker_select;
            tim_sel_state <= IDLE;
          end if;
       end case;
     end if;
  end process tim_sel_proc;
     
  CLK_Clk_Cond <= clk_uwire_sig;

  clk_50M_sig <= clk;
  
  config_started <= config_started_sig;
     
  clk_cond_array_sig(0) <= LMK03000_Reset;      -- reset LKM03000 by setting
                                                -- bit 31 of register 0
  clk_cond_array_sig(1) <= cc_R0;
  clk_cond_array_sig(2) <= cc_R1;
  
  clk_cond_array_sig(3) <= cc_R2_const; -- unused channels
  clk_cond_array_sig(4) <= cc_R3_const; 
  clk_cond_array_sig(5) <= cc_R4_const; 
  clk_cond_array_sig(6) <= cc_R5_const; 
  clk_cond_array_sig(7) <= cc_R6_const; 
  clk_cond_array_sig(8) <= cc_R7_const; -- unused channels
  
  clk_cond_array_sig(9)  <= cc_R8; 
  clk_cond_array_sig(10) <= cc_R9; 
  clk_cond_array_sig(11) <= cc_R11;
  clk_cond_array_sig(12) <= cc_R13;
  clk_cond_array_sig(13) <= cc_R14;
  clk_cond_array_sig(14) <= cc_R15;


end Behavioral;
