library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;
library fact_fad_lib;
use fact_fad_lib.fad_definitions.all;

-- -- Uncomment the following library declaration if instantiating
-- -- any Xilinx primitives in this code.
-- library UNISIM;
-- use UNISIM.VComponents.all;

entity data_generator is
generic(
	RAM_ADDR_WIDTH : integer := 12
);
port( 
	-- for debugging
	state 					: out 	std_logic_vector(7 downto 0);
	is_idle					: out  std_logic;

	clk						: in	std_logic;					-- CLK_25.
	data_out				: out	std_logic_vector (63 downto 0);
	addr_out				: out	std_logic_vector (RAM_ADDR_WIDTH-1 downto 0);
	dataRAM_write_ea_o		: out	std_logic_vector (0 downto 0) := "0";
	ram_start_addr			: in	std_logic_vector (RAM_ADDR_WIDTH-1 downto 0);
	ram_write_ea			: in	std_logic;
	ram_write_ready			: out	std_logic := '0';
	
	roi_array				: in	roi_array_type;
	roi_max					: in	roi_max_type;
	sensor_array			: in	sensor_array_type;
	sensor_ready			: in	std_logic;
	dac_array				: in	dac_array_type;
	
	config_start			: in	std_logic;
	config_done				: out	std_logic := '0';
	
-- EVT HEADER - part 1
	package_length			: in	std_logic_vector (15 downto 0);
	pll_lock				: in	std_logic_vector ( 3 downto 0);
	dwrite_enable_in		: in	std_logic;
	denable_enable_in		: in	std_logic;
	busy_enable_in			: in	std_logic;
	trigger_enable_in		: in	std_logic;
	cont_trigger_en_in		: in	std_logic;
	socket_send_mode_in		: in	std_logic;
	busy_manual_in			: in	std_logic;
	

-- EVT HEADER - part 2  --> FTM trigger informaton, comes in late ...
-- during EVT header wrinting, this field is left out ... and only written into event header,
-- when the DRS chip were read out already.
	FTM_RS485_ready			: in	std_logic;
	FTM_trigger_info		: in	std_logic_vector (55 downto 0); --7 byte
	FTM_receiver_status : in std_logic;

-- EVT HEADER - part 3
	fad_event_counter		: in	std_logic_vector (31 downto 0);
	refclk_counter			: in	std_logic_vector (11 downto 0);
	refclk_too_high			: in	std_logic;
	refclk_too_low			: in	std_logic;

-- EVT HEADER - part 4
	board_id				: in	std_logic_vector (3 downto 0);
	crate_id				: in	std_logic_vector (1 downto 0);
	DCM_PS_status			: in	std_logic_vector (7 downto 0);
	DCM_locked_status		: in	std_logic;
	DCM_ready_status		: in	std_logic;
	SPI_SCLK_enable_status	: in	std_logic;
	TRG_GEN_div				: in	std_logic_vector (15 downto 0);

-- EVT HEADER - part 5
	dna						: in	std_logic_vector (63 downto 0);

-- EVT HEADER - part 6
	runnumber				: in	std_logic_vector (31 downto 0); 
	timer_value				: in	std_logic_vector (31 downto 0); -- time in units of 100us

	hardware_trigger_in		: in	std_logic;
	software_trigger_in		: in	std_logic;
	
	adc_data_array			: in	adc_data_array_type;
	adc_output_enable_inverted					: out	std_logic := '1';
	adc_clk_en				: out	std_logic := '0';
	adc_otr					: in	std_logic_vector (3 downto 0);
	drs_channel_id			: out	std_logic_vector (3 downto 0) := (others => '0');

--drs_dwrite : out std_logic := '1';
	drs_readout_ready		: out	std_logic := '0';
	drs_readout_ready_ack	: in	std_logic;
	drs_clk_en				: out	std_logic := '0';
	start_read_drs_stop_cell: out	std_logic := '0';

	drs_srin_write_8b		: out	std_logic := '0';
	drs_srin_write_ack		: in	std_logic;
	drs_srin_data			: out	std_logic_vector (7 downto 0) := (others => '0');
	drs_srin_write_ready	: in	std_logic;

	drs_read_s_cell_ready	: in	std_logic;
	drs_s_cell_array		: in	drs_s_cell_array_type;

	drs_readout_started		: out	std_logic := '0';
	trigger_veto : out std_logic := '1'
);
end data_generator ;

architecture Behavioral of data_generator is

type state_generate_type is (
	CONFIG,		-- IDLE branches into this state, if needed.
	CONFIG_DRS_01,			-- these four states configure the DRS shift registers,
	CONFIG_DRS_02,			-- the make great use of the drs_pulser entity.
	CONFIG_DRS_03,			-- maybe they should be moved into the drs_pulser entity.
	WAIT_FOR_DRS_CONFIG_READY,

	IDLE,
	WRITE_HEADER, WRITE_FTM_INFO, WRITE_EVENTCOUNTER_AND_REFCLK_COUNTER, WRITE_BOARD_ID,
	WRITE_DNA, WRITE_TIMER, WRITE_TEMPERATURES, 
	WRITE_DAC1, WRITE_DAC2,
	WAIT_FOR_STOP_CELL,
	START_DRS_READING,
	WRITE_CHANNEL_ID, WRITE_START_CELL, WRITE_ROI, WRITE_FILLING,
	WAIT_FOR_ADC, WRITE_ADC_DATA,
	WAIT_FOR_EXTERNAL_TRIGGER_READY, WRITE_EXTERNAL_TRIGGER,
	WRITE_END_FLAG,
	WRITE_DATA_END, WRITE_DATA_END_WAIT,
	WRITE_DATA_STOP, WRITE_DATA_STOP1
);

-- configuration stuff:
		-- this flag is set, when ever a rising edge on 'config_start' is detected.
		-- this flag is cleared only, when a configuration was successfully processed

signal state_sig : std_logic_vector(7 downto 0) := (others => '0');

signal state_generate : state_generate_type := CONFIG;
signal start_addr : std_logic_vector (RAM_ADDR_WIDTH-1 downto 0) := (others => '0');

signal data_cntr : integer  range 0 to 1024 := 0;
signal addr_cntr : integer range 0 to RAM_SIZE_64B := 0;    -- counts 64 bit words
signal channel_id : integer range 0 to 9 := 0;
signal adc_wait_cnt : integer range 0 to 7 := 0;

signal hardware_trigger_sr :std_logic_vector(1 downto 0) := "00";
signal software_trigger_sr :std_logic_vector(1 downto 0) := "00";

signal ram_write_ea_flag : std_logic := '0';
signal new_config_int : std_logic := '0';

-- internal signal: to be sampled once and used instead of inputs!
signal roi_max_int : roi_max_type;
signal package_length_sig : std_logic_vector (15 downto 0);
signal sig_drs_readout_started : std_logic := '0';

signal FTM_trigger_info_local_copy : std_logic_vector (55 downto 0) := (others => '0'); --7 byte
signal runnumber_local_copy : std_logic_vector (31 downto 0);

-- self configuration signals:
signal internal_roi_array : roi_array_type;
signal internal_roi_max : roi_max_type;

signal config_start_sr : std_logic_vector(1 downto 0) := "00";

signal adc_data_sig : adc_data_array_type;
signal adc_otr_sig : std_logic_vector(3 downto 0) ;
type adc_data_16bit_t is array (3 downto 0) of std_logic_vector(15 downto 0);
signal data_16bit : adc_data_16bit_t;

begin
drs_readout_started <= sig_drs_readout_started;
state <= state_sig;
generate_data : process (clk)
begin
	if rising_edge (clk) then
	is_idle <= '0';
	-- synch
	config_start_sr <= config_start_sr(0) & config_start;

		
		hardware_trigger_sr <= hardware_trigger_sr(0) & hardware_trigger_in; --synching in of asynchrounous trigger signal.
		software_trigger_sr <= software_trigger_sr(0) & software_trigger_in; --synching in of asynchrounous trigger signal.
		addr_out <= start_addr + conv_std_logic_vector(addr_cntr, RAM_ADDR_WIDTH);
		
		case state_generate is
		
		
		when CONFIG =>
			state_sig <=  X"01";
			internal_roi_array <= roi_array;
			package_length_sig <= package_length;
			internal_roi_max <= roi_max;
			state_generate <= CONFIG_DRS_01;
		-- configure DRS
		-- all this might be done in the drs_pulser entity
		when CONFIG_DRS_01 =>							-- BEGIN CONFIG DRS
			state_sig <=  X"02";
			drs_channel_id <= DRS_WRITE_SHIFT_REG;
			drs_srin_data <= "11111111";
			drs_srin_write_8b <= '1';
			if (drs_srin_write_ack = '1') then
				drs_srin_write_8b <= '0';
				state_generate <= CONFIG_DRS_02;
			end if;
		when CONFIG_DRS_02 =>
			state_sig <=  X"03";
			if (drs_srin_write_ready = '1') then
				state_generate <= CONFIG_DRS_03;
			end if;
		when CONFIG_DRS_03 =>
			state_sig <=  X"04";
			drs_channel_id <= DRS_WRITE_CONFIG_REG;
			drs_srin_data <= "11111111";
			drs_srin_write_8b <= '1';
			if (drs_srin_write_ack = '1') then
				drs_srin_write_8b <= '0';
				state_generate <= WAIT_FOR_DRS_CONFIG_READY;
			end if;
			
		-- last state of CONFIG:
			-- here the input roi_max is sampled
			-- all other interesting input signals should be sampled here as well!
		when WAIT_FOR_DRS_CONFIG_READY =>							-- END OF CONFIG
			state_sig <=  X"05";
			if (drs_srin_write_ready = '1') then
				drs_channel_id <= DRS_ADDR_IDLE; -- to make sure not to write accidentally into DRS shift registers
				roi_max_int <= internal_roi_max;
				config_done <= '1';
				state_generate <= IDLE;
			end if;
			-- end configure DRS
		

		when IDLE =>
			is_idle <= '1';
			state_sig <=  X"10";
			state_generate <= IDLE;
			trigger_veto <= '0';
			if (config_start_sr(1) = '1') then
				state_generate <= CONFIG;
				config_done <= '0';
			end if;
			
			
			if (ram_write_ea = '1' and ( hardware_trigger_sr(1) = '1' or software_trigger_sr(1) = '1') ) then
				sig_drs_readout_started <= '1'; -- is set to '0' in next state ... just a pulse.
				runnumber_local_copy <= runnumber;
				trigger_veto <= '1';
				start_read_drs_stop_cell <= '1';
				adc_output_enable_inverted <= '0';
				-- at this moment the ADC ist beeing clocked. 
				-- this is not the start of the readout.
				-- the DRS needs to be clocked as well.
				adc_clk_en <= '1';
				start_addr <= ram_start_addr;
				state_generate <= WRITE_HEADER;
			end if;

			
		when WRITE_HEADER =>
			state_sig <=  X"11";
			sig_drs_readout_started <= '0'; -- is set to '1' in state IDLE
			dataRAM_write_ea_o <= "1";
			data_out <= 
				-- the first word contains a lot of single status bits.
					pll_lock & 				-- 4 bits
					denable_enable_in & 	-- 1 bit
					dwrite_enable_in &  	-- 1 bit
					'0' &  		-- 1 bit
					refclk_too_low &  		-- 1 bit
					DCM_locked_status & 	-- 1 bit
					DCM_ready_status & 		-- 1 bit
					SPI_SCLK_enable_status &-- 1 bit
					busy_enable_in &
					trigger_enable_in &
					cont_trigger_en_in &
					socket_send_mode_in&
					busy_manual_in &
				PACKAGE_VERSION & PACKAGE_SUB_VERSION & 
				package_length_sig & 
				X"FB01";
			addr_cntr <= addr_cntr + 1;  
			state_generate <= WRITE_FTM_INFO;
					
		when WRITE_FTM_INFO =>
			state_sig <=  X"12";
			-- THIS is just a dummy STATE just to make reading easier.
			-- at this point normally the FTM RS485 data would be written .. but we do not know it
			-- so here we do not write the FTM info ... just jump over it.
			addr_cntr <= addr_cntr + 1;  
			state_generate <= WRITE_EVENTCOUNTER_AND_REFCLK_COUNTER;
					
		when WRITE_EVENTCOUNTER_AND_REFCLK_COUNTER =>
			state_sig <=  X"13";
			data_out <= 
					"0000" & refclk_counter & 
					X"0000" &
					fad_event_counter(15 downto 0) & 
					fad_event_counter(31 downto 16) ;
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_BOARD_ID;
			
		when WRITE_BOARD_ID =>     
			state_sig <=  X"14";
			data_out <= TRG_GEN_div & 							-- this is a kind of prescaler for the continouus trigger generator
						X"0000" & 								-- this might be the number of soft triggers beeing generated in a 'burst' not implemented yet
						X"00" & DCM_PS_status &					-- number of steps, the phase shifter was shifted...
						"000000" & crate_id & "0000" & board_id;			-- position of the board inside the camera
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_DNA;

		when WRITE_DNA =>
			state_sig <=  X"15";
			data_out <= 				
				dna(55 downto 48) & dna(63 downto 56) &
				dna(39 downto 32) & dna(47 downto 40) &
				dna(23 downto 16) & dna(31 downto 24) &
				dna(7 downto 0) & dna(15 downto 8);
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_TIMER;

		when WRITE_TIMER =>
			state_sig <=  X"16";
			data_out <= 
				runnumber_local_copy(15 downto 0) & 		-- 2times 16bit reserved for additional status info
				runnumber_local_copy(31 downto 16) & 
				timer_value(15 downto 0) &
				timer_value(31 downto 16);
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_TEMPERATURES;

		-- DANGER: thist state can wait endlessly, if somethings wrong.
		when WRITE_TEMPERATURES =>     -- temperatures
				state_sig <=  X"17";
			if (sensor_ready = '1') then
				data_out <=	
					conv_std_logic_vector(sensor_array (3), 16)(15) & 
					conv_std_logic_vector(sensor_array (3), 16)(15) & 
					conv_std_logic_vector(sensor_array (3), 16)(15) & 
					conv_std_logic_vector(sensor_array (3), 16)(15 downto 3) &
					conv_std_logic_vector(sensor_array (2), 16)(15) & 
					conv_std_logic_vector(sensor_array (2), 16)(15) & 
					conv_std_logic_vector(sensor_array (2), 16)(15) & 
					conv_std_logic_vector(sensor_array (2), 16)(15 downto 3) &
					conv_std_logic_vector(sensor_array (1), 16)(15) & 
					conv_std_logic_vector(sensor_array (1), 16)(15) & 
					conv_std_logic_vector(sensor_array (1), 16)(15) & 
					conv_std_logic_vector(sensor_array (1), 16)(15 downto 3) &
					conv_std_logic_vector(sensor_array (0), 16)(15) & 
					conv_std_logic_vector(sensor_array (0), 16)(15) & 
					conv_std_logic_vector(sensor_array (0), 16)(15) & 
					conv_std_logic_vector(sensor_array (0), 16)(15 downto 3) ;
--							conv_std_logic_vector (sensor_array (2), 16) &
--							conv_std_logic_vector (sensor_array (1), 16) &
--							conv_std_logic_vector (sensor_array (0), 16);
				addr_cntr <= addr_cntr + 1;
				state_generate <= WRITE_DAC1;
			end if;

		when WRITE_DAC1 =>
			state_sig <=  X"18";
			data_out <= conv_std_logic_vector (dac_array (3), 16) &
						conv_std_logic_vector (dac_array (2), 16) &
						conv_std_logic_vector (dac_array (1), 16) &
						conv_std_logic_vector (dac_array (0), 16);
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_DAC2;
		when WRITE_DAC2 =>
			state_sig <=  X"19";
			data_out <= conv_std_logic_vector (dac_array (7), 16) &
						conv_std_logic_vector (dac_array (6), 16) &
						conv_std_logic_vector (dac_array (5), 16) &
						conv_std_logic_vector (dac_array (4), 16);
			addr_cntr <= addr_cntr + 1;
			state_generate <= WAIT_FOR_STOP_CELL;

		when WAIT_FOR_STOP_CELL =>
			state_sig <=  X"1A";
			start_read_drs_stop_cell <= '0';
			if (drs_read_s_cell_ready = '1') then
				state_generate <= START_DRS_READING;
			end if;

		when START_DRS_READING =>
			--drs channel number
			drs_channel_id <= conv_std_logic_vector (channel_id, 4);
			
			--adc_output_enable_inverted <= '0'; -- nur fr Emulator ??????????????????
			-- this has been done earlier already ... why does it need to be repeated?

			--starte drs-clocking
			-- this is an interesting point:
				-- here the DRS clock starts to tick. but only some states later
				-- the ADC data is actually read out. 
				-- the reason is, that the ADC has a latency of 7 clock cycles, which means, 
				-- when the next rising edge of the DRS clock is produced. 
				-- an analog value is put out.
				-- when the next rising edge of the ADC clock is produced.
				-- this very analog value is sampled.
				-- but only seven clock ticks later, the degital result is available.
				-- from that point on, every clock tick produces a valid digital result.
			drs_clk_en <= '1';
			adc_wait_cnt <= 0;
			state_generate <= WRITE_CHANNEL_ID;

		when WRITE_CHANNEL_ID =>    -- write DRS and Channel IDs
			state_sig <=  X"1B";
			data_out <=		conv_std_logic_vector(3,12) & conv_std_logic_vector(channel_id,4) & 
							conv_std_logic_vector(2,12) & conv_std_logic_vector(channel_id,4) &
							conv_std_logic_vector(1,12) & conv_std_logic_vector(channel_id,4) &
							conv_std_logic_vector(0,12) & conv_std_logic_vector(channel_id,4);
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_START_CELL;
		when WRITE_START_CELL =>    -- write start cells
			state_sig <=  X"1C";
			data_out <= 	"000000" & drs_s_cell_array (3) &
							"000000" & drs_s_cell_array (2) &
							"000000" & drs_s_cell_array (1) &
							"000000" & drs_s_cell_array (0); 
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_ROI;
		
		when WRITE_ROI =>    -- write ROI
			state_sig <=  X"1D";
			data_out <=	"00000" & conv_std_logic_vector (internal_roi_array((3) * 9 + channel_id), 11) &
						"00000" & conv_std_logic_vector (internal_roi_array((2) * 9 + channel_id), 11) &
						"00000" & conv_std_logic_vector (internal_roi_array((1) * 9 + channel_id), 11) &
						"00000" & conv_std_logic_vector (internal_roi_array((0) * 9 + channel_id), 11);
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_FILLING;

		when WRITE_FILLING =>    -- write FILLING
			state_sig <=  X"1E";
			data_out <= conv_std_logic_vector(0,64); -- filling
			addr_cntr <= addr_cntr + 1;
			state_generate <= WAIT_FOR_ADC;

		when WAIT_FOR_ADC =>
			state_sig <=  X"1F";
		-- !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
			if (adc_wait_cnt < 4 ) then -- anpassen!!!! -- 3 fr Simulation, 4 fr FPGA???
				adc_wait_cnt <= adc_wait_cnt + 1;
			else
				state_generate <= WRITE_ADC_DATA;
			end if;

--		when CATCH_ADC_DATA =>
--			adc_data_sig <= adc_data_array;
--			adc_otr_sig <= adc_otr;
--			state_generate <= PREPARE_ADC_DATA_01;
--			
--		when PREPARE_ADC_DATA_01 =>
--			for i in 3 downto 0 loop
--				if (adc_otr_sig(i)='1') then 
--					if (adc_data_sig(i)(11)='1') then
--						data_16bit(i) = X"8000" --this is decimal -1
--					else -- adc_data_sig(i)(11)='0'
--						data_16bit(i) = X"1001" --this is decimal +4097
--					end if;
--				else -- no over/underflow
--					data_16bit(i) = 
--			end loop;
			
		
			
		when WRITE_ADC_DATA =>
			state_sig <=  X"20";
			if (data_cntr < roi_max_int (channel_id)) then
				data_out <=
				--DRS chip 3 LOW BYTE
					(adc_otr(3)  xor adc_data_array(3)(7)) & 
					(adc_otr(3)  xor adc_data_array(3)(6)) & 
					(adc_otr(3)  xor adc_data_array(3)(5)) & 
					(adc_otr(3)  xor adc_data_array(3)(4)) & 
					(adc_otr(3)  xor adc_data_array(3)(3)) & 
					(adc_otr(3)  xor adc_data_array(3)(2)) & 
					(adc_otr(3)  xor adc_data_array(3)(1)) & 
					(adc_otr(3)  xor adc_data_array(3)(0)) &
				--DRS chip 3 HIGH BYTE
					adc_data_array(3)(11) & 
					adc_data_array(3)(11) & 
					adc_data_array(3)(11) & 
					adc_data_array(3)(11) & 
					(adc_otr(3)  xor adc_data_array(3)(11)) & 
					(adc_otr(3)  xor adc_data_array(3)(10)) & 
					(adc_otr(3)  xor adc_data_array(3)(9)) & 
					(adc_otr(3)  xor adc_data_array(3)(8)) &
				--DRS chip 2 LOW BYTE
					(adc_otr(2)  xor adc_data_array(2)(7)) & 
					(adc_otr(2)  xor adc_data_array(2)(6)) & 
					(adc_otr(2)  xor adc_data_array(2)(5)) & 
					(adc_otr(2)  xor adc_data_array(2)(4)) & 
					(adc_otr(2)  xor adc_data_array(2)(3)) & 
					(adc_otr(2)  xor adc_data_array(2)(2)) & 
					(adc_otr(2)  xor adc_data_array(2)(1)) & 
					(adc_otr(2)  xor adc_data_array(2)(0)) &
				--DRS chip 2 HIGH BYTE
					adc_data_array(2)(11) & 
					adc_data_array(2)(11) & 
					adc_data_array(2)(11) & 
					adc_data_array(2)(11) & 
					(adc_otr(2)  xor adc_data_array(2)(11)) & 
					(adc_otr(2)  xor adc_data_array(2)(10)) & 
					(adc_otr(2)  xor adc_data_array(2)(9)) & 
					(adc_otr(2)  xor adc_data_array(2)(8)) &
				--DRS chip 1 LOW BYTE
					(adc_otr(1)  xor adc_data_array(1)(7)) & 
					(adc_otr(1)  xor adc_data_array(1)(6)) & 
					(adc_otr(1)  xor adc_data_array(1)(5)) & 
					(adc_otr(1)  xor adc_data_array(1)(4)) & 
					(adc_otr(1)  xor adc_data_array(1)(3)) & 
					(adc_otr(1)  xor adc_data_array(1)(2)) & 
					(adc_otr(1)  xor adc_data_array(1)(1)) & 
					(adc_otr(1)  xor adc_data_array(1)(0)) &
				--DRS chip 1 HIGH BYTE              
					adc_data_array(1)(11) &         
					adc_data_array(1)(11) &         
					adc_data_array(1)(11) &         
					adc_data_array(1)(11) &         
					(adc_otr(1)  xor adc_data_array(1)(11)) & 
					(adc_otr(1)  xor adc_data_array(1)(10)) & 
					(adc_otr(1)  xor adc_data_array(1)(9)) & 
					(adc_otr(1)  xor adc_data_array(1)(8)) &
				--DRS chip 0 LOW BYTE
					(adc_otr(0)  xor adc_data_array(0)(7)) & 
					(adc_otr(0)  xor adc_data_array(0)(6)) & 
					(adc_otr(0)  xor adc_data_array(0)(5)) & 
					(adc_otr(0)  xor adc_data_array(0)(4)) & 
					(adc_otr(0)  xor adc_data_array(0)(3)) & 
					(adc_otr(0)  xor adc_data_array(0)(2)) & 
					(adc_otr(0)  xor adc_data_array(0)(1)) & 
					(adc_otr(0)  xor adc_data_array(0)(0)) &
				--DRS chip 0 HIGH BYTE
					adc_data_array(0)(11) & 
					adc_data_array(0)(11) & 
					adc_data_array(0)(11) & 
					adc_data_array(0)(11) & 
					(adc_otr(0)  xor adc_data_array(0)(11)) & 
					(adc_otr(0)  xor adc_data_array(0)(10)) & 
					(adc_otr(0)  xor adc_data_array(0)(9)) & 
					(adc_otr(0)  xor adc_data_array(0)(8)) ;
				
					--adc_data_array(3)(7 downto 0) & "000" & adc_otr(3) & adc_data_array(3)(11 downto 8) &
					--adc_data_array(2)(7 downto 0) & "000" & adc_otr(2) & adc_data_array(2)(11 downto 8) &
					--adc_data_array(1)(7 downto 0) & "000" & adc_otr(1) & adc_data_array(1)(11 downto 8) &
					--adc_data_array(0)(7 downto 0) & "000" & adc_otr(0) & adc_data_array(0)(11 downto 8) ;

				addr_cntr <= addr_cntr + 1;
				state_generate <= WRITE_ADC_DATA;
				data_cntr <= data_cntr + 1;
			else
				drs_clk_en <= '0';
				--adc_output_enable_inverted <= '1'; -- nur fr Emulator
				if (channel_id = 8) then
					state_generate <= WAIT_FOR_EXTERNAL_TRIGGER_READY;
					adc_output_enable_inverted <= '1';
					-- switch off ADC_CLK
					adc_clk_en <= '0';
				else
					channel_id <= channel_id + 1;     -- increment channel_id 
					state_generate <= START_DRS_READING;
					data_cntr <= 0;
				end if;
			end if;
		
		when WAIT_FOR_EXTERNAL_TRIGGER_READY =>
			state_sig <=  X"21";
			state_generate <= WAIT_FOR_EXTERNAL_TRIGGER_READY;
			if (FTM_RS485_ready = '1') then
				--make local copy and proceed
				FTM_trigger_info_local_copy <= FTM_trigger_info;
				state_generate <= WRITE_EXTERNAL_TRIGGER;
			end if;
		
		
		when WRITE_EXTERNAL_TRIGGER =>    -- external trigger ID
			state_sig <=  X"22";
			addr_out <= start_addr + conv_std_logic_vector(1, RAM_ADDR_WIDTH);
			data_out <=	FTM_trigger_info_local_copy(15 downto 0) &
						FTM_trigger_info_local_copy(31 downto 16) &
						FTM_trigger_info_local_copy(47 downto 32) &
						"0000000"& FTM_receiver_status & FTM_trigger_info_local_copy(55 downto 48);
			state_generate <= WRITE_END_FLAG;

		when WRITE_END_FLAG =>
			state_sig <=  X"23";
			data_out <= conv_std_logic_vector(0, 32) & X"04FE" & X"4242";
			addr_cntr <= addr_cntr + 1;
			state_generate <= WRITE_DATA_END;
		when WRITE_DATA_END =>
			state_sig <=  X"24";
			dataRAM_write_ea_o <= "0";
				--information to: memory manager. 
				-- one Event was completely written into dataRAM.
				
			-- post writing handshake with MM:
			-- if 'write_enable' = '1' everything is normal and DG informs MM
					-- about finished write process by raising 'write_ready'
			-- if 'write_enable' = '0' 
				-- something strange happened and waiting for 'write_enable' going '1' might cause a dead lock.
				-- so the reaction is to go back to IDLE state without informing MM.
				-- this means actually discarding the event.
			if ( ram_write_ea ='1' ) then
				ram_write_ready <= '1';
				state_generate <= WRITE_DATA_END_WAIT;
			else
				state_generate <= WRITE_DATA_STOP;
			end if;
			
		when WRITE_DATA_END_WAIT =>
			state_sig <=  X"25";
			if (ram_write_ea = '0') then
				state_generate <= WRITE_DATA_STOP;
				ram_write_ready <= '0';
			end if;
		when WRITE_DATA_STOP =>
			state_sig <=  X"26";
			drs_readout_ready <= '1'; 			--info to: trigger manager.
			data_cntr <= 0;
			addr_cntr <= 0;
			channel_id <= 0;
			state_generate <= WRITE_DATA_STOP1;
			
		when WRITE_DATA_STOP1 =>
			state_sig <=  X"27";
			if (drs_readout_ready_ack = '1') then
				drs_readout_ready <= '0';
				state_generate <= IDLE;
			end if;
		when others =>
			null;
		end case; -- state_generate
	end if; -- rising_edge (clk)
end process generate_data;
end Behavioral;