/*
 * QtGl.cpp
 *
 *  Created on: Jul 19, 2011
 *      Author: lyard
 */

#include "RawEventsViewer.h"
#include "viewer.h"
#include <math.h>
#include <fstream>

#ifdef LOAD_RAW
int16_t eventsData[NUM_STORED_EVENTS][ACTUAL_NUM_PIXELS][1024];
#include </home/lyard/Code/display.C>
#endif

#define VALUES_SPAN 4096

#include <QtGui/QFileDialog>

#include <qwt-qt4/qwt_plot_grid.h>
#include <qwt-qt4/qwt_symbol.h>

#include "src/Configuration.h"

//Coordinates of an hexagon of radius 1 and center 0
GLfloat hexcoords[6][2] = {{-0.577367206,  1},
                           { 0.577367206,  1},
                           { 1.154734411,  0},
                           { 0.577367206, -1},
                           {-0.577367206, -1},
                           {-1.154734411,  0}};

//bounding box for diplaying the impulse curve
float bboxMin[2] = {-0.8,-0.9};
float bboxMax[2] = {0.8,-0.3};
/************************************************************
 * UPDATE NEIGHBORS recalculate the neighbors of the current pixels.
 * Only takes the previous pixels into account (and updates them, too)
 ************************************************************/
void RawDataViewer::updateNeighbors(int currentPixel)
{
    float squaredDistance = 0;
    for (int i=0;i<currentPixel;i++)
    {
        squaredDistance = (pixelsCoords[i][0] - pixelsCoords[currentPixel][0])*
                          (pixelsCoords[i][0] - pixelsCoords[currentPixel][0]) +
                          (pixelsCoords[i][1] - pixelsCoords[currentPixel][1])*
                          (pixelsCoords[i][1] - pixelsCoords[currentPixel][1]);
        if (squaredDistance < 4*hexRadius*hexRadius*(1.0f+hexTolerance))//neighbor !
        {//ok, but which one ?
            if (fabs(pixelsCoords[i][0] - pixelsCoords[currentPixel][0]) < hexTolerance &&
                pixelsCoords[i][1] < pixelsCoords[currentPixel][1]){//top
                neighbors[i][0] = currentPixel;
                neighbors[currentPixel][3] = i;
                continue;}
            if (fabs(pixelsCoords[i][0] - pixelsCoords[currentPixel][0]) < hexTolerance &&
                pixelsCoords[i][1] > pixelsCoords[currentPixel][1]){//bottom
                neighbors[i][3] = currentPixel;
                neighbors[currentPixel][0] = i;
                continue;}
            if (pixelsCoords[i][0] > pixelsCoords[currentPixel][0] &&
                pixelsCoords[i][1] > pixelsCoords[currentPixel][1]){//top right
                neighbors[i][4] = currentPixel;
                neighbors[currentPixel][1] = i;
                continue;}
            if (pixelsCoords[i][0] > pixelsCoords[currentPixel][0] &&
                pixelsCoords[i][1] < pixelsCoords[currentPixel][1]){//bottom right
                neighbors[i][5] = currentPixel;
                neighbors[currentPixel][2] = i;
                continue;}
            if (pixelsCoords[i][0] < pixelsCoords[currentPixel][0] &&
                pixelsCoords[i][1] > pixelsCoords[currentPixel][1]){//top left
                neighbors[i][2] = currentPixel;
                neighbors[currentPixel][5] = i;
                continue;}
            if (pixelsCoords[i][0] < pixelsCoords[currentPixel][0] &&
                pixelsCoords[i][1] < pixelsCoords[currentPixel][1]){//bottom left
                neighbors[i][1] = currentPixel;
                neighbors[currentPixel][4] = i;
                continue;}
        }
    }
}
/************************************************************
 * SKIP PIXELS skips a given number of pixels.
 * Only update the pixel coordinates. i.e. update neighbors must
 * called again afterwards.
 ************************************************************/
void RawDataViewer::skipPixels(int start, int howMany)
{
    for (int i=start;i<MAX_NUM_PIXELS-howMany;i++)
    {
        pixelsCoords[i][0] = pixelsCoords[i+howMany][0];
        pixelsCoords[i][1] = pixelsCoords[i+howMany][1];
    }
}
/************************************************************
 * CALCULATE PIXELS COORDS as the name suggests
 ************************************************************/
void RawDataViewer::calculatePixelsCoords()
{
    pixelsCoords[0][0] = 0;
    pixelsCoords[0][1] = 0.3;
    pixelsCoords[0][2] = 0;
    pixelsCoords[1][0] = 0;
    pixelsCoords[1][1] = 0.3+2*hexRadius;
    pixelsCoords[1][2] = 0;
    neighbors[0][0] = 1;
    neighbors[1][3] = 0;
    //from which side of the previous hexagon are we coming from ?
    int fromSide = 3;
    //to which side are we heading to ?
    int toSide = 0;
    for (int i=2;i<MAX_NUM_PIXELS;i++)
    {
        toSide = fromSide-1;
        if (toSide < 0)
            toSide =5;
        while (neighbors[i-1][toSide] >= 0)
        {
            toSide--;
            if (toSide < 0)
                toSide = 5;
        }
        fromSide = toSide + 3;
        if (fromSide > 5)
            fromSide -= 6;
        //ok. now we now in which direction we're heading
        pixelsCoords[i][0] = pixelsCoords[i-1][0];
        pixelsCoords[i][1] = pixelsCoords[i-1][1];
        pixelsCoords[i][2] = pixelsCoords[i-1][2];
        switch (toSide)
        {
        case 0:
            pixelsCoords[i][1] += 2*hexRadius;
        break;
        case 1:
            pixelsCoords[i][0] += (2*hexRadius)*sin(M_PI/3.0);
            pixelsCoords[i][1] += (2*hexRadius)*cos(M_PI/3.0);
        break;
        case 2:
            pixelsCoords[i][0] += (2*hexRadius)*sin(M_PI/3.0);
            pixelsCoords[i][1] -= (2*hexRadius)*cos(M_PI/3.0);
        break;
        case 3:
            pixelsCoords[i][1] -= 2*hexRadius;
        break;
        case 4:
            pixelsCoords[i][0] -= (2*hexRadius)*sin(M_PI/3.0);
            pixelsCoords[i][1] -= (2*hexRadius)*cos(M_PI/3.0);
        break;
        case 5:
            pixelsCoords[i][0] -= (2*hexRadius)*sin(M_PI/3.0);
            pixelsCoords[i][1] += (2*hexRadius)*cos(M_PI/3.0);
        break;
        };

        updateNeighbors(i);
    }
    //Ok. So now we've circled around all the way to MAX_NUM_PIXELS
    //do the required shifts so that it matches the fact camera up to ACTUAL_NUM_PIXELS pixels
    skipPixels(1200, 1);
    skipPixels(1218, 3);
    skipPixels(1236, 1);
    skipPixels(1256, 1);
    skipPixels(1274, 3);
    skipPixels(1292, 3);
    skipPixels(1309, 6);
    skipPixels(1323, 7);
    skipPixels(1337, 6);
    skipPixels(1354, 6);
    skipPixels(1368, 7);
    skipPixels(1382, 9);
    skipPixels(1394, 12);
    skipPixels(1402, 15);
    skipPixels(1410, 12);
    skipPixels(1422, 12);
    skipPixels(1430, 15);
}
/************************************************************
 * BUILD VERTICES LIST. from the coordinates of the camera pixels,
 * calculate the list and coordinates of the vertices required to draw the
 * entire camera.
 ************************************************************/
void RawDataViewer::buildVerticesList()
{
    numVertices = 0;
    GLfloat cVertex[2];
    for (int i=0;i<ACTUAL_NUM_PIXELS;i++)
    {
        for (int j=0;j<6;j++)
        {
            for (int k=0;k<2;k++)
                cVertex[k] = hexcoords[j][k]*hexRadius + pixelsCoords[i][k];
            bool found = false;
            for (int k=0;k<numVertices;k++)
            {
                if ((cVertex[0] - verticesList[k][0])*
                    (cVertex[0] - verticesList[k][0]) +
                    (cVertex[1] - verticesList[k][1])*
                    (cVertex[1] - verticesList[k][1]) < hexTolerance*hexTolerance)
                    {
                        found = true;
                        break;
                    }
            }
            if (!found)
            {
                for (int k=0;k<2;k++)
                    verticesList[numVertices][k] = cVertex[k];
                numVertices++;
            }
        }
    }
    for (int i=0;i<ACTUAL_NUM_PIXELS;i++)
    {
        for (int j=0;j<6;j++)
        {
            for (int k=0;k<2;k++)
                cVertex[k] = hexcoords[j][k]*hexRadius + pixelsCoords[i][k];
            for (int k=0;k<numVertices;k++)
            {
                if ((cVertex[0] - verticesList[k][0])*
                     (cVertex[0] - verticesList[k][0]) +
                     (cVertex[1] - verticesList[k][1])*
                     (cVertex[1] - verticesList[k][1]) < hexTolerance*hexTolerance)
                     {
                        verticesIndices[i][j] = k;
                        break;
                     }
            }
        }
    }
}
/************************************************************
 * BUILD PATCHES INDICES. from the list of patches, crawls through
 * the list of camera pixels and build the list of edges that should be kept
 * in order to display the patches' contours.
 ************************************************************/
void RawDataViewer::buildPatchesIndices()
{
    vector<edge>::iterator it;
    bool erased = false;
    for (int i=0;i<NTMARK;i++)//for all patches
    {
        patchesIndices[i].clear();
        for (int j=0;j<9;j++)//for all cells of the current patch
        {
            if (patches[i][j] == 690 ||
                patches[i][j] == 70)
                continue;
            for (int k=0;k<6;k++)//for all sides of the current cell
            {
                int first = k-1;
                int second = k;
                if (first < 0)
                    first = 5;
                erased = false;
                for (it=(patchesIndices[i]).begin(); it != (patchesIndices[i]).end(); it++)//check if this side is here already or not
                {
                    if (((*it).first == verticesIndices[patches[i][j]][first] &&
                         (*it).second == verticesIndices[patches[i][j]][second]) ||
                        ((*it).first == verticesIndices[patches[i][j]][second] &&
                         (*it).second == verticesIndices[patches[i][j]][first]))
                    {
                        patchesIndices[i].erase(it);
                        erased = true;
                        break;
                    }
                }
                if (!erased)
                {
                    edge temp;
                    temp.first = verticesIndices[patches[i][j]][first];
                    temp.second = verticesIndices[patches[i][j]][second];
                    patchesIndices[i].push_back(temp);
                }
            }
        }
    }
}
/************************************************************
 * CALC BLUR COLOR if in blur display mode, calculate the interpolated
 * colour for a given vertex
 ************************************************************/
void RawDataViewer::calcBlurColor(int pixel,  int vertex)
{
    GLfloat color[3];
    int first, second;
    first = vertex-1;
    second = vertex;
    if (first < 0)
        first = 5;
    first = neighbors[pixel][first];
    second = neighbors[pixel][second];
    for (int i=0;i<3;i++)
        color[i] = pixelsColor[pixel][i];
    float divide = 1;
    if (first != -1)
    {
        divide++;
        for (int i=0;i<3;i++)
            color[i] += pixelsColor[first][i];
    }
    if (second != -1)
    {
        divide++;
        for (int i=0;i<3;i++)
            color[i] += pixelsColor[second][i];
    }
    for (int i=0;i<3;i++)
        color[i] /= divide;
    glColor3fv(color);
}
/************************************************************
 * DRAW BLURRY HEXAGON. draws a solid hexagon, with interpolated colours
 ************************************************************/
void RawDataViewer::drawBlurryHexagon(int index)
{
    GLfloat color[3];
    for (int i=0;i<3;i++)
        color[i] = pixelsColor[index][i];
    glBegin(GL_TRIANGLES);
    calcBlurColor(index, 0);
    glVertex2fv(verticesList[verticesIndices[index][0]]);
    glColor3fv(color);
    glVertex2fv(pixelsCoords[index]);
    calcBlurColor(index, 1);
    glVertex2fv(verticesList[verticesIndices[index][1]]);

    glVertex2fv(verticesList[verticesIndices[index][1]]);
    glColor3fv(color);
    glVertex2fv(pixelsCoords[index]);
    calcBlurColor(index, 2);
    glVertex2fv(verticesList[verticesIndices[index][2]]);

    glVertex2fv(verticesList[verticesIndices[index][2]]);
    glColor3fv(color);
    glVertex2fv(pixelsCoords[index]);
    calcBlurColor(index, 3);
    glVertex2fv(verticesList[verticesIndices[index][3]]);

    glVertex2fv(verticesList[verticesIndices[index][3]]);
    glColor3fv(color);
    glVertex2fv(pixelsCoords[index]);
    calcBlurColor(index, 4);
    glVertex2fv(verticesList[verticesIndices[index][4]]);

    glVertex2fv(verticesList[verticesIndices[index][4]]);
    glColor3fv(color);
    glVertex2fv(pixelsCoords[index]);
    calcBlurColor(index, 5);
    glVertex2fv(verticesList[verticesIndices[index][5]]);

    glVertex2fv(verticesList[verticesIndices[index][5]]);
    glColor3fv(color);
    glVertex2fv(pixelsCoords[index]);
    calcBlurColor(index, 0);
    glVertex2fv(verticesList[verticesIndices[index][0]]);
    glEnd();

    return;
}
/************************************************************
 * DRAW HEXAGON. draws a single hexagon.
 ************************************************************/
void RawDataViewer::drawHexagon(int index, bool solid)
{
    if (solid)
        glBegin(GL_POLYGON);
    else
        glBegin(GL_LINE_LOOP);

    glVertex2fv(verticesList[verticesIndices[index][0]]);
    glVertex2fv(verticesList[verticesIndices[index][1]]);
    glVertex2fv(verticesList[verticesIndices[index][2]]);
    glVertex2fv(verticesList[verticesIndices[index][3]]);
    glVertex2fv(verticesList[verticesIndices[index][4]]);
    glVertex2fv(verticesList[verticesIndices[index][5]]);
    if (solid)
        glVertex2fv(verticesList[verticesIndices[index][0]]);

    glEnd();

    return;
}


float ss[5] = {0.00, 0.25, 0.5, 0.75, 1.00};
float rr[5] = {0.15, 0.00, 0.00, 1.00, 0.85};
float gg[5] = {0.15, 0.00, 1.00, 0.00, 0.85};
float bb[5] = {0.15, 1.00, 0.00, 0.00, 0.85};
/*
float ss[5] = {0., 0.47, 0.475, 0.48, 1.00};
float rr[5] = {0., 0.35, 0.85, 1.00, 1.00};
float gg[5] = {0., 0.10, 0.20, 0.73, 1.00};
float bb[5] = {0., 0.03, 0.06, 0.00, 1.00};
*/
/************************************************************
 * DRAW PATCHES. draws the clustering patches
 ************************************************************/
void RawDataViewer::drawPatches()
{
    glLineWidth(2.0f);
    float backupRadius = hexRadius;
    hexRadius *= 0.95;
    glBegin(GL_LINES);
    for (int i=0;i<NTMARK;i++)
    {
        glColor3fv(patchesColor[i]);
        for (unsigned int j=0;j<patchesIndices[i].size();j++)
        {
            glVertex2fv(verticesList[patchesIndices[i][j].first]);
            glVertex2fv(verticesList[patchesIndices[i][j].second]);
        }
    }
    glEnd();
    hexRadius = backupRadius;
}
/************************************************************
 * DRAW CAMERA draws all the camera pixels
 ************************************************************/
void RawDataViewer::drawCamera(bool alsoWire)
{
    glColor3f(0.5,0.5,0.5);
    glLineWidth(1.0);
    float color;

    for (int i=0;i<ACTUAL_NUM_PIXELS;i++)
    {
        if (!nRoi)
          color = (float)(i)/(float)(ACTUAL_NUM_PIXELS);
        else
#ifdef LOAD_RAW
        color = float(eventsData[eventNum][i][whichSlice]+(VALUES_SPAN/2))/(float)(VALUES_SPAN-1);
#else
        color = float(eventData[nRoi*i + whichSlice]+(VALUES_SPAN/2))/(float)(VALUES_SPAN-1);
#endif
        int index = 0;
        while (ss[index] < color)
            index++;
        index--;
        float weight0 = (color-ss[index]) / (ss[index+1]-ss[index]);
        float weight1 = 1.0f-weight0;
        pixelsColor[i][0] = weight1*rr[index] + weight0*rr[index+1];
        pixelsColor[i][1] = weight1*gg[index] + weight0*gg[index+1];
        pixelsColor[i][2] = weight1*bb[index] + weight0*bb[index+1];
    }

    for (int i=0;i<ACTUAL_NUM_PIXELS;i++)
    {
        if (i == 690 ||
            i == 70)
            continue;
        glColor3fv(pixelsColor[i]);
        glLoadName(i);
if (drawBlur)
    drawBlurryHexagon(i);
else
    drawHexagon(i,true);

    }
    if (!alsoWire)
        return;
    glColor3f(0.0f,0.0f,0.0f);
    for (int i=0;i<ACTUAL_NUM_PIXELS;i++)
    {
        if (i == 690 ||
            i == 70)
            continue;
        drawHexagon(i, false);
    }

}

/************************************************************
 * TRIM. FIXME this should not be here but taken from an existing class (somewhere)
 ************************************************************/
string Trim(const string &str)
{
    // Trim Both leading and trailing spaces
    const size_t start = str.find_first_not_of(' '); // Find the first character position after excluding leading blank spaces
    const size_t end   = str.find_last_not_of(' ');  // Find the first character position from reverse af

    // if all spaces or empty return an empty string
    if (string::npos==start || string::npos==end)
        return string();

    return str.substr(start, end-start+1);
}
/************************************************************
 * DRAW PIXEL CURVE. draws the raw impulse curve of the currently selected pixel
 ************************************************************/
void RawDataViewer::drawPixelCurve()
{
    if (!nRoi)
        return;

    float xZoom, yZoom;
    xZoom = yZoom = 1.0f;

    glBegin(GL_LINES);
    glLineWidth(1.0f);
    glColor3f(0.5,0.5,0.5);
    glVertex2f(bboxMin[0], bboxMin[1]);
    glVertex2f(bboxMax[0], bboxMin[1]);
    glVertex2f(bboxMin[0], bboxMin[1]);
    glVertex2f(bboxMin[0], bboxMax[1]);
    glVertex2f(bboxMin[0], (bboxMin[1]+bboxMax[1])/2.0f);
    glVertex2f(bboxMax[0], (bboxMin[1]+bboxMax[1])/2.0f);
    float xRange = bboxMax[0] - bboxMin[0];
    float yRange = bboxMax[1] - bboxMin[1];
    glColor3f(1.0,1.0,1.0);
    float divideMe = (float)(VALUES_SPAN-1);
    float plusMe = VALUES_SPAN/2;
    if (drawCalibrationLoaded && calibrationLoaded)
    {
        divideMe /=2;
        plusMe /=2;
    }
    for (int i=0;i<nRoi-1;i++)
    {
#ifdef LOAD_RAW
        glVertex2f(bboxMin[0] + xRange*i/(float)nRoi,
                   bboxMin[1] + yRange*(eventsData[eventNum][selectedPixel][i]+plusMe) /divideMe);
        glVertex2f(bboxMin[0] + xRange*(i+1)/(float)nRoi,
                   bboxMin[1] + yRange*(eventsData[eventNum][selectedPixel][i+1]+plusMe) /divideMe);
#else

        glVertex2f(bboxMin[0] + xRange*i/(float)nRoi,
                   bboxMin[1] + yRange*(eventData[nRoi*selectedPixel + i]+plusMe) /divideMe);
        glVertex2f(bboxMin[0] + xRange*(i+1)/(float)nRoi,
                   bboxMin[1] + yRange*(eventData[nRoi*selectedPixel + i+1]+plusMe) /divideMe);
#endif
    }
    glColor3f(1.0,0.0,0.0);
    glVertex2f(bboxMin[0] + xRange*whichSlice/(float)nRoi,
               bboxMin[1]);
    glVertex2f(bboxMin[0] + xRange*whichSlice/(float)nRoi,
               bboxMax[1]);
    glEnd();
    glEnable(GL_MULTISAMPLE);
    setFont(QFont("Times", 12));
    qglColor(QColor(255,223,127));
    float xShift = 0.10f;
    float yShift = 0.01f;
    renderText(bboxMin[0]-xShift/2.0f, bboxMax[1]+3*yShift, 0, QString("Volts"));
    renderText(bboxMin[0]-xShift, bboxMax[1]-yShift,0,QString("+1.05"));
    renderText(bboxMin[0]-xShift,  ((bboxMin[1]+bboxMax[1])/2.0f) - yShift, 0, QString("+0.00"));//((bboxMin[1]+bboxMax[1])/2.0f)
    renderText(bboxMin[0]-xShift, bboxMin[1]-yShift, 0, QString("-1.05"));

    renderText(bboxMax[0]+xShift/2.0f, bboxMin[1]-4*yShift, 0, QString("Slices"));
    renderText(bboxMin[0]-yShift/2.0f, bboxMin[1]-4*yShift, 0, QString("0"));
    ostringstream str;
    str << nRoi/2;
    renderText(((bboxMin[0]+bboxMax[0])/2.0f)-xShift/2.0f, bboxMin[1]-4*yShift, 0, QString(str.str().c_str()));
    str.str("");
    str << nRoi;
    renderText(bboxMax[0]-xShift/2.0f, bboxMin[1]-4*yShift, 0, QString(str.str().c_str()));

}
/************************************************************
 * CONSTRUCTOR.
 ************************************************************/
RawDataViewer::RawDataViewer(QWidget *cParent) : QGLWidget(cParent)
{
    setFormat(QGLFormat(QGL::DoubleBuffer));// | QGL::DepthBuffer));
    hexRadius = 0.015f;
    hexTolerance = hexRadius/100.0f;
    viewSize = 1.0f;
    whichSlice = 0;
#ifdef LOAD_RAW
    nRoi = 1024;
#else
    nRoi = 0;
#endif
    eventNum = 0;
    rowNum = -1;
    eventStep = 1;
    selectedPixel = 393;
    inputFile = NULL;
    eventData = NULL;
    drawPatch = true;
    drawImpulse = false;
    drawBlur = false;
    loopCurrentEvent = false;
#ifdef LOAD_RAW
    loadEvents("/scratch/00000043.001_T.bin");
#endif
    calculatePixelsCoords();

    ifstream fin2("MasterList-v3.txt");
    if (!fin2.is_open())
    {
        cout << "Error: file \"MasterList-v3\" missing. aborting." << endl;
        exit(-1);
    }
    int l = 0;
    string buf;
    while (getline(fin2, buf, '\n'))
    {
        buf = Trim(buf);
        if (buf[0]=='#')
            continue;

        unsigned int softid, hardid, dummy;

        stringstream str(buf);

        str >> softid;
        str >> dummy;
        str >> hardid;

        if (softid>=1440)
            continue;

        hardwareMapping[softid] = hardid;
        softwareMapping[hardid] = softid;

        l++;
    }
    GLfloat tempPixelsCoords[MAX_NUM_PIXELS][3];
    for (int i=0;i<1440;i++)
        for (int j=0;j<3;j++)
            tempPixelsCoords[hardwareMapping[i]][j] = pixelsCoords[i][j];
    for (int i=0;i<1440;i++)
        for (int j=0;j<3;j++)
            pixelsCoords[i][j] = tempPixelsCoords[i][j];
    buildVerticesList();
   ifstream fin1("Trigger-Patches.txt");
   if (!fin1.is_open())
   {
       cout << "Error: file \"Trigger-Patches.txt\" missing. Aborting." << endl;
       exit(-1);
   }
   l=0;
    while (getline(fin1, buf, '\n'))
    {
        buf = Trim(buf);
        if (buf[0]=='#')
            continue;

        stringstream str(buf);
        for (int i=0; i<9; i++)
        {
            unsigned int n;
            str >> n;

            if (n>=1440)
                continue;

            patches[l][i] = hardwareMapping[n];
        }
        l++;
    }
    buildPatchesIndices();
    float color[3];
    for (int i=0;i<160;i++)
    {
        color[0] = 0.5; color[1] = 0.5; color[2] = 0.3;
        for (int j=0;j<3;j++)
            patchesColor[i][j] = color[j];
    }

    for (int i=0;i<1440;i++)
        updateNeighbors(i);

    calibrationLoaded = false;
    drawCalibrationLoaded = false;

}
/************************************************************
 *  DESTRUCTOR
 ************************************************************/
RawDataViewer::~RawDataViewer()
{
    if (inputFile != NULL)
    {
        inputFile->close();
        delete inputFile;
    }
    if (eventData != NULL) {
        delete[] eventData;
        delete[] rawEventData;
    }
}
/************************************************************
 * INITIALIZE GL. does not do much.
 ************************************************************/
void RawDataViewer::initializeGL()
{
    qglClearColor(QColor(25,25,38));
    glShadeModel(GL_FLAT);
    glDisable(GL_DEPTH_TEST);
    glDisable(GL_CULL_FACE);
//    glEnable(GL_LINE_SMOOTH);
//    glHint(GL_LINE_SMOOTH_HINT,GL_NICEST);
}

/************************************************************
 * RESIZE GL. reshapes the ortho projection to match the current window size
 ************************************************************/
void RawDataViewer::resizeGL(int cWidth, int cHeight)
{
    glViewport(0, 0, cWidth, cHeight);
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    GLfloat windowRatio = (float)cWidth/(float)cHeight;
    if (windowRatio < 1)
    {
        windowRatio = 1.0f/windowRatio;
        gluOrtho2D(-viewSize, viewSize, -viewSize*windowRatio, viewSize*windowRatio);
        pixelSize = 2*viewSize/(float)cWidth;
        shownSizex = 2*viewSize;
        shownSizey = 2*viewSize*windowRatio;
    }
    else
    {
        gluOrtho2D(-viewSize*windowRatio, viewSize*windowRatio, -viewSize, viewSize);
        pixelSize = 2*viewSize/(float)cHeight;
        shownSizex = 2*viewSize*windowRatio;
        shownSizey = 2*viewSize;
    }
    glMatrixMode(GL_MODELVIEW);
}

/************************************************************
 * PAINT GL. main drawing function.
 ************************************************************/
void RawDataViewer::paintGL()
{
    glClear(GL_COLOR_BUFFER_BIT);
    glLoadIdentity();

    if (!drawImpulse)
    {
        glTranslatef(0,-0.44,0);
        glScalef(1.5,1.5,1);
    }
    if (drawBlur)
    {
        glShadeModel(GL_SMOOTH);
        drawCamera(false);
    }
    else
    {
        glShadeModel(GL_FLAT);
        drawCamera(true);
    }
    if (drawImpulse)
    {
        glLineWidth(2.0);
        drawPixelCurve();
    }
    if (drawPatch)
        drawPatches();

   if (!drawBlur)
   {
        glLineWidth(1.0f);
        glColor3f(1.0,1.0,1.0);
        drawHexagon(selectedPixel, false);
   }
}

/************************************************************
 * MOUSE PRESS EVENT. mouse click handler.
 ************************************************************/
void RawDataViewer::mousePressEvent(QMouseEvent *cEvent)
{
    lastPos = cEvent->pos();
    setCorrectSlice(cEvent);
    updateGL();
}

/************************************************************
 * SET CORRECT SLICE. if displayed, figures out if the graph was
 * clicked, and if so, which slice should be displayed
 ************************************************************/
void RawDataViewer::setCorrectSlice(QMouseEvent* cEvent)
{
    if (!drawImpulse)
        return;
    float cx = (float)cEvent->x() * pixelSize - shownSizex/2;
    float cy = ((float)height()-(float)cEvent->y())*pixelSize - shownSizey/2;
    if (cx < bboxMin[0] ||
        cx > bboxMax[0] ||
        cy < bboxMin[1] ||
        cy > bboxMax[1])
        return;
    whichSlice = (cx - bboxMin[0])*1024/(bboxMax[0] - bboxMin[0]);
    emit signalCurrentSlice(whichSlice);
}

/************************************************************
 * MOUSE MOVE EVENT. used to track the dragging of slices display
 ************************************************************/
void RawDataViewer::mouseMoveEvent(QMouseEvent *cEvent)
{
    if (cEvent->buttons() & Qt::LeftButton) {
        setCorrectSlice(cEvent);
        updateGL();
    } else if (cEvent->buttons() & Qt::RightButton) {
        updateGL();
    }
    lastPos = cEvent->pos();
}

/************************************************************
 * MOUSE DOUBLE CLICK EVENT. used to select pixels
 ************************************************************/
void RawDataViewer::mouseDoubleClickEvent(QMouseEvent *cEvent)
{
    int face = PixelAtPosition(cEvent->pos());
    if (face != -1) {
        selectedPixel = face;
        emit signalCurrentPixel(face);
        updateGL();
        }
}

/************************************************************
 * PIXEL AT POSITION. figures out which camera pixel was clicked.
 ************************************************************/
int RawDataViewer::PixelAtPosition(const QPoint &cPos)
{
    const int MaxSize = 512;
    GLuint buffer[MaxSize];
    GLint viewport[4];

    makeCurrent();

    glGetIntegerv(GL_VIEWPORT, viewport);
    glSelectBuffer(MaxSize, buffer);
    glRenderMode(GL_SELECT);

    glInitNames();
    glPushName(0);

    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    GLfloat windowRatio = GLfloat(width()) / GLfloat(height());
    gluPickMatrix(GLdouble(cPos.x()), GLdouble(viewport[3] - cPos.y()),
            1.0, 1.0, viewport);

    if (windowRatio < 1)
     {
         windowRatio = 1.0f/windowRatio;
         gluOrtho2D(-viewSize, viewSize, -viewSize*windowRatio, viewSize*windowRatio);
     }
     else
     {
         gluOrtho2D(-viewSize*windowRatio, viewSize*windowRatio, -viewSize, viewSize);
     }

    glMatrixMode(GL_MODELVIEW);
    drawCamera(false);
    glMatrixMode(GL_PROJECTION);
    glPopMatrix();

    //for some reason that I cannot understand, the push/pop matrix doesn't do the trick here... bizarre
    //ok, so re-do the resizeGL thing.
    resizeGL(width(), height());

    if (!glRenderMode(GL_RENDER))
        return -1;

    return buffer[3];
}
/************************************************************
 * OPEN FILE. opens a new fits file
 ************************************************************/
void RawDataViewer::openFile(string& file)
{
    if (inputFile)
    {
        inputFile->close();
        delete inputFile;
    }
    inputFile = new fits(file);
    if (!*inputFile)
    {
        delete inputFile;
        inputFile = NULL;
        return;
    }
    vector<string> entriesToCheck;
    entriesToCheck.push_back("NAXIS2");
    entriesToCheck.push_back("NROI");
    entriesToCheck.push_back("NTMARK");
    entriesToCheck.push_back("RUNTYPE");
    entriesToCheck.push_back("REVISION");
    entriesToCheck.push_back("BLDVER");
    entriesToCheck.push_back("RUNID");
    entriesToCheck.push_back("NBOARD");
    entriesToCheck.push_back("NPIX");
    entriesToCheck.push_back("NROITM");
    entriesToCheck.push_back("TIMESYS");
    entriesToCheck.push_back("DATE");
    entriesToCheck.push_back("NIGHT");
    entriesToCheck.push_back("CAMERA");
    entriesToCheck.push_back("DAQ");
    entriesToCheck.push_back("TSTART");
    entriesToCheck.push_back("TSTOP");
    //entriesToCheck.push_back("ADCRANGE");
    //entriesToCheck.push_back("NBEVTOK");
    //entriesToCheck.push_back("NBEVTREJ");
    //entriesToCheck.push_back("NBEVTBAD");

    for (vector<string>::const_iterator it=entriesToCheck.begin(); it != entriesToCheck.end(); it++)
    {
        if (!inputFile->HasKey(*it)){
            cout << *it << " missing. Aborting load..." << endl;
            return;}
    }
    nRows = inputFile->GetInt("NAXIS2");
    nRoi = inputFile->GetInt("NROI");
    runNumber = inputFile->GetInt("RUNID");
    nTM = inputFile->GetInt("NTMARK");
    runType = inputFile->GetInt("RUNTYPE");
    firstDataTime = inputFile->GetInt("TSTART");
    lastDataTime = inputFile->GetInt("TSTOP");
    nRoiTM = inputFile->GetInt("NROITM");
    revision = inputFile->GetInt("REVISION");
    builderVersion = inputFile->GetInt("BLDVER");
    nBoards = inputFile->GetInt("NBOARD");
    nPixels = inputFile->GetInt("NPIX");
    timeSystem = inputFile->GetStr("TIMESYS");
    creationDate = inputFile->GetStr("DATE");
    nightInt = inputFile->GetInt("NIGHT");
    camera = inputFile->GetStr("CAMERA");
    daq = inputFile->GetStr("DAQ");
    adcCount = inputFile->HasKey("ADCRANGE") ? inputFile->GetFloat("ADCRANGE") : 2000;
    nbOk = 0;//inputFile->GetInt("NBEVTOK");
    nbRej = 0;//inputFile->GetInt("NBEVTREJ");
    nbBad = 0;//inputFile->GetInt("NBEVTBAD");

    eventNum = 0;

#ifdef LOAD_RAW
    nRows = NUM_STORED_EVENTS;
#endif

    if (eventData != NULL) {
        delete[] eventData;
        delete[] rawEventData;
    }
    eventData = new float[(1440+160)*nRoi];
    rawEventData = new int16_t[(1440+160)*nRoi];
    if (!inputFile->SetPtrAddress("Data", rawEventData)){
        cout << "Missing column " << "Data" << " Aborting load..." << endl;
        nRoi = nRows = 0;return;}
    if (!inputFile->SetPtrAddress("EventNum", &eventNum)){
        cout << "Missing column " << "EventNum" << " Aborting load..." << endl;
        nRoi = nRows = 0;return;}
    if (!inputFile->SetPtrAddress("TriggerType", &triggerType)){
        cout << "Missing column " << "TriggerType" << " Aborting load..." << endl;
        nRoi = nRows = 0;return;}
    if (!inputFile->SetPtrAddress("SoftTrig", &softTrig)){
        cout << "Missing column " << "SoftTrig" << " Aborting load..." << endl;
        nRoi = nRows = 0;return;}
    if (!inputFile->SetPtrAddress("PCTime", &pcTime)){
        cout << "Missing column " << "PCTime" << " Aborting load..." << endl;
        nRoi = nRows = 0;return;}
    if (!inputFile->SetPtrAddress("BoardTime", boardTime)){
        cout << "Missing column " << "BoardTime" << " Aborting load..." << endl;
        nRoi = nRows = 0;return;}
    if (!inputFile->SetPtrAddress("StartCellData", startPix)){
        cout << "Missing column " << "StartCellData" << " Aborting load..." << endl;
        nRoi = nRows = 0;return;}
    if (!inputFile->SetPtrAddress("StartCellTimeMarker", startTM)){
        cout << "Missing column " << "StartCellTimeMarker" << " Aborting load..." << endl;
        nRoi = nRows = 0;return;}
    int backupStep = eventStep;
    rowNum = -1;
    eventStep = 1;
    plusEvent();
    eventStep = backupStep;
    emit newFileLoaded();
    emit signalCurrentPixel(selectedPixel);
}
void RawDataViewer::openCalibFile(string& file)
{
    calibrationLoaded = false;
    calibInputFile = new fits(file);
    if (!*calibInputFile)
    {
        delete calibInputFile;
        calibInputFile = NULL;
        return;
    }

    if (!calibInputFile->SetPtrAddress("BaselineMean", baseLineMean)){
        cout << "Missing column " << "BaseLineMean" << " Aborting load..." << endl;
        return;}
    if (!calibInputFile->SetPtrAddress("GainMean", gainMean)){
        cout << "Missing column " << "GainMean" << " Aborting load..." << endl;
        return;}
    if (!calibInputFile->SetPtrAddress("TriggerOffsetMean", triggerOffsetMean)){
        cout << "Missing column " << "TriggerOffsetMean" << " Aborting load..." << endl;
        return;}

    calibInputFile->GetNextRow();

    for (int i=0;i<1024;i++)
        cout << gainMean[i] << " ";
    cout << endl << endl;

    delete calibInputFile;

    calibrationLoaded = true;
    emit newFileLoaded();
    if (drawCalibrationLoaded)
        updateGL();
}
/************************************************************
 * PLUS EVENT
 ************************************************************/
void RawDataViewer::plusEvent()
{
    eventStepping(true);
}
/************************************************************
 * MINUS EVENT
 ************************************************************/
void RawDataViewer::minusEvent()
{
    eventStepping(false);
}
/************************************************************
 * SET EVENT STEP
 ************************************************************/
void RawDataViewer::setEventStep(int step)
{
    eventStep = step;
}
/************************************************************
 * EVENT STEPPING
 ************************************************************/
void RawDataViewer::eventStepping(bool plus)
{
    if (plus)
        rowNum += eventStep;
    else
        rowNum -= eventStep;
    if (rowNum >= nRows)
        rowNum -= nRows;
    if (rowNum < 0)
        rowNum += nRows;
#ifdef LOAD_RAW
    eventNum+=eventStep;
#else
    if (inputFile == NULL)
        return;
    inputFile->GetRow(rowNum);
    for (int i=0;i<(1440+160)*nRoi;i++)
        eventData[i] = (float)rawEventData[i];
#endif

    if (drawCalibrationLoaded && calibrationLoaded)
    {
        for (int i=0;i<1440;i++)
            for (int j=0;j<nRoi;j++)
            {
                eventData[i*1024+j] *= 2000.f/4096.f;
                eventData[i*1024+j] -= (baseLineMean[i*1024+j]+triggerOffsetMean[i*1024+j]);
                eventData[i*1024+j] /= gainMean[i*1024+j];
                eventData[i*1024+j] *= (50000.f/65536.f) * 2500.f;
            }
    }
    updateGL();
    emit signalCurrentEvent(eventNum);
    emit signalCurrentPixel(selectedPixel);
}
/************************************************************
 * NEXT SLICE. deprec ?
 ************************************************************/
void RawDataViewer::nextSlice()
{
    whichSlice++;
    if (whichSlice >= nRoi)
    {
        whichSlice = 0;
        if (!loopCurrentEvent)
        {
            int backupStep = eventStep;
            eventStep = 1;
            eventStepping(true);
            eventStep = backupStep;
        }
    }
    emit signalCurrentSlice(whichSlice);
    updateGL();
}
void RawDataViewer::previousSlice()
{
    whichSlice--;
    if (whichSlice < 0)
    {
        whichSlice = nRoi-1;
        if (!loopCurrentEvent)
        {
            int backupStep = eventStep;
            eventStep = 1;
            eventStepping(false);
            eventStep = backupStep;
        }
    }
    emit signalCurrentSlice(whichSlice);
    updateGL();
}
/************************************************************
 * UICONNECTOR CONSTRUCTOR
 ************************************************************/
UIConnector::UIConnector(QWidget*)
{
    timer.setInterval(10.0);
    QObject::connect(&timer, SIGNAL(timeout()),
                      this, SLOT(nextSlicePlease()));
    hwID = 393;
    swID = 0;
    crateID = 9;
    boardID = 8;
    patchID = 1;
    rescaleWholeCamera = true;
    currentFile = "none";
    currentCalibFile = "none";
}
void UIConnector::slicesPlusPlus()
{
    viewer->nextSlice();
}
void UIConnector::slicesMinusMinus()
{
    viewer->previousSlice();
}
void UIConnector::drawCalibratedDataChanged(int state)
{
     if (state)
    {
        if (viewer->calibrationLoaded)
        {
            viewer->drawCalibrationLoaded = true;
            for (int i=0;i<1440;i++)
                for (int j=0;j<viewer->nRoi;j++)
                {
//                    cout << "start value: " << viewer->eventData[i*1024+j] << " baseline: " << viewer->baseLineMean[i*1024+j];
//                    cout << " triggerOffset: " << viewer->triggerOffsetMean[i*1024+j] << " gain: " << viewer->gainMean[i*1024+j];
                    viewer->eventData[i*1024+j] *= 2000.f/4096.f;
                    viewer->eventData[i*1024+j] -= (viewer->baseLineMean[i*1024+j]+viewer->triggerOffsetMean[i*1024+j]);
                    viewer->eventData[i*1024+j] /= viewer->gainMean[i*1024+j];
                    viewer->eventData[i*1024+j] *= (50000.f/65536.f) * 2500.f;
//                    cout << " end value: " << viewer->eventData[i*1024+j] << endl;
                }
            viewer->updateGL();
        }
        else
        {
            drawCalibrationCheckBox->setChecked(false);
        }
    }
    else
    {
        viewer->drawCalibrationLoaded = false;
        if (viewer->calibrationLoaded)
        {
            for (int i=0;i<1440;i++)
                for (int j=0;j<viewer->nRoi;j++)
                {

                    viewer->eventData[i*1024+j] /= (50000.f/65536.f) * 2500.f;
                    viewer->eventData[i*1024+j] *= viewer->gainMean[i*1024+j];
                    viewer->eventData[i*1024+j] += (viewer->baseLineMean[i*1024+j]+viewer->triggerOffsetMean[i*1024+j]);
                    viewer->eventData[i*1024+j] /= 2000.f/4096.f;
               }
            viewer->updateGL();
        }

    }

}
/************************************************************
 * DRAW PATCHES CHECK CHANGE. checkbox handler
 ************************************************************/
void UIConnector::drawPatchesCheckChange(int state)
{
    if (state)
        viewer->drawPatch = true;
    else
        viewer->drawPatch = false;
    viewer->updateGL();
}
/************************************************************
 * DRAW IMPULSE CHECK CHANGE. checkbox handler
 ************************************************************/
void UIConnector::drawImpulseCheckChange(int state)
{
    if (state)
        viewer->drawImpulse = true;
    else
        viewer->drawImpulse = false;
    viewer->updateGL();
}
/************************************************************
 * DRAW BLUR CHECK CHANGE. checkbox handler
 ************************************************************/
void UIConnector::drawBlurCheckChange(int state)
{
    if (state)
        viewer->drawBlur = true;
    else
        viewer->drawBlur = false;
    viewer->updateGL();
}
void UIConnector::loopEventCheckChange(int state)
{
    if (state)
        viewer->loopCurrentEvent = true;
    else
        viewer->loopCurrentEvent = false;
}
/************************************************************
 * NEXT SLICE PLEASE
 ************************************************************/
void UIConnector::nextSlicePlease()
{
    viewer->nextSlice();
}
/************************************************************
 * SET VIEWER.
 ************************************************************/
void UIConnector::setViewer(RawDataViewer* v)
{
    viewer = v;
}
/************************************************************
 * SLICES PER SECOND CHANGED. timing ui handler
 ************************************************************/
void UIConnector::slicesPerSecondChanged(double value)
{
    timer.setInterval(1000.0/value);
}
/************************************************************
 * RANGE CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::rangeChanged0(double value)
{
    ss[0] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * RANGE CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::rangeChanged1(double value)
{
    ss[1] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * RANGE CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::rangeChanged2(double value)
{
    ss[2] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * RANGE CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::rangeChanged3(double value)
{
    ss[3] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * RANGE CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::rangeChanged4(double value)
{
    ss[4] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * RANGE CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::redChanged0(double value)
{
    rr[0] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * RED CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::redChanged1(double value)
{
    rr[1] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * RED CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::redChanged2(double value)
{
    rr[2] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * RED CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::redChanged3(double value)
{
    rr[3] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * RED CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::redChanged4(double value)
{
    rr[4] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * GREEN CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::greenChanged0(double value)
{
    gg[0] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * GREEN CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::greenChanged1(double value)
{
    gg[1] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * GREEN CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::greenChanged2(double value)
{
    gg[2] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * GREEN CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::greenChanged3(double value)
{
    gg[3] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * GREEN CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::greenChanged4(double value)
{
    gg[4] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * BLUE CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::blueChanged0(double value)
{
    bb[0] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * BLUE CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::blueChanged1(double value)
{
    bb[1] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * BLUE CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::blueChanged2(double value)
{
    bb[2] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * BLUE CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::blueChanged3(double value)
{
    bb[3] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * BLUE CHANGED . colors tweaking handler
 ************************************************************/
void UIConnector::blueChanged4(double value)
{
    bb[4] = (float)value;
    viewer->updateGL();
}
/************************************************************
 * LOAD NEW FILE CLICKED. button handler
 ************************************************************/
void UIConnector::loadNewFileClicked()
{
    QFileDialog dialog;
    dialog.setFileMode(QFileDialog::ExistingFile);
    dialog.open(this, SLOT(fileSelected(QString)));
    dialog.setVisible(true);
    dialog.exec();
}
void UIConnector::loadNewCalibFileClicked()
{
    QFileDialog dialog;
    dialog.setFileMode(QFileDialog::ExistingFile);
    dialog.open(this, SLOT(calibFileSelected(QString)));
    dialog.setVisible(true);
    dialog.exec();
}
/************************************************************
 * FILE SELECTED. return of the file open dialog handler
 ************************************************************/
void UIConnector::fileSelected(QString file)
{
    currentFile = file.toStdString();
    if (currentFile != "")
        viewer->openFile(currentFile);
}
void UIConnector::calibFileSelected(QString file)
{
    currentCalibFile = file.toStdString();
    if (currentCalibFile != "")
        viewer->openCalibFile(currentCalibFile);
}
/************************************************************
 * NEW FILE LOADED. update of the UI after a new file has been loaded
 ************************************************************/
void UIConnector::newFileLoaded()
{
    ostringstream str;
    str << "File loaded: " << currentFile << "\n";
    str << "Calibration file loaded: " << currentCalibFile << "\n";
//    fileLoadedLabel->setText(QString(str.str().c_str()));
//    str.str("");
    str << "Run number: " << viewer->runNumber << "\n";
//    runNumberLabel->setText(QString(str.str().c_str()));
//    str.str("");
    str << "Number of Events: " << viewer->nRows << "\n";
    str << "Number of Slices: " << viewer->nRoi << "\n";// << "/1024";
//    numberOfSlicesLabel->setText(QString(str.str().c_str()));
//    str.str("");
    str << "Number of Time Marks: " << viewer->nTM << "\n";
//    numberOfTimeMarksLabel->setText(QString(str.str().c_str()));

//    str.str("");
    str << "Run Type: " << viewer->runType << "\n";
//    runTypeLabel->setText(QString(str.str().c_str()));
//    str.str("");
    str << "Time of 1st data: " << viewer->firstDataTime << "\n";
//    firstTimeLabel->setText(QString(str.str().c_str()));
//    str.str("");
    str << "Time of last data: " << viewer->lastDataTime << "\n";
//    lastTimeLabel->setText(QString(str.str().c_str()));
//    str.str("");
    str << "SVN revision: " << viewer->revision << '\n';
    str << "Number of boards: " << viewer->nBoards << '\n';
    str << "Number of pixels: " << viewer->nPixels << '\n';
    str << "Number of Slices TM: " << viewer->nRoiTM << '\n';
    str << "Time system: " << viewer->timeSystem << '\n';
    str << "Date: " << viewer->creationDate << '\n';
    str << "Night: " << viewer->nightInt << '\n';
    str << "Camera: " << viewer->camera << '\n';
    str << "DAQ: " << viewer->daq << '\n';
    str << "ADC Count: " << viewer->adcCount << '\n';
    str << "NB Evts OK:" << viewer->nbOk << '\n';
    str << "NB Evts Rejected: " << viewer->nbRej << '\n';
    str << "NB Evts Bad: " << viewer->nbBad << '\n';
    extraInfoLabel->setText(QString(str.str().c_str()));


}
/************************************************************
 * PLAY PAUSE CLICKED. ui handler
 ************************************************************/
void UIConnector::playPauseClicked()
{
    if (timer.isActive())
        timer.stop();
    else
        timer.start();
}
/************************************************************
 * CURRENT SLICE HAS CHANGE. ui handler
 ************************************************************/
void UIConnector::currentSliceHasChanged(int slice)
{
    ostringstream str;
    str << "Displaying Slice " << slice;
    QString qstr(str.str().c_str());
    emit updateCurrentSliceDisplay(qstr);

    str.str("");
    str << "Current Pixel val.: " << viewer->eventData[viewer->nRoi*viewer->selectedPixel + viewer->whichSlice];
    qstr = qstr.fromStdString(str.str());
    emit updateCurrentPixelSliceValue(qstr);

}
/************************************************************
 * CURRENT EVENT HAS CHANGED. ui handler
 ************************************************************/
double xval[4096];
double yval[4096];

void UIConnector::currentEventHasChanged(int cEvent)
{
    ostringstream str;
    str << "Displaying Event " << cEvent;
    QString qstr(str.str().c_str());
    emit updateCurrentEventDisplay(qstr);
    //retrieve the data that we want to display
    str.str("");
    str << "PC Time: " << viewer->pcTime;
    qstr = qstr.fromStdString(str.str());
    emit updateCurrentPCTime(qstr);

    str.str("");
    str << "Software Trigger: " << viewer->softTrig;
    qstr = qstr.fromStdString(str.str());
    emit updateCurrentSoftTrigger(qstr);

    str.str("");
    str << "Trigger Type: " << viewer->triggerType;
    qstr = qstr.fromStdString(str.str());
    emit updateCurrentTriggerType(qstr);

    str.str("");
    str << "Current Pixel val.: " << viewer->eventData[viewer->nRoi*viewer->selectedPixel + viewer->whichSlice];
    qstr = qstr.fromStdString(str.str());
    emit updateCurrentPixelSliceValue(qstr);

    boardsTimeList->clear();
    startPixelsList->clear();
    startTimeMarksList->clear();
    triggerDelayList->clear();
    std::map<int, int> boardsHistoMap;
    for (int i=0;i <NBOARDS; i++)
    {
        str.str("");
        str << i;
        if (i<10) str << " ";
        if (i<100) str << " ";
        if (i<1000) str << " ";
        str << ": " << viewer->boardTime[i];
        boardsTimeList->addItem(QString(str.str().c_str()));
        if (boardsHistoMap.find(viewer->boardTime[i]) != boardsHistoMap.end())
            boardsHistoMap[viewer->boardTime[i]]++;
        else
            boardsHistoMap[viewer->boardTime[i]] = 1;
    }
    std::map<int, int> pixelHistoMap;
    for (int i=0;i <NPIX; i++)
    {
        str.str("");
        str << i;
        if (i<10) str << " ";
        if (i<100) str << " ";
        if (i<1000) str << " ";
        str << ": " << viewer->startPix[i];
        startPixelsList->addItem(QString(str.str().c_str()));
        if (pixelHistoMap.find(viewer->startPix[i]) != pixelHistoMap.end())
            pixelHistoMap[viewer->startPix[i]]++;
        else
            pixelHistoMap[viewer->startPix[i]] = 1;
    }
    std::map<int, int> timeMarksMap;
    for (int i=0;i <NTMARK; i++)
    {
        str.str("");
        str << i;
        if (i<10) str << " ";
        if (i<100) str << " ";
        if (i<1000) str << " ";
        str << ": " << viewer->startTM[i];
        startTimeMarksList->addItem(QString(str.str().c_str()));
        if (timeMarksMap.find(viewer->startTM[i]) != timeMarksMap.end())
            timeMarksMap[viewer->startTM[i]]++;
        else
            timeMarksMap[viewer->startTM[i]] = 1;
    }
    std::map<int,int> delayMap;
    triggerDelayList->addItem(QString("Patch | Slice:Delay Slice:Delay..."));
    for (int i=0;i<NTMARK; i++)
    {
        str.str("");
        str << i << " | ";
        for (int j=0;j<viewer->nRoi;j++)
        {
            int value = viewer->eventData[1440*viewer->nRoi + i*viewer->nRoi + j];
            if (delayMap.find(value) != delayMap.end())
                 delayMap[value]++;
             else
                 delayMap[value] = 1;
            str << j << ":" << value << " ";
         }
        triggerDelayList->addItem(QString(str.str().c_str()));
    }

    std::map<int,int>::iterator it = boardsHistoMap.begin();
    int nsamples = 0;
    int previousValue = it->first-10;
    for (unsigned int i=0;i<boardsHistoMap.size();i++)
    {
        if (previousValue != it->first-1)
        {
            xval[nsamples] = previousValue+1;
            yval[nsamples] = 0;
            nsamples++;
            xval[nsamples] = it->first-1;
            yval[nsamples] = 0;
            nsamples++;
        }
        xval[nsamples] = it->first;
        yval[nsamples] = it->second;
        previousValue = it->first;
        it++;
        nsamples++;
        xval[nsamples] = previousValue;
        yval[nsamples] = 0;
        nsamples++;
        if (nsamples > 4090)
        {
            cout << "Error: Maximum number of samples reached for histograms. skipping what's remaining" << endl;
            break;
        }
    }
    xval[nsamples] = it==boardsHistoMap.begin() ? 0 : (--it)->first+1;
    yval[nsamples] = 0;
    nsamples++;
 //   if (nsamples > 5)
#if QWT_VERSION < 0x060000
       boardsTimeHistoItem.setData(xval, yval, nsamples);
#else
       boardsTimeHistoItem.setSamples(xval, yval, nsamples);
#endif

    it = pixelHistoMap.begin();
    nsamples = 0;
    previousValue = it->first-10;
    for (unsigned int i=0;i<pixelHistoMap.size();i++)
    {
        if (previousValue != it->first-1)
        {
            xval[nsamples] = previousValue+1;
            yval[nsamples] = 0;
            nsamples++;
            xval[nsamples] = it->first-1;
            yval[nsamples] = 0;
            nsamples++;
        }
        xval[nsamples] = it->first;
        yval[nsamples] = it->second;
        previousValue = it->first;
        it++;
        nsamples++;
        xval[nsamples] = previousValue;
        yval[nsamples] = 0;
        nsamples++;
        if (nsamples > 4090)
        {
            cout << "Error: Maximum number of samples reached for histograms. skipping what's remaining" << endl;
            break;
        }
   }
    xval[nsamples] = it==pixelHistoMap.begin() ? 0 : (--it)->first+1;
    yval[nsamples] = 0;
    nsamples++;
//    if (nsamples > 5)
#if QWT_VERSION < 0x060000
       startCellHistoItem.setData(xval, yval, nsamples);
#else
       startCellHistoItem.setSamples(xval, yval, nsamples);
#endif

    it = timeMarksMap.begin();
    nsamples = 0;
    previousValue = it->first-10;
    for (unsigned int i=0;i<timeMarksMap.size();i++)
    {
        if (previousValue != it->first-1)
        {
            xval[nsamples] = previousValue+1;
            yval[nsamples] = 0;
            nsamples++;
            xval[nsamples] = it->first-1;
            yval[nsamples] = 0;
            nsamples++;
        }
        xval[nsamples] = it->first;
        yval[nsamples] = it->second;
        previousValue = it->first;
        it++;
        nsamples++;
        xval[nsamples] = previousValue;
        yval[nsamples] = 0;
        nsamples++;
        if (nsamples > 4090)
        {
            cout << "Error: Maximum number of samples reached for histograms. skipping what's remaining" << endl;
            break;
        }
    }
    xval[nsamples] = it==timeMarksMap.begin() ? 0 : (--it)->first+1;
    yval[nsamples] = 0;
    nsamples++;
 //   if (nsamples > 5)
#if QWT_VERSION < 0x060000
       startTimeMarkHistoItem.setData(xval, yval, nsamples);
#else
       startTimeMarkHistoItem.setSamples(xval, yval, nsamples);
#endif

    it = delayMap.begin();
    nsamples = 0;
    previousValue = it->first-10;
    for (unsigned int i=0;i<delayMap.size();i++)
    {
        if (previousValue != it->first-1)
        {
            xval[nsamples] = previousValue+1;
            yval[nsamples] = 0;
            nsamples++;
            xval[nsamples] = it->first-1;
            yval[nsamples] = 0;
            nsamples++;
        }
        xval[nsamples] = it->first;
        yval[nsamples] = it->second;
        previousValue = it->first;
        it++;
        nsamples++;
        xval[nsamples] = previousValue;
        yval[nsamples] = 0;
        nsamples++;
        if (nsamples > 4090)
        {
            cout << "Error: Maximum number of samples reached for histograms. skipping what's remaining" << endl;
            break;
        }
    }
    xval[nsamples] = it==delayMap.begin() ? 0 : (--it)->first+1;
    yval[nsamples] = 0;
    nsamples++;
  //  if (nsamples > 5)
#if QWT_VERSION < 0x060000
       triggerDelayHistoItem.setData(xval, yval, nsamples);
#else
       triggerDelayHistoItem.setSamples(xval, yval, nsamples);
#endif

//    startCellHistoZoom->setZoomBase(startCellHistoItem.boundingRect());
    QStack< QRectF > stack;
//    QRectF cRectangle = boardsTimeHistoItem.boundingRect();
    stack.push(scaleBoundingRectangle(boardsTimeHistoItem.boundingRect(), 1.05f));//cRectangle);//boardsTimeHistoItem.boundingRect());
    boardsTimeHistoZoom->setZoomStack(stack);
    stack.pop();
    stack.push(scaleBoundingRectangle(startCellHistoItem.boundingRect(), 1.05f));
    startCellHistoZoom->setZoomStack(stack);
    stack.pop();
    stack.push(scaleBoundingRectangle(startTimeMarkHistoItem.boundingRect(), 1.05f));
    startTimeMarkHistoZoom->setZoomStack(stack);
    stack.pop();
    stack.push(scaleBoundingRectangle(triggerDelayHistoItem.boundingRect(), 1.05f));
    triggerDelayHistoZoom->setZoomStack(stack);
    stack.pop();
    pixelChanged(viewer->selectedPixel);
}
//can't use a ref to rectangle, as the type must be converted first
QRectF UIConnector::scaleBoundingRectangle(QRectF rectangle, float scale)
{
    QPointF bottomRight = rectangle.bottomRight();
    QPointF topLeft = rectangle.topLeft();
    QPointF center = rectangle.center();
    return QRectF(topLeft + (topLeft-center)*(scale-1.0f), //top left
                  bottomRight + (bottomRight-center)*(scale-1.0f)); //bottom right
}
void UIConnector::initHistograms()
{
//    QwtPlot*     boardsTimeHisto;
//    QwtPlotHistogram boardsTimeHistoItem;
    QwtPlotGrid* grid = new QwtPlotGrid;
    grid->enableX(false);
    grid->enableY(true);
    grid->enableXMin(false);
    grid->enableYMin(false);
    grid->setMajPen(QPen(Qt::black, 0, Qt::DotLine));
    grid->attach(boardsTimeHisto);

    grid = new QwtPlotGrid;
    grid->enableX(false);
    grid->enableY(true);
    grid->enableXMin(false);
    grid->enableYMin(false);
    grid->setMajPen(QPen(Qt::black, 0, Qt::DotLine));
    grid->attach(startCellHisto);

    grid = new QwtPlotGrid;
    grid->enableX(false);
    grid->enableY(true);
    grid->enableXMin(false);
    grid->enableYMin(false);
    grid->setMajPen(QPen(Qt::black, 0, Qt::DotLine));
    grid->attach(startTimeMarkHisto);

    grid = new QwtPlotGrid;
    grid->enableX(false);
    grid->enableY(true);
    grid->enableXMin(false);
    grid->enableYMin(false);
    grid->setMajPen(QPen(Qt::black, 0, Qt::DotLine));
    grid->attach(pixelValueCurve);

    grid = new QwtPlotGrid;
    grid->enableX(false);
    grid->enableY(true);
    grid->enableXMin(false);
    grid->enableYMin(false);
    grid->setMajPen(QPen(Qt::black, 0, Qt::DotLine));
    grid->attach(triggerDelayHisto);


    boardsTimeHisto->setAutoReplot(true);
    startCellHisto->setAutoReplot(true);
    startTimeMarkHisto->setAutoReplot(true);
    pixelValueCurve->setAutoReplot(true);
    triggerDelayHisto->setAutoReplot(true);
    boardsTimeHisto->setTitle("Boards time values");
    startCellHisto->setTitle("Start Cell values");
    startTimeMarkHisto->setTitle("Start Time Marks values");
    pixelValueCurve->setTitle("Current pixel values");
    triggerDelayHisto->setTitle("Trigger Delays");

 //   boardsTimeHistoItem.setBrush(QBrush(Qt::red));
//    startCellHistoItem.setBrush(QBrush(Qt::red));
//    startTimeMarkHistoItem.setBrush(QBrush(Qt::red));
//    triggerDelayHistoItem.setBrush(QBrush(Qt::red));
//    pixelValueCurveItem.setBrush(QBrush(Qt::red));

    boardsTimeHistoItem.setPen(QColor(Qt::darkGreen));
    boardsTimeHistoItem.setStyle(QwtPlotCurve::Steps);
    startCellHistoItem.setPen(QColor(Qt::darkGreen));
    startCellHistoItem.setStyle(QwtPlotCurve::Steps);
    startTimeMarkHistoItem.setPen(QColor(Qt::darkGreen));
    startTimeMarkHistoItem.setStyle(QwtPlotCurve::Steps);
    triggerDelayHistoItem.setPen(QColor(Qt::darkGreen));
    triggerDelayHistoItem.setStyle(QwtPlotCurve::Steps);

    boardsTimeHistoItem.attach(boardsTimeHisto);
    startCellHistoItem.attach(startCellHisto);
    startTimeMarkHistoItem.attach(startTimeMarkHisto);
    triggerDelayHistoItem.attach(triggerDelayHisto);

    //curve
//    pixelValueCurveItem.setSymbol(new QwtSymbol(QwtSymbol::Cross, Qt::NoBrush, QPen(Qt::black), QSize(5,5)));
    pixelValueCurveItem.setPen(QColor(Qt::darkGreen));
    pixelValueCurveItem.setStyle(QwtPlotCurve::Lines);
//    pixelValueCurveItem.setCurveAttribute(QwtPlotCurve::Fitted);
    pixelValueCurveItem.attach(pixelValueCurve);

    //FIXME delete these pointers with the destructor
    curveZoom = new QwtPlotZoomer(pixelValueCurve->canvas());
    curveZoom->setRubberBandPen(QPen(Qt::gray, 2, Qt::DotLine));
    curveZoom->setTrackerPen(QPen(Qt::gray));
    boardsTimeHistoZoom = new QwtPlotZoomer(boardsTimeHisto->canvas());
    boardsTimeHistoZoom->setRubberBandPen(QPen(Qt::gray, 2, Qt::DotLine));
    boardsTimeHistoZoom->setTrackerPen(QPen(Qt::gray));
    startCellHistoZoom = new QwtPlotZoomer(startCellHisto->canvas());
    startCellHistoZoom->setRubberBandPen(QPen(Qt::gray, 2, Qt::DotLine));
    startCellHistoZoom->setTrackerPen(QPen(Qt::gray));
    startTimeMarkHistoZoom = new QwtPlotZoomer(startTimeMarkHisto->canvas());
    startTimeMarkHistoZoom->setRubberBandPen(QPen(Qt::gray, 2, Qt::DotLine));
    startTimeMarkHistoZoom->setTrackerPen(QPen(Qt::gray));
    triggerDelayHistoZoom = new QwtPlotZoomer(triggerDelayHisto->canvas());
    triggerDelayHistoZoom->setRubberBandPen(QPen(Qt::gray, 2, Qt::DotLine));
    triggerDelayHistoZoom->setTrackerPen(QPen(Qt::gray));


}
void UIConnector::pixelChanged(int pixel)
{
    if (!viewer->nRoi)
        return;

    for (int i=0;i<viewer->nRoi;i++)
    {
        xval[i] = i;
#ifdef LOAD_RAW
        yval[i] = eventsData[0][pixel][i];
#else
        yval[i] = viewer->eventData[viewer->nRoi*pixel + i];
#endif
    }
#if QWT_VERSION < 0x060000
       pixelValueCurveItem.setData(xval, yval, viewer->nRoi);
#else
       pixelValueCurveItem.setSamples(xval, yval, viewer->nRoi);
#endif

    QStack< QRectF > stack;
    stack.push(scaleBoundingRectangle(pixelValueCurveItem.boundingRect(), 1.05f));
    curveZoom->setZoomStack(stack);
    stack.pop();

    hwID = pixel;
    swID = viewer->softwareMapping[pixel];
    crateID = (pixel/4)/10;
    boardID = (pixel/4)%10;
    patchID = pixel%4;
    HwIDBox->setValue(hwID);
    SwIDBox->setValue(swID);
    crateIDBox->setValue(crateID);
    boardIDBox->setValue(boardID);
    patchIDBox->setValue(patchID);

    ostringstream str;
    QString qstr;
    str.str("");
    str << "Current Pixel val.: " << viewer->eventData[viewer->nRoi*viewer->selectedPixel + viewer->whichSlice];
    qstr = qstr.fromStdString(str.str());
    emit updateCurrentPixelSliceValue(qstr);

//    cout << "Pixel: " << viewer->softwareMapping[pixel] << " Hardware ID: " << pixel << " Patch: " << pixel%4;
//    cout << " Board: " << (pixel/4)%10 << " Crate: " << (pixel/4)/10 << endl;

 //   curveZoom->setZoomBase();
//    cout << "pixel changed ! " << pixel << endl;
}

void UIConnector::hwIDChanged(int hwid)
{
    hwID = hwid;
    swID = viewer->softwareMapping[hwid];
    crateID = (hwid/4)/10;
    boardID = (hwid/4)%10;
    patchID = hwid%4;
    HwIDBox->setValue(hwID);
    SwIDBox->setValue(swID);
    crateIDBox->setValue(crateID);
    boardIDBox->setValue(boardID);
    patchIDBox->setValue(patchID);

    viewer->selectedPixel = hwid;
    pixelChanged(hwid);
    viewer->updateGL();

}
void UIConnector::swIDChanged(int swid)
{
    swID = swid;
    hwID = viewer->hardwareMapping[swid];
    crateID = (hwID/4)/10;
    boardID = (hwID/4)%10;
    patchID = hwID%4;
    HwIDBox->setValue(hwID);
    SwIDBox->setValue(swID);
    crateIDBox->setValue(crateID);
    boardIDBox->setValue(boardID);
    patchIDBox->setValue(patchID);

    viewer->selectedPixel = hwID;
    pixelChanged(hwID);
    viewer->updateGL();
}
void UIConnector::crateIDChanged(int cid)
{
    crateID = cid;
    hwID = 40*crateID + 4*boardID + patchID;
    swID = viewer->softwareMapping[hwID];
    HwIDBox->setValue(hwID);
    SwIDBox->setValue(swID);
    viewer->selectedPixel = hwID;
    pixelChanged(hwID);
    viewer->updateGL();
}
void UIConnector::boardIDChanged(int bid)
{
    boardID = bid;
    hwID = 40*crateID + 4*boardID + patchID;
    swID = viewer->softwareMapping[hwID];
    HwIDBox->setValue(hwID);
    SwIDBox->setValue(swID);
    viewer->selectedPixel = hwID;
    pixelChanged(hwID);
   viewer->updateGL();
}
void UIConnector::patchIDChanged(int pid)
{
    patchID = pid;
    hwID = 40*crateID + 4*boardID + patchID;
    swID = viewer->softwareMapping[hwID];
    HwIDBox->setValue(hwID);
    SwIDBox->setValue(swID);
    viewer->selectedPixel = hwID;
    pixelChanged(hwID);
    viewer->updateGL();
}
void UIConnector::autoScalePressed()
{
    if (!viewer->nRoi)
        return;
    int start, end;
    if (rescaleWholeCamera)
    {
        start = 0;
        end = 1440;
    }
    else
    {
        start = viewer->selectedPixel;
        end = viewer->selectedPixel+1;
    }

    int min = 10000; //real min = -2048
    int max = -10000; //real max = 2047
    long average = 0;
    int numSamples = 0;
    for (int i=start;i<end;i++)
    {
        for (int j=0;j<viewer->nRoi;j++)
        {
            int cValue = viewer->eventData[i*viewer->nRoi+j];
            if (cValue > max)
                max = cValue;
            if (cValue < min)
               min = cValue;
            average+=cValue;
            numSamples++;
        }
    }
    average /= numSamples;

    double minRange = (double)(min+(VALUES_SPAN/2))/(double)VALUES_SPAN;
    double maxRange = (double)(max+(VALUES_SPAN/2))/(double)VALUES_SPAN;
    double midRange = (double)(average+(VALUES_SPAN/2))/(double)VALUES_SPAN;
    ss[0] = minRange;
    range0->setValue(ss[0]);
    ss[4] = maxRange;
    range4->setValue(ss[4]);
    ss[2] = midRange;
    range2->setValue(ss[2]);
    ss[1] = (minRange+midRange)/2;
    range1->setValue(ss[1]);
    ss[3] = (maxRange+midRange)/2;
    range3->setValue(ss[3]);


}
void UIConnector::entireCameraChanged(bool state)
{
    if (state)
    {
        rescaleWholeCamera = true;
        currentPixelScale->setChecked(false);
    }
    else
    {
        rescaleWholeCamera = false;
        currentPixelScale->setChecked(true);
    }
}
void UIConnector::currentPixelChanged(bool state)
{

    if (state)
    {
        rescaleWholeCamera = false;
        entireCameraScale->setChecked(false);
    }
    else
    {
        rescaleWholeCamera = true;
        entireCameraScale->setChecked(true);
    }
}



void PrintHelp()
{
    cout <<
            "\n"
         << endl;
}
void SetupConfiguration(Configuration& conf)
{
    po::options_description configs("Raw Events Viewer Options");
    configs.add_options()
        ("color.range", vars<double>(), "Range of the display colours")
        ("color.red", vars<double>(), "Range of red values")
        ("color.green", vars<double>(), "Range of green values")
        ("color.blue", vars<double>(), "Range of blue values")
        ("file,f", var<string>(), "File to be loaded at startup")
        ;
    conf.AddOptions(configs);

    po::positional_options_description p;
    p.add("file", 1); // The first positional options
    conf.SetArgumentPositions(p);

}
/************************************************************
 * MAIN PROGRAM FUNCTION.
 ************************************************************/
int main(int argc, char *argv[])
{
    Configuration conf(argv[0]);
    conf.SetPrintUsage(PrintHelp);
    SetupConfiguration(conf);
    if (!conf.DoParse(argc, const_cast<const char**>(argv), PrintHelp))
        return -1;

    if (conf.Has("color.range"))
    {
        vector<double> value = conf.Vec<double>("color.range");
        if (value.size() != 5)
        {
            cout << "Error, colorRange option should have exactly 5 double values" << endl;
            return -1;
        }
        for (int i=0;i<5;i++)
            ss[i] = value[i];
    }

    if (conf.Has("color.red"))
    {
        vector<double> value = conf.Vec<double>("color.red");
        if (value.size() != 5)
        {
            cout << "Error, colorRed option should have exactly 5 double values" << endl;
            return -1;
        }
        for (int i=0;i<5;i++)
            rr[i] = value[i];
    }

    if (conf.Has("color.green"))
    {
        vector<double> value = conf.Vec<double>("color.green");
        if (value.size() != 5)
        {
            cout << "Error, colorGreen option should have exactly 5 double values" << endl;
            return -1;
        }
        for (int i=0;i<5;i++)
            gg[i] = value[i];
    }

    if (conf.Has("color.blue"))
    {
        vector<double> value = conf.Vec<double>("color.blue");
        if (value.size() != 5)
        {
            cout << "Error, colorBlue option should have exactly 5 double values" << endl;
            return -1;
        }
        for (int i=0;i<5;i++)
            bb[i] = value[i];
    }

    QApplication app(argc, argv);

    if (!QGLFormat::hasOpenGL()) {
        std::cerr << "This system has no OpenGL support" << std::endl;
        return 1;
    }

    QMainWindow mainWindow;

    Ui_MainWindow myUi;
    myUi.setupUi(&mainWindow);

    RawDataViewer *canvas = myUi.GLWindow;

    QObject::connect(myUi.eventsMinusButton, SIGNAL(clicked()),
                     canvas, SLOT(minusEvent()));
    QObject::connect(myUi.eventsPlusButton, SIGNAL(clicked()),
                     canvas, SLOT(plusEvent()));
    QObject::connect(myUi.eventsStepBox, SIGNAL(valueChanged(int)),
                     canvas, SLOT(setEventStep(int)));
    myUi.colorRange0->setValue(ss[0]);
    myUi.colorRange1->setValue(ss[1]);
    myUi.colorRange2->setValue(ss[2]);
    myUi.colorRange3->setValue(ss[3]);
    myUi.colorRange4->setValue(ss[4]);
    myUi.redValue0->setValue(rr[0]);
    myUi.redValue1->setValue(rr[1]);
    myUi.redValue2->setValue(rr[2]);
    myUi.redValue3->setValue(rr[3]);
    myUi.redValue4->setValue(rr[4]);
    myUi.greenValue0->setValue(gg[0]);
    myUi.greenValue1->setValue(gg[1]);
    myUi.greenValue2->setValue(gg[2]);
    myUi.greenValue3->setValue(gg[3]);
    myUi.greenValue4->setValue(gg[4]);
    myUi.blueValue0->setValue(bb[0]);
    myUi.blueValue1->setValue(bb[1]);
    myUi.blueValue2->setValue(bb[2]);
    myUi.blueValue3->setValue(bb[3]);
    myUi.blueValue4->setValue(bb[4]);

    UIConnector connector;
    connector.setViewer(canvas);
    connector.boardsTimeList = myUi.boardsTimeList;
    connector.boardsTimeHisto = myUi.boardsTimeHisto;
    connector.startPixelsList = myUi.startPixelsList;
    connector.startCellHisto = myUi.startCellsHisto;
    connector.startTimeMarkHisto = myUi.startTimeMarkHisto;
    connector.pixelValueCurve = myUi.pixelValueCurve;
    connector.triggerDelayHisto = myUi.triggerDelayHisto;
    connector.triggerDelayList = myUi.triggerDelayList;

    connector.startTimeMarksList = myUi.startTimeMarksList;
//    connector.fileLoadedLabel = myUi.fileLoadedLabel;
//    connector.runNumberLabel = myUi.runNumberLabel;
//    connector.numberOfSlicesLabel = myUi.numberOfSlicesLabel;
//    connector.numberOfTimeMarksLabel = myUi.numberOfTimeMarksLabel;
//    connector.runTypeLabel = myUi.runTypeLabel;
//    connector.firstTimeLabel = myUi.timeOfFirstDataLabel;
//    connector.lastTimeLabel = myUi.timeOfLastDataLabel;
    connector.currentPixelValue = myUi.currentPixelValue;

    connector.currentPixelScale = myUi.currentPixelScale;
    connector.entireCameraScale = myUi.entireCameraScale;

    connector.extraInfoLabel = myUi.extraInfoLabel;

    connector.HwIDBox = myUi.HwIDBox;
    connector.SwIDBox = myUi.SwIDBox;
    connector.crateIDBox = myUi.crateIDBox;
    connector.boardIDBox = myUi.boardIDBox;
    connector.patchIDBox = myUi.patchIDBox;

    connector.range0 = myUi.colorRange0;
    connector.range1 = myUi.colorRange1;
    connector.range2 = myUi.colorRange2;
    connector.range3 = myUi.colorRange3;
    connector.range4 = myUi.colorRange4;
    connector.drawCalibrationCheckBox = myUi.calibratedCheckBox;

    connector.initHistograms();

    QObject::connect(myUi.slicesPlusPlusButton, SIGNAL(clicked()),
                    &connector, SLOT(slicesPlusPlus()));
    QObject::connect(myUi.slicesMinusMinusButton, SIGNAL(clicked()),
                     &connector, SLOT(slicesMinusMinus()));
    QObject::connect(myUi.autoScaleColor, SIGNAL(clicked()),
                     &connector, SLOT(autoScalePressed()));
    QObject::connect(myUi.currentPixelScale, SIGNAL(toggled(bool)),
                     &connector, SLOT(currentPixelChanged(bool)));
    QObject::connect(myUi.entireCameraScale, SIGNAL(toggled(bool)),
                     &connector, SLOT(entireCameraChanged(bool)));

    QObject::connect(myUi.HwIDBox, SIGNAL(valueChanged(int)),
                         &connector, SLOT(hwIDChanged(int)));
    QObject::connect(myUi.SwIDBox, SIGNAL(valueChanged(int)),
                         &connector, SLOT(swIDChanged(int)));
    QObject::connect(myUi.crateIDBox, SIGNAL(valueChanged(int)),
                         &connector, SLOT(crateIDChanged(int)));
    QObject::connect(myUi.boardIDBox, SIGNAL(valueChanged(int)),
                         &connector, SLOT(boardIDChanged(int)));
    QObject::connect(myUi.patchIDBox, SIGNAL(valueChanged(int)),
                         &connector, SLOT(patchIDChanged(int)));

 //   connector.pixelChanged(0);
    QObject::connect(canvas, SIGNAL(signalCurrentPixel(int)),
                     &connector, SLOT(pixelChanged(int)));
    QObject::connect(myUi.drawPatchCheckBox, SIGNAL(stateChanged(int)),
                     &connector, SLOT(drawPatchesCheckChange(int)));
    QObject::connect(myUi.drawImpulseCheckBox, SIGNAL(stateChanged(int)),
                     &connector, SLOT(drawImpulseCheckChange(int)));
    QObject::connect(myUi.drawBlurCheckBox, SIGNAL(stateChanged(int)),
                     &connector, SLOT(drawBlurCheckChange(int)));
    QObject::connect(myUi.loopOverCurrentEventBox, SIGNAL(stateChanged(int)),
                     &connector, SLOT(loopEventCheckChange(int)));
    QObject::connect(canvas, SIGNAL(newFileLoaded()),
                     &connector, SLOT(newFileLoaded()));

    QObject::connect(myUi.calibratedCheckBox, SIGNAL(stateChanged(int)),
                     &connector, SLOT(drawCalibratedDataChanged(int)));
    QObject::connect(myUi.loadNewFileButton, SIGNAL(clicked()),
                     &connector, SLOT(loadNewFileClicked()));
    QObject::connect(myUi.loadDRSCalibButton, SIGNAL(clicked()),
                     &connector, SLOT(loadNewCalibFileClicked()));

    QObject::connect(myUi.colorRange0, SIGNAL(valueChanged(double)),
                      &connector, SLOT(rangeChanged0(double)));

    QObject::connect(myUi.colorRange1, SIGNAL(valueChanged(double)),
                      &connector, SLOT(rangeChanged1(double)));

    QObject::connect(myUi.colorRange2, SIGNAL(valueChanged(double)),
                      &connector, SLOT(rangeChanged2(double)));

    QObject::connect(myUi.colorRange3, SIGNAL(valueChanged(double)),
                      &connector, SLOT(rangeChanged3(double)));

    QObject::connect(myUi.colorRange4, SIGNAL(valueChanged(double)),
                      &connector, SLOT(rangeChanged4(double)));

    QObject::connect(myUi.redValue0, SIGNAL(valueChanged(double)),
                      &connector, SLOT(redChanged0(double)));

    QObject::connect(myUi.redValue1, SIGNAL(valueChanged(double)),
                      &connector, SLOT(redChanged1(double)));

    QObject::connect(myUi.redValue2, SIGNAL(valueChanged(double)),
                      &connector, SLOT(redChanged2(double)));

    QObject::connect(myUi.redValue3, SIGNAL(valueChanged(double)),
                      &connector, SLOT(redChanged3(double)));

    QObject::connect(myUi.redValue4, SIGNAL(valueChanged(double)),
                      &connector, SLOT(redChanged4(double)));

    QObject::connect(myUi.greenValue0, SIGNAL(valueChanged(double)),
                      &connector, SLOT(greenChanged0(double)));

    QObject::connect(myUi.greenValue1, SIGNAL(valueChanged(double)),
                      &connector, SLOT(greenChanged1(double)));

    QObject::connect(myUi.greenValue2, SIGNAL(valueChanged(double)),
                      &connector, SLOT(greenChanged2(double)));

    QObject::connect(myUi.greenValue3, SIGNAL(valueChanged(double)),
                      &connector, SLOT(greenChanged3(double)));

    QObject::connect(myUi.greenValue4, SIGNAL(valueChanged(double)),
                      &connector, SLOT(greenChanged4(double)));

    QObject::connect(myUi.blueValue0, SIGNAL(valueChanged(double)),
                      &connector, SLOT(blueChanged0(double)));

    QObject::connect(myUi.blueValue1, SIGNAL(valueChanged(double)),
                      &connector, SLOT(blueChanged1(double)));

    QObject::connect(myUi.blueValue2, SIGNAL(valueChanged(double)),
                      &connector, SLOT(blueChanged2(double)));

    QObject::connect(myUi.blueValue3, SIGNAL(valueChanged(double)),
                      &connector, SLOT(blueChanged3(double)));

    QObject::connect(myUi.blueValue4, SIGNAL(valueChanged(double)),
                      &connector, SLOT(blueChanged4(double)));

    QObject::connect(myUi.slicesPerSecValue, SIGNAL(valueChanged(double)),
                      &connector, SLOT(slicesPerSecondChanged(double)));
    QObject::connect(myUi.playPauseButton, SIGNAL(clicked()),
                     &connector, SLOT(playPauseClicked()));

    QObject::connect(canvas, SIGNAL(signalCurrentSlice(int)),
                     &connector, SLOT(currentSliceHasChanged(int)));
    QObject::connect(canvas, SIGNAL(signalCurrentEvent(int)),
                     &connector, SLOT(currentEventHasChanged(int)));

    QObject::connect(&connector, SIGNAL(updateCurrentSliceDisplay(QString)),
                     myUi.displayingSliceLabel, SLOT(setText(const QString)));
    QObject::connect(&connector, SIGNAL(updateCurrentEventDisplay(QString)),
                     myUi.displayingEventLabel, SLOT(setText(const QString)));
    QObject::connect(&connector, SIGNAL(updateCurrentPCTime(QString)),
                     myUi.PCTimeLabel, SLOT(setText(const QString)));
    QObject::connect(&connector, SIGNAL(updateCurrentSoftTrigger(QString)),
                     myUi.softwareTriggerLabel, SLOT(setText(const QString)));
    QObject::connect(&connector, SIGNAL(updateCurrentTriggerType(QString)),
                     myUi.triggerTypeLabel, SLOT(setText(const QString)));
    QObject::connect(&connector, SIGNAL(updateCurrentPixelSliceValue(const QString)),
                     myUi.currentPixelValue, SLOT(setText(const QString)));

    if (conf.Has("file"))
    {
        string str = conf.Get<string>("file");
        QString qstr(str.c_str());
        connector.fileSelected(qstr);
    }

    mainWindow.show();

    return app.exec();
}

