//
// Class processing user input
//

#include "User.h"
#include <readline/readline.h>

using namespace std;

// Branch table for command evaluation
static const struct CL_Struct { const char *Name;    
								void (User::*CommandPointer)();
								unsigned int MinNumParameter;
								bool NeedCrate;
								const char *Parameters;
								const char *Help;
  } CommandList[] = 
   {{"pixel", &User::cmd_hv, 2, true, "<range> <voltage|default|info>", "Change bias of pixels (to default)"},
	{"channel", &User::cmd_hv, 2, true, "<range> <voltage|default|info>", "Change bias of channels of active crate (to default)"},
    {"gs", &User::cmd_gs, 1, true, "[crate] <volt>", "Global voltage set active crate"},
	{"reset", &User::cmd_reset, 0, true, "", "Reset active crate"},
	{"synch", &User::cmd_synch, 0, true, "", "Synchronize active crate"},
	{"crate", &User::cmd_crate, 1, true, "<number>", "Select active crate"},
	{"status", &User::cmd_status, 0, false, "[dac|current]", "Show status information (DAC values if requested)"},
	{"mode", &User::cmd_mode, 1, false, "<static|dynamic>", "Set voltage stabilization mode (experimental)"},
	{"load", &User::cmd_load, 1, true, "<file>", "Load and set bias settings from file"},
	{"save", &User::cmd_save, 1, true, "<file>", "Save current bias settings to file"},
	{"help", &User::cmd_help, 0, false, "", "Print help"},
	{"exit", &User::cmd_exit, 0, false, "", "Exit program"},
	{".", &User::cmd_shell, 1, false, "<command>", "Execute shell command"}};   

//
// Constructor
//
User::User(): EvidenceServer(SERVER_NAME) {

  MainThread = pthread_self();

  // DIM console service used in PrintMessage()
  ConsoleText = NULL;
  ConsoleOut = new DimService(SERVER_NAME"/ConsoleOut", (char *) "");

  // Get/initialize configuration data
  vector<string> Boards = Tokenize(GetConfig("Boards", "dummy"), " \t");
  GetConfig("TimeOut");
  GetConfig("VoltageLimit");
  GetConfig("MinResetPeriod");
  GetConfig("RampSpeed");
  GetConfig("UpdatePeriod");

  vector<string> Text = Tokenize(GetConfig("DefaultVoltage", ""), " \t");
  
  for (unsigned int i=0; i<Text.size(); i++) {
	DefaultVoltage.push_back(atof(Text[i].c_str()));
  }

  // Open devices
  for (unsigned int i=0; i<Boards.size(); i++) {
  
    class Crate *New = new class Crate(Boards[i], Crates.size(), this);

    if (New->InitOK) {
       PrintMessage("Synchronized and reset crate %s (#%d)\n", Boards[i].c_str(), Crates.size());
       Crates.push_back(New);
    }
    else {
      Message(WARN, "Failed to synchronize crate %s", Boards[i].c_str());
	  delete New;
    }
  }
  ActiveCrate = 0;

  // Create PixelMap instance (map from config server)
  DimRpcInfo RPC((char *) "ConfigRequest", (char *) "");
  RPC.setData((char *) "Misc PixelMap");
  PixMap = new PixelMap(std::string(RPC.getString(), RPC.getSize()));
  
  // Install DIM command (after all initialized)
  DIMCommand = new DimCommand((char *) SERVER_NAME"/Command", (char *) "C", this);

  // Create monitor thread and make accessible for sending signal
  if ((pthread_create(&Thread, NULL, (void * (*)(void *)) LaunchMonitor,(void *) this)) != 0) {
    Message(FATAL, "pthread_create() failed with Monitor thread");
  }
}

//
// Destructor
//
User::~User() {

  int Ret;
  
  // Wait for thread to quit (ignore error if thread did already exit)
  if ((Ret = pthread_cancel(Thread)) != 0) {
	if (Ret != ESRCH) Message(ERROR, "pthread_cancel() failed (%s)", strerror(Ret));
  }
  if ((Ret = pthread_join(Thread, NULL)) != 0) Message(ERROR, "pthread_join() failed (%s)", strerror(Ret));

  // Delete all crates
  for (unsigned int i=0; i<Crates.size(); i++) delete Crates[i];

  delete DIMCommand;    
  delete PixMap;
  delete ConsoleOut;	
  free(ConsoleText);  
}

//
// Process user input
//
void User::commandHandler() {

  // Build string safely
  string Command = string(getCommand()->getString(), getCommand()->getSize());

  // Check if command is legal and ignore empty commands 
  if (getCommand() != DIMCommand || Command.size() < 2) return;

  // Parse command into tokens
  Parameter = Tokenize(Command, " ");

  // Special handling of shell execution
  if (Command[0] == '.') {
    Parameter.clear();
	Parameter.push_back(".");
	Parameter.push_back(Command.substr(1));
  }
  
  // Search for command in command list
  for(unsigned int CmdNumber=0; CmdNumber<sizeof(CommandList)/sizeof(CL_Struct); CmdNumber++) {
    if (Match(Parameter[0], CommandList[CmdNumber].Name)) {
	  // Requested command help?
      if (Parameter.size() == 2 && Match(Parameter[1], "?")) {
		PrintMessage("Usage: %s %s\n%s\n", CommandList[CmdNumber].Name, CommandList[CmdNumber].Parameters, CommandList[CmdNumber].Help);
		return;
	  }

	  // Incorrect number of parameters?
      if (Parameter.size()-1 < CommandList[CmdNumber].MinNumParameter) {
		PrintMessage("Usage: %s %s\n", CommandList[CmdNumber].Name, CommandList[CmdNumber].Parameters);
		return;
	  }

	  // Check if crates needed
	  if (CommandList[CmdNumber].NeedCrate && Crates.empty()) {
		PrintMessage("No crate available\n");
		return;
	  }

	  // Jump to command function
	  (this->*CommandList[CmdNumber].CommandPointer)();
	  return;  
    }
  }
  PrintMessage("Unknown command '%s'\n", Parameter[0].c_str());
}


// Print help
void User::cmd_help() {

  for(unsigned int i=0; i<sizeof(CommandList)/sizeof(CL_Struct); i++) {
    PrintMessage("%-10s%s\n", CommandList[i].Name, CommandList[i].Help);
  }
  
  PrintMessage("\nUse '?' as argument to get more extensive help.\n"
  	"Commands 'pixel' and 'channel' allow and arbitary number of argument pairs.\n"
  	"Items in <> are mandatory, in [] optional, | indicates mutual exclusive or.\n"
    "Ranges can be 'all', a single number or in the form 'a-b'.\n"); 
} 

//
// Synchronize boards
//
void User::cmd_synch() {

  if (Crates[ActiveCrate]->Synch()) PrintMessage("Synchronized crate %d\n", ActiveCrate);
  else PrintMessage("Failed to synchronize crate %d\n", ActiveCrate);
}

//
// Select active crate
//
void User::cmd_crate() {

  int Crate;

  if (!ConvertToInt(Parameter[1], &Crate)) {
     PrintMessage("Error: Wrong number format\n");
     return;   
  }

  // Check limits
  if (Crate<0 || Crate>=(int) Crates.size()) {
    PrintMessage("Crate number %d not existing\n", Crate);
    return;
  }

  ActiveCrate = Crate;
}

//
// Set new bias voltage
//
void User::cmd_hv() {

  unsigned int Errors=0;
  double Double;
  struct Range Chan, Pixel;
  unsigned int Crate, Channel;
  vector< map<unsigned int, double> > Voltages (Crates.size());
  vector < pair <unsigned int, unsigned int> > P;

  // Loop over all parameters
  for (unsigned int n=1; n < Parameter.size()-1; n+=2) {

	// Convert voltage value and check format 
	if (!ConvertToDouble(Parameter[n+1], &Double) && !Match(Parameter[n+1], "default") && !Match(Parameter[n+1], "info")) {
	  PrintMessage("Error: Wrong number format for voltage setting\n");
	  continue;
	}

	// Extract affected channels for this argument pair
	P.clear();

	// Pixel identification?
	if (Match(Parameter[0], "pixel")) {
	  Pixel.Min = 0;
	  Pixel.Max = 1439;

	  if (!ConvertToRange(Parameter[n], Pixel)) {
		PrintMessage("Pixel ID out-of-range for parameter %d, skipping channel\n", n);
		continue;
	  }
	  
	  for (int i=Pixel.Min; i<=Pixel.Max; i++) {
	    Crate = PixMap->Pixel_to_HVcrate(i);
		Channel = PixMap->Pixel_to_HVboard(i)*NUM_CHANNELS + PixMap->Pixel_to_HVchannel(i);
		
		// Skip if pixel ID or corresponding channels not existing 
		if (Crate!=PixMap->PM_ERROR_CODE && Crate<Crates.size() && Channel<MAX_NUM_BOARDS*NUM_CHANNELS) {
		  P.push_back(make_pair(Crate, Channel));
		}
	  }
	}
	// Channel identification
	else {
	  Chan.Min = 0;
	  Chan.Max = MAX_NUM_BOARDS*NUM_CHANNELS-1;	  

	  if (!ConvertToRange(Parameter[n], Chan)) {
		PrintMessage("Numeric conversion or out-of-range error for parameter %d, skipping channel\n", n);
		continue;
	  }
	  
	  for (int i=Chan.Min; i<=Chan.Max; i++) P.push_back(make_pair(ActiveCrate, i));	  
	}
  
	// Loop over all given channels
	for (unsigned int i=0; i<P.size(); i++) {
	  Crate = P[i].first;
	  Channel = P[i].second;

	  // Should only information be printed?
	  if (Match(Parameter[n+1], "info")) {
		PrintMessage("Crate %2d, channel %3d  ", Crate, Channel);
		if (!Crates[Crate]->Present[Channel]) PrintMessage("(channel not present in crate)");
		PrintMessage("\n  Channel is on board %d, board channel %d\n", Channel/32, Channel%32);

		// Print pixel information
		vector<unsigned int> List = PixMap->HV_to_Pixel(Crate, Channel/NUM_CHANNELS, Channel%NUM_CHANNELS);
		PrintMessage("\n  Default voltage: %.2f    Pixel IDs: ", Channel < DefaultVoltage.size() ? DefaultVoltage[Channel] : 0);
		for (unsigned int j=0; j<List.size(); j++) PrintMessage("%u ", List[j]);
		if (List.empty()) PrintMessage("none");

		// Print voltage and current
		PrintMessage("\n  Voltage setpoint: %.2f V (DAC %u)    Current: %.2f uA ", Crates[Crate]->GetVoltage(Channel), Crates[Crate]->GetDAC(Channel), Crates[Crate]->GetCurrent(Channel));

		if (Crates[Crate]->OC[Channel]) PrintMessage("(overcurrent)\n");
		else PrintMessage("\n");

		continue;
	  }

	  // Check that indices are in legal range (safety check, should be unnecessary)
	  if (Crate >= Crates.size() || Channel >=MAX_NUM_BOARDS*NUM_CHANNELS) continue;
 
	  // Voltage change (number starts with + oder -) ignored if current DAC value is zero
	  if (Parameter[n+1][0]=='+' || Parameter[n+1][0]=='-') {
		if (Crates[Crate]->GetDAC(Channel) == 0) continue;
	  }
	
	  // Should the default value be set?
	  if (Match(Parameter[n+1], "default")) {
		if (Channel < DefaultVoltage.size()) Double = DefaultVoltage[Channel];
		else Double = 0;
	  }

	  // Relative or absolute change?
	  if (isdigit(Parameter[n+1][0]) == 0) Voltages[Crate][Channel] = Crates[Crate]->GetVoltage(Channel) + Double;
	  else Voltages[Crate][Channel] = Double;
	} // Channels
  } // Loop over command argument

  // Ramp voltages and update DIM services
  for (unsigned int i=0; i<Voltages.size(); i++) {
	Errors += RampVoltages(i, Voltages[i]);
    Crates[i]->UpdateDIM();
  }

  // Error message only if not yet too many errors
  if (Errors > 0) {
	for (unsigned int i=0; i<Crates.size(); i++) {
	  if (Crates[i]->ErrorCount > MAX_ERR_COUNT) return;
	}
    Message(ERROR, "%d errors occurred from SetChannels()", Errors);
  }
}

//
// Load bias settings from file
//
void User::cmd_load() {

  char Buffer[MAX_COM_SIZE];
  int Errors = 0;
  unsigned int Channel;
  double Value;
  FILE *File;
  map<unsigned int, double> Voltages;

  // Open file
  if ((File=fopen(Parameter[1].c_str(), "r")) == NULL) {
    PrintMessage("Error: Could not open file '%s' (%s)\n", Parameter[1].c_str(), strerror(errno));
    return;
  }

  // Scan through file line by line
  while (fgets(Buffer, sizeof(Buffer), File) != NULL) {
    for (unsigned int Crate=0; Crate<Crates.size(); Crate++) if (Match(Crates[Crate]->Name, Buffer)) {

	  if (Crate != ActiveCrate) continue;
	  PrintMessage("Found bias settings for crate %s (#%d)\n\r", Crates[Crate]->Name, Crate);

	  Voltages.clear();
	  Channel = 0;
	  while (fscanf(File, "%lf", &Value)==1 && Channel<MAX_NUM_BOARDS*NUM_CHANNELS) {
		Voltages[Channel++] = Value;
	  }

	  // Ramp channels
	  Errors += RampVoltages(Crate, Voltages);

      // Update DIM service
	  Crates[Crate]->UpdateDIM();

	  if (ferror(File) != 0) {
		PrintMessage("Error reading DAC value from file, terminating. (%s)\n",strerror(errno));
    	return;
	  }
	  else PrintMessage("\nFinished updating board\n");
    } // Loop over boards
  } // while()
    	    
  if (Errors != 0) PrintMessage("Warning: %d error(s) occurred\n", Errors);
  if (fclose(File) != 0) PrintMessage("Error: Could not close file '%s'\n", Parameter[1].c_str());
}
	     
//
// Reset crates
//
void User::cmd_reset() {

  if (Crates[ActiveCrate]->SystemReset()) PrintMessage("System reset of crate %d\n", ActiveCrate);
  else PrintMessage("Error: Could not reset crate %d\n", ActiveCrate);
}

//
// Read channel
//
void User::cmd_gs() {

  double Voltage;

  if (!ConvertToDouble(Parameter[1], &Voltage)) {
    PrintMessage("Error: Wrong number format\n");
	return;
  }
  
  if (!Crates[ActiveCrate]->GlobalSet(Voltage)) {
	PrintMessage("Error: Could not global set crate %d\n", ActiveCrate);
  }  
}

//
// Save bias settings of all boards
//
void User::cmd_save() {

  FILE *File;
  time_t Time = time(NULL);

  if ((File = fopen(Parameter[1].c_str(), "w")) == NULL) {
    PrintMessage("Error: Could not open file '%s' (%s)\n", Parameter[1].c_str(), strerror(errno));
    return;
  }

  fprintf(File,"********** Bias settings of %s **********\n\n", ctime(&Time));

  for (unsigned int i=0; i<Crates.size(); i++) {
    fprintf(File, "%s\n\n", Crates[i]->Name);

    for (unsigned int j=0; j<MAX_NUM_BOARDS*NUM_CHANNELS; j++) fprintf(File,"%.3f ",Crates[i]->GetVoltage(j));
    fprintf(File, "\n");
  }

  if (fclose(File) != 0) {
    PrintMessage("Error: Could not close file '%s' (%s)\n", Parameter[1].c_str(), strerror(errno));  
  }
}

//
// Set operation mode
//
void User::cmd_mode() {

  if (Match(Parameter[1], "static")) Mode = mode_static;
  else {
    Mode = mode_dynamic;
	for (unsigned int i=0; i<Crates.size(); i++) {
	  Crates[i]->SetRefCurrent();
	}
  }
} 

//
// Print status
//
void User::cmd_status() {

  PrintMessage(" Number of crates:   %d\n", Crates.size());
  PrintMessage(" Active crate:       %d\n", ActiveCrate);  
  PrintMessage(" Update delay:       %d sec\n", min(atoi(GetConfig("UpdatePeriod").c_str()), 1));
  PrintMessage(" Time out:           %.2f sec\n", atof(GetConfig("TimeOut").c_str()));
  PrintMessage(" Max ramp speed      %.2f V/10 ms\n", atof(GetConfig("RampSpeed").c_str()));
  PrintMessage(" Voltage limit:      %.2f V\n", atof(GetConfig("VoltageLimit").c_str()));
  PrintMessage(" Minium reset delay: %u sec\n", atoi(GetConfig("MinResetPeriod").c_str()));
  
  for (unsigned int i=0; i<Crates.size(); i++) {

    PrintMessage(" CRATE %d (%s)\n   Wrap counter: %s (%d)  Reset: %s  Error count: %d %s\n ",
		i, Crates[i]->Name,	Crates[i]->WrapOK ? "ok":"error", Crates[i]->WrapCount, 
		Crates[i]->ResetHit ? "yes" : "no", Crates[i]->ErrorCount, Crates[i]->Disabled ? "(DISABLED)":"");

	// Read all channels
	if (!Crates[i]->ReadAll()) {
	  PrintMessage("Could not update status from crate %d\n", i);
	  continue;
	}

	if (Parameter.size() == 1) PrintMessage("Channel voltages (in V)");
    else if (Match(Parameter[1], "dac")) PrintMessage("Channel voltages (in DAC values)");
	else PrintMessage("Channel currents (in uA)");

    for (unsigned int j=0; j<MAX_NUM_BOARDS*NUM_CHANNELS; j++) {
	  if (j%12 == 0) PrintMessage("\n%3.1d:  ", j);
	  if (!Crates[i]->Present[j]) PrintMessage("  -   ");
      else if (Parameter.size() == 1) PrintMessage("%#5.2f ",Crates[i]->GetVoltage(j));
	  else if (Match(Parameter[1], "dac")) PrintMessage("%5d ", Crates[i]->GetDAC(j));
	  else PrintMessage("%#5.2f %s ", Crates[i]->GetCurrent(j), Crates[i]->OC[j] ? "OC":"");
    }
	PrintMessage("\n");
  }
} 

//
// Exit program (Signal makes readline return and sets ExitRequest)
//
void User::cmd_exit() {

  pthread_kill(MainThread, SIGTERM);
}

//
// Execute shell command
//
void User::cmd_shell() {

  if (system(Parameter[1].c_str()) == -1) PrintMessage("Error with system() call\n");
}


//
// Print message to screen and to DIM text service
//
void User::PrintMessage(const char *Format, ...) {

  static char Error[] = "vasprintf() failed in PrintMessage()";
  char *Text;

  // Evaluate arguments    
  va_list ArgumentPointer;
  va_start(ArgumentPointer, Format);
  if (vasprintf(&Text, Format, ArgumentPointer) == -1) Text = Error;
  va_end(ArgumentPointer);

  if (strlen(Text) == 0) return;
 
  // Print to console
  printf("%s", Text);
  fflush(stdout);
  if (Text[strlen(Text)-1] == '\n') rl_on_new_line(); // New prompt

  // Send to DIM text service
  ConsoleOut->updateService(Text); 

  // Free old text
  if (ConsoleText != Error) free(ConsoleText);
  ConsoleText = Text; 
}


// Ramp to new voltage with given maximum step
// No ramping when decreasing voltage
unsigned int User::RampVoltages(int Crate, map<unsigned int, double> Voltages) {

  map<unsigned int, double> Target;
  int Errors = 0;
  double MaxDiff = atof(GetConfig("RampSpeed").c_str());

  // Ramp until all channels at desired value
  while (!Voltages.empty() && Errors < MAX_ERR_COUNT) {
    // Remove channels already at target (check for DAC, not for floating-point voltage)
	for (map<unsigned int, double>::iterator it = Voltages.begin(); it != Voltages.end(); ++it) {
	  if (fabs(Crates[Crate]->GetVoltage(it->first)-it->second) < 0.001) Voltages.erase(it);
	}
	
	// Limit voltage changes to fMaxDiff
	Target = Voltages;
	for (map<unsigned int, double>::iterator it = Target.begin(); it != Target.end(); ++it) {
	  if (Crates[Crate]->GetVoltage(it->first) + MaxDiff < it->second) {
		it->second = Crates[Crate]->GetVoltage(it->first) + MaxDiff;
	  }
	}	
	
	// Set channels to next target and wait 10 ms
	if (!Crates[Crate]->SetChannels(Target)) Errors++;
	usleep(10000);
  }
  
  return Errors;
}


//
// Check status
//
void User::Monitor() {

  int Ret;

  while (!ExitRequest) {

  	// Wait (this is the only allowed cancelation point)
	if ((Ret=pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL)) != 0) {
	  Message(FATAL, "pthread_setcancelstate() failed (%s)", strerror(Ret));
	}
	sleep(min(atoi(GetConfig("UpdatePeriod").c_str()), 1));
	if ((Ret=pthread_setcancelstate(PTHREAD_CANCEL_DISABLE, NULL)) != 0) {
	  Message(FATAL, "pthread_setcancelstate() failed (%s)", strerror(Ret));
	}

	// Check all crates
	for (unsigned int i=0; i<Crates.size(); i++) {
	  // Crate disabled because of too many errors?
      if (Crates[i]->Disabled) continue;

	  // Read all channels
      if (!Crates[i]->ReadAll()) {
    	Message(ERROR, "Monitor thread could not read status from crate %d", i);
		continue;
      }

	  // Check if crate push button was hit
      if (Crates[i]->ResetHit) {
    	Message(INFO, "Manual reset of crate %d, setting voltages to zero and issuing system reset", i);
		if (!Crates[i]->GlobalSet(0)) Message(ERROR, "Global set to zero voltage of crate %d failed", i);
		if (!Crates[i]->SystemReset()) Message(ERROR, "System reset of crate %d failed", i);
      }

	  // Check for overcurrent and set voltage to zero if occurred
	  map<unsigned int, double> Voltages;
	  
      for (unsigned int j=0; j<MAX_NUM_BOARDS*NUM_CHANNELS; j++) {
    	if (Crates[i]->OC[j]) {
		  Message(WARN, "Overcurrent on crate %d, board %d, channel %d, setting voltage to zero", i, j/NUM_CHANNELS, j%NUM_CHANNELS);
		  Voltages[j] = 0;
    	}
      }
	  if (!Crates[i]->SetChannels(Voltages)) Message(ERROR, "Error when setting voltages of crate %d", i);
	  
	  if (Mode == mode_dynamic) Crates[i]->AdaptVoltages();
	} // for
  } // while
}

// Call monitor loop inside class
void User::LaunchMonitor(User *m) {

  m->Monitor();
}


//
// Check if two strings match (min 1 character must match)
//
bool User::Match(string str, const char *cmd) {

  return strncasecmp(str.c_str(),cmd,strlen(str.c_str())==0 ? 1:strlen(str.c_str())) ? false:true;
}

//
// Conversion function from string to double or int
//
// Return false if conversion did not stop on whitespace or EOL character
bool User::ConvertToDouble(string String, double *Result) {

  char *EndPointer;
  
  *Result = strtod(String.c_str(), &EndPointer);
  if(!isspace(*EndPointer) && *EndPointer!='\0') return false;
  return true;
}

bool User::ConvertToInt(string String, int *Result) {

  char *EndPointer;
  
  *Result = (int) strtol(String.c_str(), &EndPointer, 0);
  if(!isspace(*EndPointer) && *EndPointer!='\0') return false;
  return true;
}

//
// Interprets a range
//
bool User::ConvertToRange(string String, struct User::Range &R) {

  int N=0, M=0; // Init to avoid compiler warning

  // Full range
  if (Match(String, "all")) return true;

  // Single number
  if (ConvertToInt(String, &N)) {
	if (N>= R.Min && N<=R.Max) {
	  R.Max = R.Min = N;
	  return true;
	}
	return false;
  }
  
  // Range a-b
  vector<string> V = EvidenceServer::Tokenize(String, "-");
  if (V.size()==2 && ConvertToInt(V[0], &N) && ConvertToInt(V[1], &M) && N>=R.Min && M<=R.Max) {
	R.Min = N;
	R.Max = M;
	return true;
  }
  
  return false;
}
