----------------------------------------------------------------------------------
-- Company:        ETH Zurich, Institute for Particle Physics
-- Engineer:       P. Vogler, Q. Weitzel
-- 
-- Create Date:    08 December 2010
-- Design Name:    
-- Module Name:    FTM_top - Behavioral 
-- Project Name: 
-- Target Devices: 
-- Tool versions: 
-- Description:    Top level entity for FTM firmware
--                                              
--
-- Dependencies: 
--
-- Revision: 
-- Revision 0.01 - File Created
-- Additional Comments: 
--
----------------------------------------------------------------------------------

library IEEE;
use IEEE.STD_LOGIC_1164.ALL;
use IEEE.STD_LOGIC_ARITH.ALL;
use IEEE.STD_LOGIC_UNSIGNED.ALL;

library ftm_definitions;
USE ftm_definitions.ftm_array_types.all;
USE ftm_definitions.ftm_constants.all;

---- Uncomment the following library declaration if instantiating
---- any Xilinx primitives in this code.
library UNISIM;
use UNISIM.VComponents.all;


entity FTM_top is
  port(
    
    -- Clock
    clk : IN  STD_LOGIC;  -- external clock from oscillator U47
    
    -- connection to the WIZnet W5300 ethernet controller
    -- on IO-Bank 1
    -------------------------------------------------------------------------------
    -- W5300 data bus
    W_D  : inout STD_LOGIC_VECTOR(15 downto 0);  -- 16-bit data bus to W5300	

    -- W5300 address bus
    W_A  : out STD_LOGIC_VECTOR(9 downto 0);   -- there is no real net W_A0 because
                                               -- the W5300 is operated in the 
                                               -- 16-bit mode
                                               -- -> W_A<0> assigned to unconnected pin

    -- W5300 control signals
    -- the signals W_INT, W_RD, W_WR and W_RES also go to testpoints T17
    -- W_CS is also routed to testpoint JP7
    W_CS   : out  STD_LOGIC := '1';               -- W5300 chip select
    W_INT  : IN   STD_LOGIC;                      -- interrupt
    W_RD   : out  STD_LOGIC := '1';               -- read
    W_WR   : out  STD_LOGIC := '1';               -- write
    W_RES  : out  STD_LOGIC := '1';               -- reset W5300 chip

    -- W5300 buffer ready indicator
    -- W_BRDY : in STD_LOGIC_VECTOR(3 downto 0); 

    -- testpoints (T18) associated with the W5300 on IO-bank 1
    -- W_T    : inout STD_LOGIC_VECTOR(3 downto 0);  
 

    -- SPI Interface
    -- connection to the EEPROM U36 (AL25L016M) and 
    -- temperature sensors U45, U46, U48 and U49 (all MAX6662)
    -- on IO-Bank 1
    -------------------------------------------------------------------------------
    -- S_CLK  : out  STD_LOGIC;     -- SPI clock

    -- EEPROM
    --   MOSI   : out  STD_LOGIC;     -- master out slave in
    --   MISO   : in   STD_LOGIC;     -- master in slave out
    --   EE_CS  : out  STD_LOGIC;     -- EEPROM chip select

    -- temperature sensors U45, U46, U48 and U49
    -- SIO    : inout  STD_LOGIC;          -- serial IO
    -- TS_CS  : out STD_LOGIC_VECTOR(3 downto 0);     -- temperature sensors chip select

 
    -- Trigger primitives inputs
    -- on IO-Bank 2
    -------------------------------------------------------------------------------
    Trig_Prim_A  : in STD_LOGIC_VECTOR(9 downto 0);  -- crate 0
    Trig_Prim_B  : in STD_LOGIC_VECTOR(9 downto 0);  -- crate 1
    Trig_Prim_C  : in STD_LOGIC_VECTOR(9 downto 0);  -- crate 2
    Trig_Prim_D  : in STD_LOGIC_VECTOR(9 downto 0);  -- crate 3

  
    -- NIM inputs
    ------------------------------------------------------------------------------
    -- on IO-Bank 3  
    ext_Trig  : in  STD_LOGIC_VECTOR(2 downto 1);      -- external trigger input
    Veto      : in  STD_LOGIC;                         -- trigger veto input
    -- NIM_In    : in  STD_LOGIC_VECTOR(2 downto 0);      -- auxiliary inputs

    -- on IO-Bank 0
    -- alternative external clock input for FPGA
    -- NIM_In3_GCLK  : in  STD_LOGIC;      -- input with global clock buffer available 

   
    -- LEDs on IO-Banks 0 and 3
    -------------------------------------------------------------------------------
    LED_red  : out STD_LOGIC_VECTOR(3 downto 0);  -- red
    LED_ye   : out STD_LOGIC_VECTOR(1 downto 0);    -- yellow
    LED_gn   : out STD_LOGIC_VECTOR(1 downto 0);    -- green

    
    -- Clock conditioner LMK03000
    -- on IO-Bank 3
    -------------------------------------------------------------------------------
    CLK_Clk_Cond  : out STD_LOGIC;  -- MICROWIRE interface serial clock
    LE_Clk_Cond   : out STD_LOGIC;  -- MICROWIRE interface latch enable   
    DATA_Clk_Cond : out STD_LOGIC;  -- MICROWIRE interface data
   
    SYNC_Clk_Cond : out STD_LOGIC;  -- global clock synchronization
    LD_Clk_Cond   : in STD_LOGIC;   -- lock detect                  

    
    -- various RS-485 Interfaces
    -- on IO-Bank 3
    -------------------------------------------------------------------------------
    -- Bus 1: FTU slow control   
    Bus1_Tx_En    : out STD_LOGIC;  -- bus 1: transmitter enable                                 
    Bus1_Rx_En    : out STD_LOGIC;  -- bus 1: receiver enable

    Bus1_RxD_0    : in STD_LOGIC;   -- crate 0
    Bus1_TxD_0    : out STD_LOGIC;

    Bus1_RxD_1    : in STD_LOGIC;   -- crate 1
    Bus1_TxD_1    : out STD_LOGIC;

    Bus1_RxD_2    : in STD_LOGIC;   -- crate 2
    Bus1_TxD_2    : out STD_LOGIC;

    Bus1_RxD_3    : in STD_LOGIC;   -- crate 3
    Bus1_TxD_3    : out STD_LOGIC;


    -- Bus 2: Trigger-ID to FAD boards
    Bus2_Tx_En    : out STD_LOGIC;  -- bus 2: transmitter enable                                 
    Bus2_Rx_En    : out STD_LOGIC;  -- bus 2: receiver enable

    Bus2_RxD_0    : in STD_LOGIC;   -- crate 0
    Bus2_TxD_0    : out STD_LOGIC;

    Bus2_RxD_1    : in STD_LOGIC;   -- crate 1
    Bus2_TxD_1    : out STD_LOGIC;

    Bus2_RxD_2    : in STD_LOGIC;   -- crate 2
    Bus2_TxD_2    : out STD_LOGIC;

    Bus2_RxD_3    : in STD_LOGIC;   -- crate 3
    Bus2_TxD_3    : out STD_LOGIC;  
   

    -- auxiliary access
    --   Aux_Rx_D      : in STD_LOGIC;
    --   Aux_Tx_D      : out STD_LOGIC;
    --   Aux_Rx_En     : out STD_LOGIC;   --   Rx- and Tx enable 
    --   Aux_Tx_En     : out STD_LOGIC;   --   also for auxiliary Trigger-ID
    		      	      			    	   	  

    -- auxiliary Trigger-ID (i.e. to send the Trigger-ID to the counting hut/house/container)
    --   TrID_Rx_D     : in STD_LOGIC;
    --   TrID_Tx_D     : out STD_LOGIC;


    -- Crate-Resets
    -- on IO-Bank 3
    -------------------------------------------------------------------------------
    Crate_Res0   : out STD_LOGIC;
    Crate_Res1   : out STD_LOGIC;
    Crate_Res2   : out STD_LOGIC;
    Crate_Res3   : out STD_LOGIC;


    -- Busy signals from the FAD boards
    -- on IO-Bank 3
    -------------------------------------------------------------------------------
    Busy0     : in STD_LOGIC;
    Busy1     : in STD_LOGIC;
    Busy2     : in STD_LOGIC;
    Busy3     : in STD_LOGIC;


    -- NIM outputs
    -- on IO-Bank 0
    -- LVDS output at the FPGA followed by LVDS to NIM conversion stage
    -------------------------------------------------------------------------------
    -- calibration
    --   Cal_NIM1_p  : out STD_LOGIC;     --  Cal_NIM1+ 
    --   Cal_NIM1_n  : out STD_LOGIC;     --  Cal_NIM1-
    --   Cal_NIM2_p  : out STD_LOGIC;     --  Cal_NIM2+  
    --   Cal_NIM2_n  : out STD_LOGIC;     --  Cal_NIM2- 

    -- auxiliarry / spare NIM outputs
    --   NIM_Out0_p  : out STD_LOGIC;   -- NIM_Out0+
    --   NIM_Out0_n  : out STD_LOGIC;   -- NIM_Out0-
    --   NIM_Out1_p  : out STD_LOGIC;   -- NIM_Out1+
    --   NIM_Out1_n  : out STD_LOGIC;   -- NIM_Out1-

  
    -- fast control signal outputs
    -- LVDS output at the FPGA followed by LVDS to NIM conversion stage
    -------------------------------------------------------------------------------
    RES_p      : out STD_LOGIC;   -- RES+  Reset
    RES_n      : out STD_LOGIC;   -- RES-  IO-Bank 0

    TRG_p      : out STD_LOGIC;   -- TRG+  Trigger
    TRG_n      : out STD_LOGIC;   -- TRG-  IO-Bank 0

    TIM_Run_p  : out STD_LOGIC;   -- TIM_Run+  Time Marker
    TIM_Run_n  : out STD_LOGIC;   -- TIM_Run-  IO-Bank 2
    TIM_Sel    : out STD_LOGIC;   -- Time Marker selector on IO-Bank 2
                                                    
    -- CLD_FPGA   : in STD_LOGIC;    -- DRS-Clock feedback into FPGA


    -- LVDS calibration outputs
    -- on IO-Bank 0
    -------------------------------------------------------------------------------
    -- to connector J13
    -- for light pulsar in the mirror dish
    Cal_0_p    : out STD_LOGIC;  
    Cal_0_n    : out STD_LOGIC;
    Cal_1_p    : out STD_LOGIC;
    Cal_1_n    : out STD_LOGIC;
    Cal_2_p    : out STD_LOGIC;
    Cal_2_n    : out STD_LOGIC;
    Cal_3_p    : out STD_LOGIC;
    Cal_3_n    : out STD_LOGIC;

    -- to connector J12
    -- for light pulsar inside shutter
    Cal_4_p    : out STD_LOGIC;
    Cal_4_n    : out STD_LOGIC;
    Cal_5_p    : out STD_LOGIC;
    Cal_5_n    : out STD_LOGIC;
    Cal_6_p    : out STD_LOGIC;
    Cal_6_n    : out STD_LOGIC; 
    Cal_7_p    : out STD_LOGIC;
    Cal_7_n    : out STD_LOGIC;  


    -- Testpoints
    -------------------------------------------------------------------------------
    TP    : inout STD_LOGIC_VECTOR(32 downto 0)
    --   TP_in    : in STD_LOGIC_VECTOR(34 downto 33);    -- input only

    -- Board ID - inputs 
    -- local board-ID "solder programmable"
    -- all on 'input only' pins
    -------------------------------------------------------------------------------
    -- brd_id : in STD_LOGIC_VECTOR(7 downto 0)    -- input only		    

  );
end FTM_top;

architecture Behavioral of FTM_top is

  signal cc_R0_sig              : std_logic_vector(31 DOWNTO 0);
  signal cc_R1_sig              : std_logic_vector(31 DOWNTO 0);
  signal cc_R11_sig             : std_logic_vector(31 DOWNTO 0);
  signal cc_R13_sig             : std_logic_vector(31 DOWNTO 0);
  signal cc_R14_sig             : std_logic_vector(31 DOWNTO 0);
  signal cc_R15_sig             : std_logic_vector(31 DOWNTO 0);
  signal cc_R8_sig              : std_logic_vector(31 DOWNTO 0);
  signal cc_R9_sig              : std_logic_vector(31 DOWNTO 0);
  signal coin_n_c_sig           : std_logic_vector(15 DOWNTO 0);
  signal coin_n_p_sig           : std_logic_vector(15 DOWNTO 0);
  signal dead_time_sig          : std_logic_vector(15 DOWNTO 0);
  signal ftu_active_cr0_sig     : std_logic_vector(15 DOWNTO 0);
  signal ftu_active_cr1_sig     : std_logic_vector(15 DOWNTO 0);
  signal ftu_active_cr2_sig     : std_logic_vector(15 DOWNTO 0);
  signal ftu_active_cr3_sig     : std_logic_vector(15 DOWNTO 0);
  signal general_settings_sig   : std_logic_vector(15 DOWNTO 0);
  signal lp1_amplitude_sig      : std_logic_vector(15 DOWNTO 0);
  signal lp1_delay_sig          : std_logic_vector(15 DOWNTO 0);
  signal lp2_amplitude_sig      : std_logic_vector(15 DOWNTO 0);
  signal lp2_delay_sig          : std_logic_vector(15 DOWNTO 0);
  signal lp_pt_freq_sig         : std_logic_vector(15 DOWNTO 0);
  signal lp_pt_ratio_sig        : std_logic_vector(15 DOWNTO 0);
  signal timemarker_delay_sig   : std_logic_vector(15 DOWNTO 0);
  signal trigger_delay_sig      : std_logic_vector(15 DOWNTO 0);
  signal sd_addr_ftu_sig        : std_logic_vector(11 DOWNTO 0);
  signal sd_busy_sig            : std_logic;
  signal sd_data_out_ftu_sig    : std_logic_vector(15 DOWNTO 0) := (others => '0');
  signal sd_read_ftu_sig        : std_logic;
  signal sd_ready_sig           : std_logic;
  signal sd_started_ftu_sig     : std_logic := '0';
  signal new_config_sig         : std_logic := '0';
  signal config_started_sig     : std_logic := '0';
  signal config_start_eth_sig   : std_logic := '0';
  signal config_started_eth_sig : std_logic := '0';
  signal config_ready_eth_sig   : std_logic := '0';
  signal config_started_ack_sig : std_logic := '0';
  signal ping_ftu_start_sig     : std_logic := '0';
  signal ping_ftu_started_sig   : std_logic := '0';
  signal ping_ftu_ready_sig     : std_logic := '0';
  signal config_start_ftu_sig   : std_logic := '0';
  signal config_started_ftu_sig : std_logic := '0';
  signal config_ready_ftu_sig   : std_logic := '0';
  signal rates_ftu_start_sig    : std_logic := '0';
  signal rates_ftu_started_sig  : std_logic := '0';
  signal rates_ftu_ready_sig    : std_logic := '0';
  signal fl_busy_sig            : std_logic;
  signal fl_ready_sig           : std_logic;
  signal fl_write_sig           : std_logic := '0';
  signal fl_started_ftu_sig     : std_logic := '0';
  signal fl_addr_sig            : std_logic_vector(11 DOWNTO 0) := (others => '0');
  signal fl_data_sig            : std_logic_vector(15 DOWNTO 0) := (others => '0');
  signal ping_ftu_start_ftu_sig : std_logic := '0';
  signal ping_ftu_started1_sig  : std_logic := '0';
  signal ping_ftu_ready1_sig    : std_logic := '0';
  signal coin_win_c_sig         : std_logic_vector(15 DOWNTO 0) := (others => '0');
  signal coin_win_p_sig         : std_logic_vector(15 DOWNTO 0) := (others => '0');
  --new or changed stuff
  signal dd_data_sig                : std_logic_vector(15 DOWNTO 0) := (others => '0');
  signal dd_addr_sig                : std_logic_vector(11 DOWNTO 0) := (others => '0');  
  signal dd_block_start_ftu_sig     : std_logic := '0';
  signal dd_block_start_ack_ftu_sig : std_logic := '0';
  signal dd_block_ready_ftu_sig     : std_logic := '0';
  signal dd_busy_sig                : std_logic;
  signal dd_write_sig               : std_logic := '0';
  signal dd_started_ftu_sig         : std_logic := '0';
  signal dd_ready_sig               : std_logic;
  signal dd_send_sig                : std_logic := '1';
  signal dd_send_ack_sig            : std_logic := '1';
  signal dd_send_ready_sig          : std_logic := '1';
  --very new stuff
  SIGNAL ftu_error_send_ack_sig     : std_logic                      := '1';
  SIGNAL ftu_error_send_ready_sig   : std_logic                      := '1';
  SIGNAL ftu_error_calls_sig        : std_logic_vector(15 DOWNTO 0)  := X"0000";
  SIGNAL ftu_error_data_sig         : std_logic_vector(223 DOWNTO 0) := (others => '0');
  SIGNAL ftu_error_send_sig         : std_logic                      := '0';
  signal prescaling_FTU01_sig       : std_logic_vector (15 DOWNTO 0);
  signal trigger_counter_sig        : std_logic_vector (31 DOWNTO 0);
  signal trigger_counter_read_sig   : std_logic;
  signal trigger_counter_valid_sig  : std_logic;
  
  signal config_start_cc_sig   : std_logic;  -- initialized in central control
  signal config_started_cc_sig : std_logic := '0';
  signal config_ready_cc_sig   : std_logic := '0';
  signal cc_locked_sig         : std_logic := '0';

  signal config_start_lp_sig   : std_logic;  -- initialized in central control
  signal config_started_lp_sig : std_logic;  -- initialized in light pulser interface
  signal config_ready_lp_sig   : std_logic;  -- initialized in light pulser interface
  
  signal config_trigger_sig : std_logic;
  signal config_trigger_done_sig : std_logic;
  
  signal clk_buf_sig     : std_logic;
  signal clk_1M_sig      : STD_LOGIC;         -- generated from 50M clock by divider
  signal clk_50M_sig     : STD_LOGIC;         -- generated by internal DCM
  signal clk_250M_sig    : STD_LOGIC;         -- generated by internal DCM
  signal clk_250M_ps_sig : STD_LOGIC;         -- generated by internal DCM
  signal clk_ready_sig   : STD_LOGIC := '0';  -- set high by FTM_clk_gen when DCMs have locked

  signal trigger_ID_ready_sig : std_logic;  -- initialized in trigger manager
  signal trigger_ID_sig       : std_logic_vector(55 downto 0);  -- initialized in trigger manager
  signal trigger_ID_read_sig  : std_logic;  -- initialized in FTM_fad_broadcast

  signal trigger_active_sig : std_logic;  -- initialized in trigger manager
  
  signal reset_sig : STD_LOGIC := '0';  -- initialize to 0 on power-up

  signal trigger_signal_sig : std_logic := '0';
  signal TIM_signal_sig : std_logic := '0';

  --signals for FPGA DNA identifier
  signal dna_sig : STD_LOGIC_VECTOR(63 downto 0);  -- initialized in FTM_dna_gen
  signal dna_start_sig : STD_LOGIC;  -- initialized in FTM_central_control
  signal dna_ready_sig : STD_LOGIC;  -- initialized in FTM_dna_gen
  
  signal led_sig : std_logic_vector(7 downto 0) := (others => '0');

  signal get_ot_counter_sig : std_logic;
  signal get_ot_counter_started_sig : std_logic;
  signal get_ot_counter_ready_sig : std_logic;
  signal on_time_counter_sig : std_logic_vector(47 downto 0);

  signal get_ts_counter_sig : std_logic;
  signal get_ts_counter_started_sig : std_logic;
  signal get_ts_counter_ready_sig : std_logic;
  signal timestamp_counter_sig : std_logic_vector(47 downto 0);

  signal crate_reset_sig            : std_logic;
  signal crate_reset_ack_sig        : std_logic;
  signal crate_reset_param_sig      : std_logic_vector (15 DOWNTO 0);
  signal start_run_sig              : std_logic;
  signal start_run_ack_sig          : std_logic;
  signal stop_run_sig               : std_logic;
  signal stop_run_ack_sig           : std_logic;
  signal current_cc_state_sig       : std_logic_vector (15 DOWNTO 0);
  signal cc_state_test_sig          : std_logic_vector ( 7 downto 0);
  signal start_run_param_sig        : std_logic_vector (15 DOWNTO 0);
  signal start_run_num_events_sig   : std_logic_vector (31 DOWNTO 0);

  signal trigger_start_sig : std_logic;
  signal trigger_stop_sig : std_logic;

  signal enable_ID_sending_sig : std_logic;
  signal reset_timer_sig : std_logic;  -- initialized in FTM_central_control

  signal crate_res0_sig : std_logic;  -- initialized in FTM_central_control
  signal crate_res1_sig : std_logic;  -- initialized in FTM_central_control
  signal crate_res2_sig : std_logic;  -- initialized in FTM_central_control
  signal crate_res3_sig : std_logic;  -- initialized in FTM_central_control

  signal LP1_pulse_sig : std_logic := '0';
  signal LP2_pulse_sig : std_logic := '0';

  signal new_config_ftu_sig       : std_logic;  -- initialized in ethernet_modul
  signal new_config_ftu_ack_sig   : std_logic;  -- initialized in FTM_central_control
  signal new_config_ftu_param_sig : std_logic_vector (15 DOWNTO 0);  -- initialized in ethernet_modul

  signal config_single_FTU_sig : std_logic;  -- initialized in FTM_central_control
  signal config_single_FTU_started_sig : std_logic;  -- initialized in FTM_ftu_control
  signal config_single_FTU_done_sig : std_logic;  -- initialized in FTM_ftu_control
  
--  component FTM_clk_gen
--    port(
--      clk        : IN  STD_LOGIC;
--      rst        : IN  STD_LOGIC;
--      clk_1      : OUT STD_LOGIC;
--      clk_50     : OUT STD_LOGIC;
--      clk_250    : OUT STD_LOGIC;
--      clk_250_ps : OUT STD_LOGIC;
--      ready      : OUT STD_LOGIC
--    );
--  end component;

  component FTM_clk_gen_2
    port(
      clk        : IN  STD_LOGIC;
      rst        : IN  STD_LOGIC;
      clk_1      : OUT STD_LOGIC;
      clk_50     : OUT STD_LOGIC;
      clk_250    : OUT STD_LOGIC;
      clk_250_ps : OUT STD_LOGIC;
      ready      : OUT STD_LOGIC
    );
  end component;

  component FTM_dna_gen
    port(
      clk   : IN  STD_LOGIC;
      start : IN  STD_LOGIC;
      dna   : OUT STD_LOGIC_VECTOR(63 downto 0);
      ready : OUT STD_LOGIC
    );
  end component;
  
  component trigger_manager
    port(
      --clocks
      clk_50MHz           : in  std_logic;
      clk_250MHz          : in  std_logic;
      clk_250MHz_180      : in  std_logic;
      --trigger primitives from FTUs
      trig_prim_0         : in  std_logic_vector(9 downto 0);  --crate 0
      trig_prim_1         : in  std_logic_vector(9 downto 0);  --crate 1
      trig_prim_2         : in  std_logic_vector(9 downto 0);  --crate 2
      trig_prim_3         : in  std_logic_vector(9 downto 0);  --crate 3
      --external signals
      ext_trig_1          : in  std_logic;
      ext_trig_2          : in  std_logic;
      ext_veto            : in  std_logic;
      FAD_busy_0          : in  std_logic;  --crate 0
      FAD_busy_1          : in  std_logic;  --crate 1
      FAD_busy_2          : in  std_logic;  --crate 2
      FAD_busy_3          : in  std_logic;  --crate 3
      --control signals from e.g. main control
      start_run           : in  std_logic;  --enable trigger output
      stop_run            : in  std_logic;  --disable trigger output
      new_config          : in  std_logic;
      --settings register (see FTM Firmware Specifications)
      general_settings    : in  std_logic_vector(15 downto 0);
      LP_and_PED_freq     : in  std_logic_vector(15 downto 0);
      LP1_LP2_PED_ratio   : in  std_logic_vector(15 downto 0);
      maj_coinc_n_phys    : in  std_logic_vector(15 downto 0);
      maj_coinc_n_calib   : in  std_logic_vector(15 downto 0);
      trigger_delay       : in  std_logic_vector(15 downto 0);
      TIM_delay           : in  std_logic_vector(15 downto 0);
      dead_time           : in  std_logic_vector(15 downto 0);
      coinc_window_phys   : in  std_logic_vector(15 downto 0);
      coinc_window_calib  : in  std_logic_vector(15 downto 0);
      active_FTU_list_0   : in  std_logic_vector(15 downto 0);
      active_FTU_list_1   : in  std_logic_vector(15 downto 0);
      active_FTU_list_2   : in  std_logic_vector(15 downto 0);
      active_FTU_list_3   : in  std_logic_vector(15 downto 0);
      --control signals or information for other entities
      trigger_ID_read     : in  std_logic;
      trig_cnt_copy_read  : in  std_logic;
      trigger_ID_ready    : out std_logic;
      trigger_ID          : out std_logic_vector(55 downto 0);
      trig_cnt_copy       : out std_logic_vector(31 downto 0);  --counter reading
      trig_cnt_copy_valid : out std_logic;  --trigger counter reading is valid
      trigger_active      : out std_logic;  --phys triggers are enabled/active
      config_done         : out std_logic;
      LP1_pulse           : out std_logic;  --send start signal to light pulser 1
      LP2_pulse           : out std_logic;  --send start signal to light pulser 2
      --trigger and time marker output signals to FADs
      trigger_signal      : out std_logic;
      TIM_signal          : out std_logic
    );
  end component;
  
  component Clock_cond_interface is
    port(
      clk               : IN  STD_LOGIC;
      CLK_Clk_Cond      : out STD_LOGIC;
      LE_Clk_Cond       : out STD_LOGIC;
      DATA_Clk_Cond     : out STD_LOGIC;
      SYNC_Clk_Cond     : out STD_LOGIC;
      LD_Clk_Cond       : in STD_LOGIC;
      TIM_Sel           : out STD_LOGIC;
      cc_R0             : in std_logic_vector (31 downto 0) := (others => '0');
      cc_R1             : in std_logic_vector (31 downto 0) := (others => '0');
      cc_R8             : in std_logic_vector (31 downto 0) := (others => '0');
      cc_R9             : in std_logic_vector (31 downto 0) := (others => '0');
      cc_R11            : in std_logic_vector (31 downto 0) := (others => '0');
      cc_R13            : in std_logic_vector (31 downto 0) := (others => '0');
      cc_R14            : in std_logic_vector (31 downto 0) := (others => '0');
      cc_R15            : in std_logic_vector (31 downto 0) := (others => '0');
      start_config      : in STD_LOGIC;
      config_started    : out STD_LOGIC;
      config_done       : out STD_LOGIC;
      locked            : out STD_LOGIC;
      timemarker_select : in STD_LOGIC
    );
  end component;
  
  component FTM_central_control
    port(
      clk                  : IN  std_logic;
      clk_ready            : in  std_logic;
      clk_scaler           : IN  std_logic;
      new_config           : IN  std_logic;
      config_started       : OUT std_logic := '0';
      config_started_ack   : IN  std_logic;
      config_start_eth     : OUT std_logic := '0';
      config_started_eth   : IN  std_logic;
      config_ready_eth     : IN  std_logic;
      config_start_ftu     : OUT std_logic := '0';
      config_started_ftu   : IN  std_logic;
      config_ready_ftu     : IN  std_logic;
      ping_ftu_start       : IN  std_logic;
      ping_ftu_started     : OUT std_logic := '0';
      ping_ftu_ready       : OUT std_logic := '0';
      ping_ftu_start_ftu   : OUT std_logic := '0';
      ping_ftu_started_ftu : IN  std_logic;
      ping_ftu_ready_ftu   : IN  std_logic;
      rates_ftu            : OUT std_logic := '0';
      rates_started_ftu    : IN  std_logic;
      rates_ready_ftu      : IN  std_logic;
      prescaling_FTU01     : IN  std_logic_vector(7 downto 0);
      dd_send              : OUT std_logic := '0';
      dd_send_ack          : IN  std_logic;
      dd_send_ready        : IN  std_logic;
      dd_block_ready_ftu     : out std_logic := '0';
      dd_block_start_ack_ftu : in  std_logic;
      dd_block_start_ftu     : out std_logic := '0';
      config_start_cc      : out std_logic := '0';
      config_started_cc    : in  std_logic;
      config_ready_cc      : in  std_logic;
      cc_locked            : in  std_logic;
      config_start_lp      : out std_logic := '0';
      config_started_lp    : in  std_logic;
      config_ready_lp      : in  std_logic;
      config_trigger       : out std_logic;
      config_trigger_done  : in  std_logic;
      dna_start            : out std_logic;
      dna_ready            : in  std_logic;
      crate_reset            : IN  std_logic;
      crate_reset_ack        : OUT std_logic;
      crate_reset_param      : IN  std_logic_vector (15 DOWNTO 0);
      start_run              : IN  std_logic;
      start_run_ack          : OUT std_logic;
      stop_run               : IN  std_logic;
      stop_run_ack           : OUT std_logic;
      current_cc_state       : OUT std_logic_vector (15 DOWNTO 0);
      cc_state_test          : OUT std_logic_vector ( 7 downto 0);
      start_run_param        : IN  std_logic_vector (15 DOWNTO 0);
      start_run_num_events   : IN  std_logic_vector (31 DOWNTO 0);
      trigger_start : out std_logic;
      trigger_stop : out std_logic;
      enable_ID_sending : out std_logic;
      reset_timer : out std_logic;
      crate_res_0 : out std_logic;
      crate_res_1 : out std_logic;
      crate_res_2 : out std_logic;
      crate_res_3 : out std_logic;
      new_config_ftu     : in std_logic;
      new_config_ftu_ack : out std_logic := '0';
      config_single_FTU : out std_logic := '0';
      config_single_FTU_started : in std_logic;
      config_single_FTU_done : in std_logic
    );
  end component;

  component FTM_ftu_control
    port(
      clk_50MHz           : in  std_logic;
      rx_en               : out STD_LOGIC;
      tx_en               : out STD_LOGIC; 
      rx_d_0              : in  STD_LOGIC;
      tx_d_0              : out STD_LOGIC;
      rx_d_1              : in  STD_LOGIC;
      tx_d_1              : out STD_LOGIC;
      rx_d_2              : in  STD_LOGIC;
      tx_d_2              : out STD_LOGIC;
      rx_d_3              : in  STD_LOGIC;
      tx_d_3              : out STD_LOGIC;
      new_config          : in std_logic;
      ping_all            : in std_logic;
      read_rates          : in std_logic;
      config_single_FTU   : in std_logic;
      read_rates_started  : out std_logic := '0';
      read_rates_done     : out std_logic := '0';
      new_config_started  : out std_logic := '0';
      new_config_done     : out std_logic := '0';
      ping_all_started    : out std_logic := '0';
      ping_all_done       : out std_logic := '0';
      config_single_FTU_started : out std_logic := '0';
      config_single_FTU_done    : out std_logic := '0';
      ftu_active_cr0      :  in std_logic_vector (15 downto 0);
      ftu_active_cr1      :  in std_logic_vector (15 downto 0);
      ftu_active_cr2      :  in std_logic_vector (15 downto 0);
      ftu_active_cr3      :  in std_logic_vector (15 downto 0);
      config_single_FTU_param : in std_logic_vector (15 DOWNTO 0);
      ftu_error_calls     : out std_logic_vector (15 DOWNTO 0) := (others => '0');
      ftu_error_data      : out std_logic_vector ((FTU_RS485_BLOCK_WIDTH - 1) downto 0) := (others => '0');
      ftu_error_send      : out std_logic := '0';
      ftu_error_send_ack   :  in std_logic;
      ftu_error_send_ready :  in std_logic;
      static_RAM_busy     :  in std_logic;
      static_RAM_started  :  in std_logic;
      static_RAM_ready    :  in std_logic;
      data_static_RAM     :  in std_logic_vector(15 downto 0) := (others => '0');
      read_static_RAM     : out std_logic := '0';
      addr_static_RAM     : out std_logic_vector(11 downto 0) := (others => '0');
      dynamic_RAM_busy    :  in std_logic;
      dynamic_RAM_started :  in std_logic;
      dynamic_RAM_ready   :  in std_logic;
      data_dynamic_RAM    : out std_logic_vector(15 downto 0) := (others => '0');
      write_dynamic_RAM   : out std_logic := '0';
      addr_dynamic_RAM    : out std_logic_vector(11 downto 0) := (others => '0');
      FTUlist_RAM_busy    :  in std_logic;
      FTUlist_RAM_started :  in std_logic;
      FTUlist_RAM_ready   :  in std_logic;
      data_FTUlist_RAM    : out std_logic_vector(15 downto 0) := (others => '0');
      write_FTUlist_RAM   : out std_logic := '0';
      addr_FTUlist_RAM    : out std_logic_vector(11 downto 0) := (others => '0')
    );
  end component;

  component FTM_fad_broadcast
    port(
      clk_50MHz         : in  std_logic;
      rx_en             : out STD_LOGIC;
      tx_en             : out STD_LOGIC;
      rx_d_0            : in  STD_LOGIC;
      tx_d_0            : out STD_LOGIC;
      rx_d_1            : in  STD_LOGIC;
      tx_d_1            : out STD_LOGIC;
      rx_d_2            : in  STD_LOGIC;
      tx_d_2            : out STD_LOGIC;
      rx_d_3            : in  STD_LOGIC;
      tx_d_3            : out STD_LOGIC;
      enable_ID_sending :  in std_logic;
      TIM_source        : in std_logic;
      LP_settings       : in std_logic_vector(3 downto 0);
      trigger_ID_ready  :  in std_logic;
      trigger_ID        :  in std_logic_vector(FAD_RS485_BLOCK_WIDTH - 1 downto 0);
      trigger_ID_read   : out std_logic              
    );
  end component;
  
  component ethernet_modul
    port(
      wiz_reset          : OUT   std_logic := '1';
      wiz_addr           : OUT   std_logic_vector (9 DOWNTO 0);
      wiz_data           : INOUT std_logic_vector (15 DOWNTO 0);
      wiz_cs             : OUT   std_logic := '1';
      wiz_wr             : OUT   std_logic := '1';
      wiz_rd             : OUT   std_logic := '1';
      wiz_int            : IN    std_logic ;
      clk                : IN    std_logic ;
      sd_ready           : OUT   std_logic ;
      sd_busy            : OUT   std_logic ;
      led                : OUT   std_logic_vector (7 DOWNTO 0);
      sd_read_ftu        : IN    std_logic ;
      sd_started_ftu     : OUT   std_logic := '0';
      cc_R0              : OUT   std_logic_vector (31 DOWNTO 0);
      cc_R1              : OUT   std_logic_vector (31 DOWNTO 0);
      cc_R11             : OUT   std_logic_vector (31 DOWNTO 0);
      cc_R13             : OUT   std_logic_vector (31 DOWNTO 0);
      cc_R14             : OUT   std_logic_vector (31 DOWNTO 0);
      cc_R15             : OUT   std_logic_vector (31 DOWNTO 0);
      cc_R8              : OUT   std_logic_vector (31 DOWNTO 0);
      cc_R9              : OUT   std_logic_vector (31 DOWNTO 0);
      coin_n_c           : OUT   std_logic_vector (15 DOWNTO 0);
      coin_n_p           : OUT   std_logic_vector (15 DOWNTO 0);
      dead_time          : OUT   std_logic_vector (15 DOWNTO 0);
      general_settings   : OUT   std_logic_vector (15 DOWNTO 0);
      lp1_amplitude      : OUT   std_logic_vector (15 DOWNTO 0);
      lp1_delay          : OUT   std_logic_vector (15 DOWNTO 0);
      lp2_amplitude      : OUT   std_logic_vector (15 DOWNTO 0);
      lp2_delay          : OUT   std_logic_vector (15 DOWNTO 0);
      lp_pt_freq         : OUT   std_logic_vector (15 DOWNTO 0);
      lp_pt_ratio        : OUT   std_logic_vector (15 DOWNTO 0);
      timemarker_delay   : OUT   std_logic_vector (15 DOWNTO 0);
      trigger_delay      : OUT   std_logic_vector (15 DOWNTO 0);
      sd_addr_ftu        : IN    std_logic_vector (11 DOWNTO 0);
      sd_data_out_ftu    : OUT   std_logic_vector (15 DOWNTO 0) := (others => '0');
      ftu_active_cr0     : OUT   std_logic_vector (15 DOWNTO 0);
      ftu_active_cr1     : OUT   std_logic_vector (15 DOWNTO 0);
      ftu_active_cr2     : OUT   std_logic_vector (15 DOWNTO 0);
      ftu_active_cr3     : OUT   std_logic_vector (15 DOWNTO 0);
      new_config         : OUT   std_logic := '0';
      config_started     : IN    std_logic ;
      config_start_eth   : IN    std_logic ;
      config_started_eth : OUT   std_logic := '0';
      config_ready_eth   : OUT   std_logic := '0';
      config_started_ack : OUT   std_logic := '0';
      fl_busy            : OUT   std_logic ;
      fl_ready           : OUT   std_logic ;
      fl_write_ftu       : IN    std_logic ;
      fl_started_ftu     : OUT   std_logic := '0';
      fl_addr_ftu        : IN    std_logic_vector (11 DOWNTO 0);
      fl_data_in_ftu     : IN    std_logic_vector (15 DOWNTO 0) := (others => '0');
      ping_ftu_start     : OUT   std_logic := '0';
      ping_ftu_started   : IN    std_logic ;
      ping_ftu_ready     : IN    std_logic ;
      dd_write_ftu       : IN    std_logic ;
      dd_started_ftu     : OUT   std_logic := '0';
      dd_data_in_ftu     : IN    std_logic_vector (15 DOWNTO 0);
      dd_addr_ftu        : IN    std_logic_vector (11 DOWNTO 0);
      dd_busy            : OUT   std_logic ;
      dd_ready           : OUT   std_logic ;
      coin_win_c         : OUT   std_logic_vector (15 DOWNTO 0) := (others => '0');
      coin_win_p         : OUT   std_logic_vector (15 DOWNTO 0) := (others => '0');
      --new stuff
      dd_block_ready_ftu     : IN     std_logic;
      dd_block_start_ack_ftu : OUT    std_logic                      := '0';
      dd_block_start_ftu     : IN     std_logic;
      dd_send                : IN     std_logic;
      dd_send_ack            : OUT    std_logic                      := '1';
      dd_send_ready          : OUT    std_logic                      := '1';
      --very new stuff
      ftu_error_calls        : IN     std_logic_vector (15 DOWNTO 0);
      ftu_error_data         : IN     std_logic_vector (223 DOWNTO 0);  -- (28 * 8) - 1
      ftu_error_send         : IN     std_logic;
      ftu_error_send_ack     : OUT    std_logic                      := '1';
      ftu_error_send_ready   : OUT    std_logic                      := '1';
      prescaling_FTU01       : OUT    std_logic_vector (15 DOWNTO 0) := (others => '0');
      trigger_counter        : IN     std_logic_vector (31 DOWNTO 0) := (others => '0');
      trigger_counter_read   : OUT    std_logic                      := '0';
      trigger_counter_valid  : IN     std_logic;
      --newest stuff
      board_id               : IN     std_logic_vector (63 DOWNTO 0);
      get_ts_counter         : OUT    std_logic                      := '0';
      get_ts_counter_ready   : IN     std_logic;
      get_ts_counter_started : IN     std_logic;
      timestamp_counter      : IN     std_logic_vector (47 DOWNTO 0);
      get_ot_counter         : OUT    std_logic                      := '0';
      get_ot_counter_ready   : IN     std_logic;
      get_ot_counter_started : IN     std_logic;
      on_time_counter        : IN     std_logic_vector (47 DOWNTO 0);
      temp_sensor_array      : IN     sensor_array_type;
      temp_sensor_ready      : IN     std_logic;
      crate_reset            : OUT    std_logic                      := '0';
      crate_reset_ack        : IN     std_logic;
      crate_reset_param      : OUT    std_logic_vector (15 DOWNTO 0) := (others => '0');
      start_run              : OUT    std_logic                      := '0';
      start_run_ack          : IN     std_logic;
      stop_run               : OUT    std_logic                      := '0';
      stop_run_ack           : IN     std_logic;
      current_cc_state       : IN     std_logic_vector (15 DOWNTO 0);
      start_run_param        : OUT    std_logic_vector (15 DOWNTO 0) := (others => '0');
      start_run_num_events   : OUT    std_logic_vector (31 DOWNTO 0) := (others => '0');
      new_config_ftu         : OUT    std_logic                      := '0';
      new_config_ftu_ack     : IN     std_logic;
      new_config_ftu_param   : OUT    std_logic_vector (15 DOWNTO 0) := (others => '0')
    );
  end component;

--  component counter_dummy IS
--    PORT( 
--      clk                 : IN     std_logic;
--      get_counter         : IN     std_logic;
--      get_counter_started : OUT    std_logic                      := '0';
--      get_counter_ready   : OUT    std_logic                      := '0';
--      counter             : OUT    std_logic_vector (47 DOWNTO 0) := (others => '0')
--    );
--  end component;

  component Timing_counter is
    port(
      clk              : in   STD_LOGIC;       -- 50 MHz system clock
      enable           : in   STD_LOGIC;       -- enable counter
      reset            : in   Std_LOGIC;       -- reset counter
      read_counter     : in   STD_LOGIC;       -- read counter
      reading_started  : out  STD_LOGIC;
      reading_valid    : out  STD_LOGIC;       -- counter reading at output ready
      counter_reading  : out  std_logic_vector (TC_WIDTH - 1 downto 0)
    );
  end component;

  component Lightpulser_interface_Basic is
    port(
      clk_50    : IN  STD_LOGIC;
      --clk_250   : IN  STD_LOGIC;
      Cal_0_p   : out STD_LOGIC := '0';
      Cal_0_n   : out STD_LOGIC := '1';
      Cal_1_p   : out STD_LOGIC := '0';
      Cal_1_n   : out STD_LOGIC := '1';
      Cal_2_p   : out STD_LOGIC := '0';
      Cal_2_n   : out STD_LOGIC := '1';
      Cal_3_p   : out STD_LOGIC := '0';
      Cal_3_n   : out STD_LOGIC := '1';
      Cal_4_p    : out STD_LOGIC := '0';
      Cal_4_n    : out STD_LOGIC := '1';
      Cal_5_p    : out STD_LOGIC := '0';
      Cal_5_n    : out STD_LOGIC := '1';
      Cal_6_p    : out STD_LOGIC := '0';
      Cal_6_n    : out STD_LOGIC := '1'; 
      Cal_7_p    : out STD_LOGIC := '0';
      Cal_7_n    : out STD_LOGIC := '1';  
      LP1_ampl       : in std_logic_vector (15 downto 0);
      LP2_ampl       : in std_logic_vector (15 downto 0);
      --LP1_delay      : in std_logic_vector (15 downto 0);
      --LP2_delay      : in std_logic_vector (15 downto 0);
      LP1_pulse      : in std_logic;
      LP2_pulse      : in std_logic;
      start_config   : in std_logic;
      config_started : out std_logic := '0';
      config_done    : out std_logic := '0' 
    );
  end component;
  
begin

--  -- IBUFG: Single-ended global clock input buffer
--  --        Spartan-3A
--  -- Xilinx HDL Language Template, version 11.4
  
--   IBUFG_inst : IBUFG
--   generic map (
--      IBUF_DELAY_VALUE => "0", -- Specify the amount of added input delay for buffer, 
--                               -- "0"-"16" 
--      IOSTANDARD => "DEFAULT")
--   port map (
--      O => clk_buf_sig, -- Clock buffer output
--      I => clk      -- Clock buffer input (connect directly to top-level port)
--   );
  
--  Inst_FTM_clk_gen : FTM_clk_gen
--    port map(
--      clk        => clk_buf_sig,
--      rst        => reset_sig,
--      clk_1      => clk_1M_sig,
--      clk_50     => clk_50M_sig,
--      clk_250    => clk_250M_sig,
--      clk_250_ps => clk_250M_ps_sig,
--      ready      => clk_ready_sig
--    );

  Inst_FTM_clk_gen_2 : FTM_clk_gen_2
    port map(
      clk        => clk,
      rst        => reset_sig,
      clk_1      => clk_1M_sig,
      clk_50     => clk_50M_sig,
      clk_250    => clk_250M_sig,
      clk_250_ps => clk_250M_ps_sig,
      ready      => clk_ready_sig
    );

  Inst_FTM_dna_gen : FTM_dna_gen
    port map(
      clk   => clk_50M_sig,
      start => dna_start_sig,
      dna   => dna_sig,
      ready => dna_ready_sig
    );
  
  --differential output buffer for trigger signal
  OBUFDS_LVDS_33_TRG : OBUFDS_LVDS_33
    port map(
      O  => TRG_p,
      OB => TRG_n,
      I  => trigger_signal_sig
    );

  --differential output buffer for TIM signal
  OBUFDS_LVDS_33_TIM : OBUFDS_LVDS_33
    port map(
      O  => TIM_Run_p,
      OB => TIM_Run_n,
      I  => TIM_signal_sig
    );

  --differential output buffer for fast reset signal
  OBUFDS_LVDS_33_RES : OBUFDS_LVDS_33
    port map(
      O  => RES_p,
      OB => RES_n,
      I  => '0'
    );
  
   Inst_trigger_manager : trigger_manager
     port map(
      --clocks
      clk_50MHz           => clk_50M_sig,
      clk_250MHz          => clk_250M_sig,
      clk_250MHz_180      => clk_250M_ps_sig,
      --trigger primitives from FTUs
      trig_prim_0         => Trig_Prim_A,  --crate 0
      trig_prim_1         => Trig_Prim_B,  --crate 1
      trig_prim_2         => Trig_Prim_C,  --crate 2
      trig_prim_3         => Trig_Prim_D,  --crate 3
      --external signals
      ext_trig_1          => ext_Trig(1),
      ext_trig_2          => ext_Trig(2),
      ext_veto            => Veto,
      FAD_busy_0          => Busy0,  --crate 0
      FAD_busy_1          => Busy1,  --crate 1
      FAD_busy_2          => Busy2,  --crate 2
      FAD_busy_3          => Busy3,  --crate 3
      --control signals from e.g. main control
      start_run           => trigger_start_sig,  --enable trigger output
      stop_run            => trigger_stop_sig,  --disable trigger output
      new_config          => config_trigger_sig,
      --settings register (see FTM Firmware Specifications)
      general_settings    => general_settings_sig,
      LP_and_PED_freq     => lp_pt_freq_sig,
      LP1_LP2_PED_ratio   => lp_pt_ratio_sig,
      maj_coinc_n_phys    => coin_n_p_sig,
      maj_coinc_n_calib   => coin_n_c_sig,
      trigger_delay       => trigger_delay_sig,
      TIM_delay           => timemarker_delay_sig,
      dead_time           => dead_time_sig,
      coinc_window_phys   => coin_win_p_sig,
      coinc_window_calib  => coin_win_c_sig,
      active_FTU_list_0   => ftu_active_cr0_sig,
      active_FTU_list_1   => ftu_active_cr1_sig,
      active_FTU_list_2   => ftu_active_cr2_sig,
      active_FTU_list_3   => ftu_active_cr3_sig,
      --control signals or information for other entities
      trigger_ID_read     => trigger_ID_read_sig,
      trig_cnt_copy_read  => trigger_counter_read_sig,
      trigger_ID_ready    => trigger_ID_ready_sig,
      trigger_ID          => trigger_ID_sig,
      trig_cnt_copy       => trigger_counter_sig,  --counter reading
      trig_cnt_copy_valid => trigger_counter_valid_sig,  --trigger counter reading is valid
      trigger_active      => trigger_active_sig,  --phys triggers are enabled/active
      config_done         => config_trigger_done_sig,
      LP1_pulse           => LP1_pulse_sig,  --send start signal to light pulser 1
      LP2_pulse           => LP2_pulse_sig,  --send start signal to light pulser 2
      --trigger and time marker output signals to FADs
      trigger_signal      => trigger_signal_sig,
      TIM_signal          => TIM_signal_sig
    );
   
  Inst_Clock_cond_interface : Clock_cond_interface
    port map(
      clk               => clk_50M_sig,
      CLK_Clk_Cond      => CLK_Clk_Cond,
      LE_Clk_Cond       => LE_Clk_Cond,
      DATA_Clk_Cond     => DATA_Clk_Cond,
      SYNC_Clk_Cond     => SYNC_Clk_Cond,
      LD_Clk_Cond       => LD_Clk_Cond,
      TIM_Sel           => TIM_Sel,
      cc_R0             => cc_R0_sig,
      cc_R1             => cc_R1_sig,
      cc_R8             => cc_R8_sig,
      cc_R9             => cc_R9_sig,
      cc_R11            => cc_R11_sig,
      cc_R13            => cc_R13_sig,
      cc_R14            => cc_R14_sig,
      cc_R15            => cc_R15_sig,
      start_config      => config_start_cc_sig,
      config_started    => config_started_cc_sig,
      config_done       => config_ready_cc_sig,
      locked            => cc_locked_sig,
      timemarker_select => general_settings_sig(0)
    );
   
  Inst_FTM_central_control : FTM_central_control
    port map(
      clk                  => clk_50M_sig,
      clk_ready            => clk_ready_sig,
      clk_scaler           => clk_1M_sig,
      new_config           => new_config_sig,
      config_started       => config_started_sig,
      config_started_ack   => config_started_ack_sig,
      config_start_eth     => config_start_eth_sig,
      config_started_eth   => config_started_eth_sig,
      config_ready_eth     => config_ready_eth_sig,
      config_start_ftu     => config_start_ftu_sig,
      config_started_ftu   => config_started_ftu_sig,
      config_ready_ftu     => config_ready_ftu_sig,
      ping_ftu_start       => ping_ftu_start_sig,
      ping_ftu_started     => ping_ftu_started_sig,
      ping_ftu_ready       => ping_ftu_ready_sig,
      ping_ftu_start_ftu   => ping_ftu_start_ftu_sig,
      ping_ftu_started_ftu => ping_ftu_started1_sig,
      ping_ftu_ready_ftu   => ping_ftu_ready1_sig,
      rates_ftu            => rates_ftu_start_sig,
      rates_started_ftu    => rates_ftu_started_sig,
      rates_ready_ftu      => rates_ftu_ready_sig,
      prescaling_FTU01     => prescaling_FTU01_sig(7 downto 0),
      dd_send              => dd_send_sig,
      dd_send_ack          => dd_send_ack_sig,
      dd_send_ready        => dd_send_ready_sig,
      dd_block_ready_ftu     => dd_block_ready_ftu_sig,
      dd_block_start_ack_ftu => dd_block_start_ack_ftu_sig,
      dd_block_start_ftu     => dd_block_start_ftu_sig,
      config_start_cc      => config_start_cc_sig,
      config_started_cc    => config_started_cc_sig,
      config_ready_cc      => config_ready_cc_sig,
      cc_locked            => cc_locked_sig,
      config_start_lp      => config_start_lp_sig,
      config_started_lp    => config_started_lp_sig,
      config_ready_lp      => config_ready_lp_sig,
      config_trigger       => config_trigger_sig,
      config_trigger_done  => config_trigger_done_sig,
      dna_start            => dna_start_sig,
      dna_ready            => dna_ready_sig,
      crate_reset            => crate_reset_sig,
      crate_reset_ack        => crate_reset_ack_sig,
      crate_reset_param      => crate_reset_param_sig,
      start_run              => start_run_sig,
      start_run_ack          => start_run_ack_sig,
      stop_run               => stop_run_sig,
      stop_run_ack           => stop_run_ack_sig,
      current_cc_state       => current_cc_state_sig,
      cc_state_test          => cc_state_test_sig,
      start_run_param        => start_run_param_sig,
      start_run_num_events   => start_run_num_events_sig,
      trigger_start => trigger_start_sig,
      trigger_stop  => trigger_stop_sig,
      enable_ID_sending => enable_ID_sending_sig,
      reset_timer => reset_timer_sig,
      crate_res_0 => crate_res0_sig,
      crate_res_1 => crate_res1_sig,
      crate_res_2 => crate_res2_sig,
      crate_res_3 => crate_res3_sig,
      new_config_ftu     => new_config_ftu_sig,
      new_config_ftu_ack => new_config_ftu_ack_sig,
      config_single_FTU => config_single_FTU_sig,
      config_single_FTU_started => config_single_FTU_started_sig,
      config_single_FTU_done => config_single_FTU_done_sig
    );
  
  Inst_FTM_ftu_control : FTM_ftu_control
    port map(
      clk_50MHz           => clk_50M_sig,
      rx_en               => Bus1_Rx_En,
      tx_en               => Bus1_Tx_En,
      rx_d_0              => Bus1_RxD_0,
      tx_d_0              => Bus1_TxD_0,
      rx_d_1              => Bus1_RxD_1,
      tx_d_1              => Bus1_TxD_1,
      rx_d_2              => Bus1_RxD_2,
      tx_d_2              => Bus1_TxD_2,
      rx_d_3              => Bus1_RxD_3,
      tx_d_3              => Bus1_TxD_3,
      new_config          => config_start_ftu_sig,
      ping_all            => ping_ftu_start_ftu_sig,
      read_rates          => rates_ftu_start_sig,
      config_single_FTU   => config_single_FTU_sig,
      read_rates_started  => rates_ftu_started_sig,
      read_rates_done     => rates_ftu_ready_sig,
      new_config_started  => config_started_ftu_sig,
      new_config_done     => config_ready_ftu_sig,
      ping_all_started    => ping_ftu_started1_sig,
      ping_all_done       => ping_ftu_ready1_sig,
      config_single_FTU_started => config_single_FTU_started_sig,
      config_single_FTU_done    => config_single_FTU_done_sig,
      ftu_active_cr0      => ftu_active_cr0_sig,
      ftu_active_cr1      => ftu_active_cr1_sig,
      ftu_active_cr2      => ftu_active_cr2_sig,
      ftu_active_cr3      => ftu_active_cr3_sig,
      config_single_FTU_param => new_config_ftu_param_sig,
      ftu_error_calls     => ftu_error_calls_sig,
      ftu_error_data      => ftu_error_data_sig,
      ftu_error_send      => ftu_error_send_sig,
      ftu_error_send_ack  => ftu_error_send_ack_sig,
      ftu_error_send_ready=> ftu_error_send_ready_sig,
      static_RAM_busy     => sd_busy_sig,
      static_RAM_started  => sd_started_ftu_sig,
      static_RAM_ready    => sd_ready_sig,
      data_static_RAM     => sd_data_out_ftu_sig,
      read_static_RAM     => sd_read_ftu_sig,
      addr_static_RAM     => sd_addr_ftu_sig,
      dynamic_RAM_busy    => dd_busy_sig,
      dynamic_RAM_started => dd_started_ftu_sig,
      dynamic_RAM_ready   => dd_ready_sig,
      data_dynamic_RAM    => dd_data_sig,
      write_dynamic_RAM   => dd_write_sig,
      addr_dynamic_RAM    => dd_addr_sig,
      FTUlist_RAM_busy    => fl_busy_sig,
      FTUlist_RAM_started => fl_started_ftu_sig,
      FTUlist_RAM_ready   => fl_ready_sig,
      data_FTUlist_RAM    => fl_data_sig,
      write_FTUlist_RAM   => fl_write_sig,
      addr_FTUlist_RAM    => fl_addr_sig
    );

  Inst_FTM_fad_broadcast : FTM_fad_broadcast
    port map(
      clk_50MHz         => clk_50M_sig,
      rx_en             => Bus2_Rx_En,
      tx_en             => Bus2_Tx_En,
      rx_d_0            => Bus2_RxD_0,
      tx_d_0            => Bus2_TxD_0,
      rx_d_1            => Bus2_RxD_1,
      tx_d_1            => Bus2_TxD_1,
      rx_d_2            => Bus2_RxD_2,
      tx_d_2            => Bus2_TxD_2,
      rx_d_3            => Bus2_RxD_3,
      tx_d_3            => Bus2_TxD_3,
      enable_ID_sending => enable_ID_sending_sig,
      TIM_source        => general_settings_sig(0),
      LP_settings       => "0000",
      trigger_ID_ready  => trigger_ID_ready_sig,
      trigger_ID        => trigger_ID_sig,
      trigger_ID_read   => trigger_ID_read_sig                
    );
  
  Inst_ethernet_modul : ethernet_modul
    port map(
      wiz_reset          => W_RES,
      wiz_addr           => W_A,
      wiz_data           => W_D,
      wiz_cs             => W_CS,
      wiz_wr             => W_WR,
      wiz_rd             => W_RD,
      wiz_int            => W_INT,
      clk                => clk_50M_sig,
      sd_ready           => sd_ready_sig,
      sd_busy            => sd_busy_sig,
      led                => led_sig,
      sd_read_ftu        => sd_read_ftu_sig,
      sd_started_ftu     => sd_started_ftu_sig,
      cc_R0              => cc_R0_sig,
      cc_R1              => cc_R1_sig,
      cc_R11             => cc_R11_sig,
      cc_R13             => cc_R13_sig,
      cc_R14             => cc_R14_sig,
      cc_R15             => cc_R15_sig,
      cc_R8              => cc_R8_sig,
      cc_R9              => cc_R9_sig,
      coin_n_c           => coin_n_c_sig,
      coin_n_p           => coin_n_p_sig,
      dead_time          => dead_time_sig,
      general_settings   => general_settings_sig,
      lp1_amplitude      => lp1_amplitude_sig,
      lp1_delay          => lp1_delay_sig,
      lp2_amplitude      => lp2_amplitude_sig,
      lp2_delay          => lp2_delay_sig,
      lp_pt_freq         => lp_pt_freq_sig,
      lp_pt_ratio        => lp_pt_ratio_sig,
      timemarker_delay   => timemarker_delay_sig,
      trigger_delay      => trigger_delay_sig,
      sd_addr_ftu        => sd_addr_ftu_sig,
      sd_data_out_ftu    => sd_data_out_ftu_sig,
      ftu_active_cr0     => ftu_active_cr0_sig,
      ftu_active_cr1     => ftu_active_cr1_sig,
      ftu_active_cr2     => ftu_active_cr2_sig,
      ftu_active_cr3     => ftu_active_cr3_sig,
      new_config         => new_config_sig,
      config_started     => config_started_sig,
      config_start_eth   => config_start_eth_sig,
      config_started_eth => config_started_eth_sig,
      config_ready_eth   => config_ready_eth_sig,
      config_started_ack => config_started_ack_sig,
      fl_busy            => fl_busy_sig,
      fl_ready           => fl_ready_sig,
      fl_write_ftu       => fl_write_sig,
      fl_started_ftu     => fl_started_ftu_sig,
      fl_addr_ftu        => fl_addr_sig,
      fl_data_in_ftu     => fl_data_sig,
      ping_ftu_start     => ping_ftu_start_sig,
      ping_ftu_started   => ping_ftu_started_sig,
      ping_ftu_ready     => ping_ftu_ready_sig,
      dd_write_ftu       => dd_write_sig,
      dd_started_ftu     => dd_started_ftu_sig,
      dd_data_in_ftu     => dd_data_sig,
      dd_addr_ftu        => dd_addr_sig,
      dd_busy            => dd_busy_sig,
      dd_ready           => dd_ready_sig,
      coin_win_c         => coin_win_c_sig,
      coin_win_p         => coin_win_p_sig,
      --new stuff
      dd_block_ready_ftu     => dd_block_ready_ftu_sig,
      dd_block_start_ack_ftu => dd_block_start_ack_ftu_sig,
      dd_block_start_ftu     => dd_block_start_ftu_sig,
      dd_send                => dd_send_sig,
      dd_send_ack            => dd_send_ack_sig,
      dd_send_ready          => dd_send_ready_sig,
      --very new stuff
      ftu_error_calls        => ftu_error_calls_sig,
      ftu_error_data         => ftu_error_data_sig,
      ftu_error_send         => ftu_error_send_sig,
      ftu_error_send_ack     => ftu_error_send_ack_sig,
      ftu_error_send_ready   => ftu_error_send_ready_sig,
      prescaling_FTU01       => prescaling_FTU01_sig,
      trigger_counter        => trigger_counter_sig,  
      trigger_counter_read   => trigger_counter_read_sig,
      trigger_counter_valid  => trigger_counter_valid_sig,
      --newest stuff
      board_id               => dna_sig,
      get_ts_counter         => get_ts_counter_sig,
      get_ts_counter_ready   => get_ts_counter_ready_sig,
      get_ts_counter_started => get_ts_counter_started_sig,
      timestamp_counter      => timestamp_counter_sig,
      get_ot_counter         => get_ot_counter_sig, 
      get_ot_counter_ready   => get_ot_counter_ready_sig,
      get_ot_counter_started => get_ot_counter_started_sig,
      on_time_counter        => on_time_counter_sig,
      temp_sensor_array      => (35, 45, 55, 65),
      temp_sensor_ready      => '1',
      crate_reset            => crate_reset_sig,
      crate_reset_ack        => crate_reset_ack_sig,
      crate_reset_param      => crate_reset_param_sig,
      start_run              => start_run_sig,
      start_run_ack          => start_run_ack_sig,
      stop_run               => stop_run_sig,
      stop_run_ack           => stop_run_ack_sig,
      current_cc_state       => current_cc_state_sig,
      start_run_param        => start_run_param_sig,
      start_run_num_events   => start_run_num_events_sig,
      new_config_ftu         => new_config_ftu_sig,
      new_config_ftu_ack     => new_config_ftu_ack_sig,
      new_config_ftu_param   => new_config_ftu_param_sig
    );

--  Inst_counter_dummy_ts : counter_dummy
--    port map( 
--      clk                 => clk_50M_sig,
--      get_counter         => get_ts_counter_sig,
--      get_counter_started => get_ts_counter_started_sig,
--      get_counter_ready   => get_ts_counter_ready_sig,
--      counter             => timestamp_counter_sig
--    );

--  Inst_counter_dummy_ot : counter_dummy
--    port map( 
--      clk                 => clk_50M_sig,
--      get_counter         => get_ot_counter_sig,
--      get_counter_started => get_ot_counter_started_sig,
--      get_counter_ready   => get_ot_counter_ready_sig,
--      counter             => on_time_counter_sig
--    );

  Inst_Timing_counter_ts : Timing_counter
    port map( 
      clk                 => clk_50M_sig,
      enable              => '1',
      reset               => reset_timer_sig,
      read_counter        => get_ts_counter_sig,
      reading_started     => get_ts_counter_started_sig,
      reading_valid       => get_ts_counter_ready_sig,
      counter_reading     => timestamp_counter_sig
    );

  Inst_Timing_counter_ot : Timing_counter
    port map( 
      clk                 => clk_50M_sig,
      enable              => trigger_active_sig,
      reset               => reset_timer_sig,
      read_counter        => get_ot_counter_sig,
      reading_started     => get_ot_counter_started_sig,
      reading_valid       => get_ot_counter_ready_sig,
      counter_reading     => on_time_counter_sig
    );

  Inst_Lightpulser_interface_Basic : Lightpulser_interface_Basic
    port map (
      clk_50         => clk_50M_sig,
      --clk_250        => clk_250M_sig,
      Cal_0_p        => Cal_2_p,  --swapped with Cal_2_p due to connector on FLD board
      Cal_0_n        => Cal_2_n,  --swapped with Cal_2_n due to connector on FLD board
      Cal_1_p        => Cal_1_p,
      Cal_1_n        => Cal_1_n,
      Cal_2_p        => Cal_0_p,  --swapped with Cal_0_p due to connector on FLD board
      Cal_2_n        => Cal_0_n,  --swapped with Cal_0_n due to connector on FLD board
      Cal_3_p        => Cal_3_p,
      Cal_3_n        => Cal_3_n,
      Cal_4_p        => Cal_6_p,  --swapped with Cal_6_p due to connector on FLD board
      Cal_4_n        => Cal_6_n,  --swapped with Cal_6_n due to connector on FLD board
      Cal_5_p        => Cal_5_p,
      Cal_5_n        => Cal_5_n,
      Cal_6_p        => Cal_4_p,  --swapped with Cal_4_p due to connector on FLD board
      Cal_6_n        => Cal_4_n,  --swapped with Cal_4_n due to connector on FLD board
      Cal_7_p        => Cal_7_p,
      Cal_7_n        => Cal_7_n,
      LP1_ampl       => lp1_amplitude_sig,
      LP2_ampl       => lp2_amplitude_sig,
      --LP1_delay      => lp1_delay_sig,
      --LP2_delay      => lp2_delay_sig,
      LP1_pulse      => LP1_pulse_sig,
      LP2_pulse      => LP2_pulse_sig,
      start_config   => config_start_lp_sig,
      config_started => config_started_lp_sig,
      config_done    => config_ready_lp_sig
    );
  
  --LED_red <= led_sig(3 downto 0);
  --LED_ye  <= led_sig(5 downto 4);
  --LED_gn  <= led_sig(7 downto 6);
  LED_red <= "1111";
  LED_ye  <= "11";
  LED_gn  <= "11";
  
  TP(32 downto 8) <= (others => '0');
  --TP(8) <= clk_50M_sig;
  TP( 7 downto 0) <= cc_state_test_sig;

  Crate_Res0 <= crate_res0_sig;
  Crate_Res1 <= crate_res1_sig;
  Crate_Res2 <= crate_res2_sig;
  Crate_Res3 <= crate_res3_sig;
  
end Behavioral;


