/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  2001 <mailto:tbretz@uni-sw.gwdg.de>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

///////////////////////////////////////////////////////////////////////
//
// MHHillas
//
// This class contains histograms for every Hillas parameter
//
///////////////////////////////////////////////////////////////////////

#include "MHHillas.h"

#include <math.h>

#include <TH1.h>
#include <TPad.h>
#include <TCanvas.h>

#include "MHillas.h"

ClassImp(MHHillas);

// --------------------------------------------------------------------------
//
// Setup four histograms for Width, Length
//
MHHillas::MHHillas (const char *name, const char *title)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHHillas" ;
    fTitle = title ? title : "Container for Hillas histograms" ;

    //
    // loop over all Pixels and create two histograms
    // one for the Low and one for the High gain
    // connect all the histogram with the container fHist
    //
    fWidth  = new TH1F("Width [mm]",  "Width of Hillas",  100, 0, 300);
    fLength = new TH1F("Length [mm]", "Length of Hillas", 100, 0, 300);

    fLength->SetDirectory(NULL);
    fWidth->SetDirectory(NULL);

    fLength->GetXaxis()->SetTitle("Length [mm]");
    fWidth->GetXaxis()->SetTitle("Width [mm]");

    fLength->GetYaxis()->SetTitle("Counts");
    fWidth->GetYaxis()->SetTitle("Counts");
}

// --------------------------------------------------------------------------
//
// Delete the four histograms
//
MHHillas::~MHHillas()
{
    delete fWidth;
    delete fLength;
}

// --------------------------------------------------------------------------
//
// Fill the four histograms with data from a MHillas-Container.
// Be careful: Only call this with an object of type MHillas
//
void MHHillas::Fill(const MParContainer *par)
{
    const MHillas &h = *(MHillas*)par;

    fWidth ->Fill(h.GetWidth());
    fLength->Fill(h.GetLength());
}

// --------------------------------------------------------------------------
//
// Draw clones of all four histograms. So that the object can be deleted
// and the histograms are still visible in the canvas.
// The cloned object are deleted together with the canvas if the canvas is
// destroyed. If you want to handle dostroying the canvas you can get a
// pointer to it from this function
//
TObject *MHHillas::DrawClone(Option_t *opt) const
{
    TCanvas *c = MakeDefCanvas("Hillas", "Histograms of Hillas Parameters",
                               350, 500);
    c->Divide(1, 2);

    gROOT->SetSelectedPad(NULL);

    //
    // This is necessary to get the expected bahviour of DrawClone
    //
    c->cd(1);
    fLength->DrawCopy();

    c->cd(2);
    fWidth->DrawCopy();

    c->Modified();
    c->Update();

    return c;
}

// --------------------------------------------------------------------------
//
// Creates a new canvas and draws the four histograms into it.
// Be careful: The histograms belongs to this object and won't get deleted
// together with the canvas.
//
void MHHillas::Draw(Option_t *)
{
    if (!gPad)
        MakeDefCanvas("Hillas", "Histograms of Hillas Parameters", 350, 500);

    gPad->Divide(1, 2);

    gPad->cd(1);
    fLength->Draw();

    gPad->cd(2);
    fWidth->Draw();

    gPad->Modified();
    gPad->Update();
}