/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

///////////////////////////////////////////////////////////////////////
//
// MHStarMap
//
// This class contains a 2-dimensional histogram. It should show some
// kind of star map. The algorith which calculates the star map
// from the Hillas parameters (Fill) can be enhanced.
//
///////////////////////////////////////////////////////////////////////

#include "MHStarMap.h"

#include <TStyle.h>   // gStyle
#include <TColor.h>   // SetRGB
#include <TCanvas.h>  // TCanvas

#include "MHillas.h"

ClassImp(MHStarMap);

// --------------------------------------------------------------------------
//
// Create the star map histogram
//
MHStarMap::MHStarMap(const char *name, const char *title)
{
    //
    //  default constructor
    //  creates an a list of histograms for all pixels and both gain channels
    //

    //
    //   set the name and title of this object
    //
    
    fName  = name  ? name  : "MHStarMap" ;
    fTitle = title ? title : "Container for a Star Map" ;

    //
    //   loop over all Pixels and create two histograms
    //   one for the Low and one for the High gain
    //   connect all the histogram with the container fHist
    //
    fStarMap = new TH2F("StarMap", "2D Hillas Star Map",
                        150, -300, 300,
                        150, -300, 300);

    fStarMap->SetDirectory(NULL);

    fStarMap->SetXTitle("x [mm]");
    fStarMap->SetYTitle("y [mm]");
    fStarMap->SetZTitle("Counts");
}

// --------------------------------------------------------------------------
//
// delete the histogram instance
//
MHStarMap::~MHStarMap()
{
    delete fStarMap;
}

// --------------------------------------------------------------------------
//
// Fill the four histograms with data from a MHillas-Container.
// Be careful: Only call this with an object of type MHillas
//
Bool_t MHStarMap::Fill(const MParContainer *par)
{
    const MHillas &h = *(MHillas*)par;

    const float delta = h.GetDelta();

    const float m = tan(delta);
    const float t = m*h.GetMeanX()-h.GetMeanY();

    if (m>-1 && m<1)
        for (int x=-298; x<298; x+=4)
        {
            const float y = m*x+t;

            fStarMap->Fill(x, y);
        }
    else
        for (int y=-298; y<298; y+=4)
        {
            const float x = (y-t)/m;

            fStarMap->Fill(x, y);
        }

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Set the palette you wanna use:
//  - you could set the root "Pretty Palette Violet->Red" by
//    gStyle->SetPalette(1, 0), but in some cases this may look
//    confusing
//  - The maximum colors root allowes us to set by ourself
//    is 50 (idx: 51-100). This colors are set to a grayscaled
//    palette
//  - the number of contours must be two less than the number
//    of palette entries
//
void MHStarMap::PrepareDrawing() const
{
    const Int_t numg = 32; // number of gray scaled colors
    const Int_t numw = 32; // number of white

    Int_t palette[numg+numw];

    //
    // The first half of the colors are white.
    // This is some kind of optical background supression
    //
    gROOT->GetColor(51)->SetRGB(1, 1, 1);

    Int_t i;
    for (i=0; i<numw; i++)
        palette[i] = 51;

    //
    // now the (gray) scaled part is coming
    //
    for (;i<numw+numg; i++)
    {
        const Float_t gray = 1.0-(float)(i-numw)/(numg-1.0);

        gROOT->GetColor(52+i)->SetRGB(gray, gray, gray);
        palette[i] = 52+i;
    }

    //
    // Set the palette and the number of contour levels
    //
    gStyle->SetPalette(numg+numw, palette);
    fStarMap->SetContour(numg+numw-2);
}


// --------------------------------------------------------------------------
//
// Draw clones of the  histograms, so that the object can be deleted
// and the histogram is still visible in the canvas.
// The cloned object is deleted together with the canvas if the canvas is
// destroyed. If you want to handle destroying the canvas you can get a
// pointer to it from this function
//
TObject *MHStarMap::DrawClone(Option_t *opt) const
{
    TCanvas *c=MakeDefCanvas(fStarMap, 500, 500);

    //
    // This is necessary to get the expected bahviour of DrawClone
    //
    gROOT->SetSelectedPad(NULL);

    PrepareDrawing();

    fStarMap->DrawCopy("colz");

    c->Modified();
    c->Update();

    return c;
}

// --------------------------------------------------------------------------
//
// Creates a new canvas and draw the histogram into it.
// Be careful: The histogram belongs to this object and won't get deleted
// together with the canvas.
//
void MHStarMap::Draw(Option_t *)
{
    if (!gPad)
        MakeDefCanvas(fStarMap, 500, 500);

    PrepareDrawing();

    fStarMap->Draw("colz");

    gPad->Modified();
    gPad->Update();
}
