/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  2001 <mailto:tbretz@uni-sw.gwdg.de>
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

///////////////////////////////////////////////////////////////////////
//
// MHHillasSrc
//
// This class contains histograms for every Hillas parameter
//
///////////////////////////////////////////////////////////////////////
#include "MHHillasSrc.h"

#include <math.h>

#include <TH1.h>
#include <TPad.h>
#include <TCanvas.h>

#include "MParList.h"

#include "MHillas.h"
#include "MHillasSrc.h"

ClassImp(MHHillasSrc);

// --------------------------------------------------------------------------
//
// Setup four histograms for Alpha, and Dist
//
MHHillasSrc::MHHillasSrc(const char *name, const char *title)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHHillasSrc";
    fTitle = title ? title : "Container for Hillas histograms";

    //
    // loop over all Pixels and create two histograms
    // one for the Low and one for the High gain
    // connect all the histogram with the container fHist
    //
    fAlpha = new TH1F("Alpha [deg]", "Alpha of Hillas",   90, 0,  90);
    fDist  = new TH1F("Dist [mm]",   "Dist of Hillas",   100, 0, 600);

    fAlpha->SetDirectory(NULL);
    fDist->SetDirectory(NULL);

    fAlpha->GetXaxis()->SetTitle("\\alpha [\\circ]");
    fDist->GetXaxis()->SetTitle("Dist [mm]");

    fAlpha->GetYaxis()->SetTitle("Counts");
    fDist->GetYaxis()->SetTitle("Counts");
}

// --------------------------------------------------------------------------
//
// Delete the four histograms
//
MHHillasSrc::~MHHillasSrc()
{
    delete fAlpha;
    delete fDist;
}

// --------------------------------------------------------------------------
//
// Setup the Binning for the histograms automatically if the correct
// instances of MBinning (with the names 'BinningAlpha' and 'BinningDist')
// are found in the parameter list
//
Bool_t MHHillasSrc::SetupFill(const MParList *plist)
{
    const MBinning* binsa = (MBinning*)plist->FindObject("BinningAlpha");
    const MBinning* binsd = (MBinning*)plist->FindObject("BinningDist");

    if (binsa)
        SetBinning(fAlpha, binsa);

    if (binsd)
        SetBinning(fDist, binsd);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Fill the four histograms with data from a MHillas-Container.
// Be careful: Only call this with an object of type MHillas
//
Bool_t MHHillasSrc::Fill(const MParContainer *par)
{
    const MHillasSrc &h = *(MHillasSrc*)par;

    fAlpha->Fill(fabs(h.GetAlpha()));
    fDist ->Fill(h.GetDist());

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Draw clones of all two histograms. So that the object can be deleted
// and the histograms are still visible in the canvas.
// The cloned object are deleted together with the canvas if the canvas is
// destroyed. If you want to handle dostroying the canvas you can get a
// pointer to it from this function
//
TObject *MHHillasSrc::DrawClone(Option_t *opt) const
{
    TCanvas *c = MakeDefCanvas("Hillas", "Histograms of Source dependant Parameters",
                               350, 500);
    c->Divide(1, 2);

    // FIXME: Display Source position

    gROOT->SetSelectedPad(NULL);

    //
    // This is necessary to get the expected bahviour of DrawClone
    //
    c->cd(1);
    fAlpha->DrawCopy();

    c->cd(2);
    fDist->DrawCopy();

    c->Modified();
    c->Update();

    return c;
}

// --------------------------------------------------------------------------
//
// Creates a new canvas and draws the two histograms into it.
// Be careful: The histograms belongs to this object and won't get deleted
// together with the canvas.
//
void MHHillasSrc::Draw(Option_t *)
{
    if (!gPad)
        MakeDefCanvas("Hillas", "Histograms of Src dependant Parameters", 350, 500);

    // FIXME: Display Source position

    gPad->Divide(1, 2);

    gPad->cd(1);
    fAlpha->Draw();

    gPad->cd(2);
    fDist->Draw();

    gPad->Modified();
    gPad->Update();
}
