/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  1/2002 <mailto:tbretz@uni-sw.gwdg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

#include "MHEnergyTheta.h" 

#include <TCanvas.h>

#include "MH.h"
#include "MBinning.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MMcEvt.hxx"

#include "MParList.h"

ClassImp(MHEnergyTheta);

// --------------------------------------------------------------------------
//
//  Creates the three necessary histograms:
//   - selected showers (input)
//   - all showers (input)
//   - collection area (result)
//
MHEnergyTheta::MHEnergyTheta(const char *name, const char *title)
{ 
    //   initialize the histogram for the distribution r vs E
    //
    //   we set the energy range from 1 Gev to 10000 GeV (in log 5 orders
    //   of magnitude) and for each order we take 10 subdivision --> 50 xbins
    //
    //   we set the radius range from 0 m to 500 m with 10 m bin --> 50 ybins

  
    fName  = name  ? name  : "MHEnergyTheta";
    fTitle = title ? title : "Data to Calculate Collection Area";

    fHist.SetDirectory(NULL);

    fHist.SetXTitle("E [GeV]");
    fHist.SetYTitle("\\Theta [\\circ]");
    fHist.SetZTitle("N");
}

Bool_t MHEnergyTheta::SetupFill(const MParList *plist)
{
    MBinning* binsenergy = (MBinning*)plist->FindObject("BinningE");
    MBinning* binstheta  = (MBinning*)plist->FindObject("BinningTheta");
    if (!binsenergy || !binstheta)
    {
        *fLog << err << dbginf << "At least one MBinning not found... aborting." << endl;
        return kFALSE;
    }

    SetBinning(&fHist, binsenergy, binstheta);

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Delete the three histograms
//
MHEnergyTheta::~MHEnergyTheta()
{
}

// --------------------------------------------------------------------------
//
// Fill data into the histogram which contains all showers
//
Bool_t MHEnergyTheta::Fill(const MParContainer *par)
{
    const MMcEvt &mcevt = *(MMcEvt*)par;

    fHist.Fill(mcevt.GetEnergy(), mcevt.GetTheta());

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Draw the histogram with all showers
//
void MHEnergyTheta::Draw(Option_t* option)
{
    if (!gPad)
        MakeDefCanvas(&fHist);

    fHist.DrawCopy(option);
    gPad->SetLogy();

    gPad->Modified();
    gPad->Update();
}

// --------------------------------------------------------------------------
//
// Creates a new canvas and draws the histogram into it.
// Be careful: The histogram belongs to this object and won't get deleted
// together with the canvas.
//
TObject *MHEnergyTheta::DrawClone(Option_t* option) const
{
    TCanvas *c = MakeDefCanvas(&fHist);

    //
    // This is necessary to get the expected bahviour of DrawClone
    //
    gROOT->SetSelectedPad(NULL);

    ((TH2D*)&fHist)->DrawCopy(option);
    gPad->SetLogy();

    c->Modified();
    c->Update();

    return c;
}

// --------------------------------------------------------------------------
//
//  Calculate the Efficiency (collection area) and set the 'ReadyToSave'
//  flag
//
void MHEnergyTheta::Divide(const TH2D *h1, const TH2D *h2)
{
    // Description!

    fHist.Sumw2();
    fHist.Divide((TH2D*)h1, (TH2D*)h2);

    SetReadyToSave();
}
