
/********************************************************************\

  Name:         HV.cc

  Created by:   Sebastian Commichau, November 2008
                commichau@phys.ethz.ch

  Contents:     Main class for HV supply
                
\********************************************************************/

#include "HV.h"

extern bool Verbose;

//
// Constructor
//
HVBoard::HVBoard(int DeviceNumber, char *DeviceName) {
   
  char Buffer[MAX_COM_SIZE];
  struct termios tio;
 
  fTimeOut  = 10;
  BoardNumber = DeviceNumber;
  BoardName = DeviceName;
  LastWrapCount = -1;
  
  // Open device
  snprintf(Buffer, MAX_COM_SIZE, "/dev/%s",DeviceName);
  if ((fDescriptor = open(Buffer, O_RDWR|O_NOCTTY|O_NDELAY)) == -1) {
    printf("Error: Could not open device #%d: %s (%d/%s)\n", DeviceNumber, DeviceName, errno, strerror(errno));
    return;
  }

  // Get current serial port settings
  if (tcgetattr(fDescriptor, &tio) == -1) {
    printf("Error: tcgetattr() failed on device #%d: %s (%d/%s)\n", DeviceNumber, DeviceName, errno, strerror(errno));
    return;   
  }

  // Set baudrate and raw mode
  if (cfsetspeed(&tio, BAUDRATE) == -1) printf("Error: Could not set baud rate of device #%d: %s (%d/%s)\n", DeviceNumber, DeviceName, errno, strerror(errno));
  cfmakeraw(&tio);
  if (tcsetattr(fDescriptor, TCSANOW, &tio ) == -1) printf("Error: tcsetattr() failed on device #%d: %s (%d/%s)\n", DeviceNumber, DeviceName, errno, strerror(errno));

  return;
}

//
// Destructor
//
HVBoard::~HVBoard() {

  if(fDescriptor != -1) close(fDescriptor);
}


// Communicate: Write and read from HV Board until fTimeOut has been reached 
//
// Returns: 0 error, 1 success, -1 fTimeOut exceeded
int HVBoard::Communicate(unsigned char* wbuf, int Bytes) {

  unsigned char rbuf[RBUF_LEN];
  int ret;
  fd_set SelectDescriptor;

  Status.BoardNumber = -1;
	
  // === Write data ===
  ret = write(fDescriptor, wbuf, Bytes);
  if (ret == -1) {
    printf("Error: Could not write data (%d/%s)\n", errno, strerror(errno));
    return 0;
  }
  else if (ret < Bytes) {
    printf("Error: Could write only %d of %d bytes\n", ret, Bytes);
    return 0;
  }
  if (Verbose) {
	printf("  %d byte(s) written: ", Bytes);
	for (int i=0; i<Bytes; i++) printf(" %#.2x", wbuf[i]);
	printf("\n");
  }

	// TODO
	// this is bad coding here. but if I do not wait here,
	// the text output of the arduino is not completely send, and only
	// part of the message is received .. 
	// I would be better to loop over select and red until select does
	// return after the timeout, the it is clear, that the arduino send all 
	// it wanted to send, and the whole 'Communicate' is rally finished.
	// but sleeping 2secs is okay for the moment	.
	// TODO
	sleep (2);

  // === Try to read until time-out ===
  FD_ZERO(&SelectDescriptor);   FD_SET(fDescriptor, &SelectDescriptor);
  struct timeval WaitTime = {(long) fTimeOut, (long) ((fTimeOut-(long) fTimeOut)*1e6)};
  if (select(fDescriptor+1, &SelectDescriptor, NULL, NULL, &WaitTime)==-1) {
    printf("Error with select() (%d/%s)\n", errno,strerror(errno));
    return 0;
  }

  // Time-out expired?
  if (!FD_ISSET(fDescriptor, &SelectDescriptor)) {
    printf("Time-out of %.2f seconds expired while reading\n", fTimeOut);
    return -1;
  }

  // Read error?    
  if ((ret = read(fDescriptor, rbuf, RBUF_LEN)) == -1) {
    printf("Read error (%d/%s)\n", errno, strerror(errno));
    return 0;
  }
  
  // Print result in hexadecimal and binary representation 
  if (Verbose) {
	printf("  %d byte(s) read: \n", ret);
	// in case rbuf has no '\0' at its last position, the printf would go berserk!
	// I put an '\0' at the end of the received bytes
	rbuf[ret]='\0';
	// in case the sender has put an '\0' in the first ret bytes, the output with printf is 
	// truncated ... so I go and exchange '\0' for '0'
	for ( int dom=0; dom<ret ; dom++){
		if (rbuf[dom]=='\0') 
			rbuf[dom]='0';
	}
	printf("%s\n",rbuf);
  }
	
  return 1;
}


// System reset of HV board
int HVBoard::SystemReset() {
  
  unsigned char wbuf[] = {0,0,0};
  return Communicate(wbuf, 3);
}


// Read channel status
int HVBoard::ReadChannel(int Chain, int Channel) {
  
  unsigned char wbuf[] = {1<<5 | Chain<<1 | (Channel&16)>>4, Channel<<4, 0};
  return Communicate(wbuf, 3);
}


// Global set
int HVBoard::GlobalSet(int Voltage) {
  
  unsigned char wbuf[] = {1<<6 , Voltage>>8, Voltage};
  return Communicate(wbuf, 3);
}


// Channel set
int HVBoard::ChannelSet(int Chain, int Channel, int Voltage) {
  
  unsigned char wbuf[] = {3<<5 |  Chain<<1 | (Channel&16)>>4, Channel<<4 | Voltage>>8, Voltage};
  return Communicate(wbuf, 3);
}


// Synchronize board
bool HVBoard::SynchBoard() {
  
  unsigned char wbuf = 0;
  int trial = 0, ret;
  
  while(++trial<=3) {
    if((ret = Communicate(&wbuf, 1)) == 1) return true;
    if (ret==0) break;
  }
  return false;
}
