/*
Test Code to make Arduino Ethernet communicate with PLD of Volker Commichaus
Bias Crate Controller of the FACt Bias voltage supply system
instead of FT245R interface

Interface to PLD or
something that behaves similarly:

Digital I/O:
Arduino Pin		:	Net Name	:		Int/Out/Bi	:  mnemonic
-----------------------------------------------------------------------------
Pin D2 	        	:	WR    	    	:		I		:  Write to 'FIFO'
Pin D3			:	RD#	        :		I		:  Read from 'FIFO'
Pin D8			:	TXE#	    	:		O		:  transmit enable
Pin D9			:	RXF#	    	:		O		:  Read from FIFO possible
Pin D4			:	D0	        :		BI
Pin D5			:	D1	        :		BI
Pin D6			:	D2	        :		BI
Pin D7			:	D3	        :		BI
Pin A0			:	D4	        :		BI		:  Data
Pin A1			:	D5	        :		BI
Pin A2			:	D6	        :		BI
Pin A3			:	D7	        :		BI
-----------------------------------------------------------------------------

port definition:
Port D consists of Arduino-Pins 0 - 7 and is used for Data exchange with the PLD
Port C consists of Arduino-Pins A0 - A5 and is used for Data exchange with the PLD
*/
//define non transferable variables
#define MAXBOARDS 13
#define EXISTINGBOARDS 10
#define CHPERBOARD 32
#define NUMCHANNELS EXISTINGBOARDS*CHPERBOARD
#define BLANKBYTE 0x00
#define READSTATEBYTE 0x20
#define GLOBALSETBYTE 0x40
#define CHANNELSETBYTE 0x60
#define FREE1 A5
#define FREE2 A4

//#define MAXVOLTAGE 80

typedef enum {
  set_message,
  send_message,
  get_message,
  pc_serial,
  parsing
} state_contr; //Type to define what microcontroller should do

typedef enum {
  channel_status,
  sys_reset,
  global_set,
  channel_set
} function_t; //Type to define which function of PLD shell be build

typedef enum {
  transmit_PLD_answer,
  get_request,
} communication; //Type to define communication with PC


const unsigned char wr  =  2;
const unsigned char rd  =  3;
const unsigned char txe =  8;
const unsigned char rxf =  9;
//variables to save Data-Bytes in
volatile unsigned char databus = 0x00; //temp. memory for Reading Databus interrupt
unsigned char message[3];	//field where the 3 Byte message is saved
unsigned char messageindex = 0;	//index of field
unsigned char incommingByte = 0;  //for incomming serial Data
unsigned int recent_voltages[NUMCHANNELS];  //The actual Voltage of each channel is saved in this variable
unsigned int set_voltage;                   //The Value to which channel shell be set is saved in the variable
boolean stat;                               //PLD Status-bit
unsigned char wrapcounter;                  //Wrap counter:
unsigned char channel;                      //channelnumber number

function_t function;
state_contr state_controler;
communication pc_com = get_request;

//unsigned char call_function;                //variable to call a funtion
unsigned char error;                        //Errorcode from PLD
unsigned char board;                        //board number
//define Variables that mark edge-detection on RD and WR pin
volatile boolean RisingEdge_RD = false;
volatile boolean risingEdge_WR = false;

void setup() {
    //delay(1000);
//Initiate Controlpin Usage & Stati
	pinMode( wr, INPUT);
	pinMode( rd, INPUT);
	digitalWrite(txe, HIGH);
	pinMode( txe, OUTPUT);
	digitalWrite(rxf, HIGH); //set initial state of RXF#
	pinMode( rxf, OUTPUT);
	digitalWrite(FREE1, HIGH);
    pinMode(FREE1, OUTPUT);
    pinMode(FREE2, INPUT);
    //define initial I/O of pins of Port D to Input

MakeDataBusInput();  // Binary: 00000000, since the bus is Bidirectional .. we better set to input in the idle case

//init test_values
board = 0x09;
channel = 0x0f;
function = channel_set;
set_voltage = 0xaaa;
state_controler = pc_serial; //just for development
digitalWrite(FREE1, LOW);

//define Interrupts for RD and WR
attachInterrupt(1, RisingEdgeOnRD, RISING);
attachInterrupt(0, RisingEdgeOnWR, RISING);

//Open comunication to PC
Serial.begin(9600); //opens serial port, sets data rate to 9600 bps
}


void loop() {
// Communication with the PC, handle PC requests



//Here the controller builds the message that will be send to PLD
        PORTC ^= 1<<PC5;
        PORTC ^= 1<<PC5;
    if ( state_controler == set_message){
        switch( function ){

            case  channel_status:
                channelStaus(board, channel);  //calls the channelstatus-function so values in message[] are set correctly
            break;

            case sys_reset:
                systemReset();                //calls the SystemReset-function so values in message[] are set correctly
            break;

            case global_set:
                globalSet(set_voltage);       //calls the GlobalSet-function so values in message[] are set correctly
            break;

            case channel_set:
                channelSet(board, channel, set_voltage);  //calls the channelSet-function so values in message[] are set correctly
            break;

        }
    state_controler = send_message; //just for development
    }
    if ( state_controler == send_message){
        //Serial.print ("sending message");
            //RisingEdge_RD = false;
    //This is the FIFO Read Cycle, PLD reads from FIFO, ARDUINO writes to FIFO
                PORTC ^= 1<<PC5;
                PORTC ^= 1<<PC5;
                PORTC ^= 1<<PC5;
                PORTC ^= 1<<PC5;
            digitalWrite(txe, HIGH); //Forbid PLD to send Datan, needs 4 microsec
            MakeDataBusOutput();             //set Databus as Output before you tell PLD that you have Data
             //This is the Statemachine where the 3 bytes will be set at PORTD
                switch( messageindex ){
                    case 0:
                    PutOnDataBus(message[messageindex]);
                    RisingEdge_RD = false;
                    digitalWrite(rxf, LOW ); //goes to low to tell PLD that Data is availiable, after PORTD to make shure that there is Data
                    while(!RisingEdge_RD){}
                    digitalWrite(rxf, HIGH );
                    messageindex++;
                    break;

                    case 1:
                    PutOnDataBus(message[messageindex]);
                    digitalWrite(rxf, LOW ); //goes to low to tell PLD that Data is availiable
                    while(!RisingEdge_RD){}
                    digitalWrite(rxf, HIGH );
                    messageindex++;
                    break;

                    case 2:
                    PutOnDataBus(message[messageindex]);
                    digitalWrite(rxf, LOW ); //goes to low to tell PLD that Data is availiable
                    while(!RisingEdge_RD ){}
                    digitalWrite(rxf, HIGH ); //goes to high to tell PLD that no Data is availiable
                    digitalWrite(txe, LOW  ); //set TXE# to low to permit PLD to send DATA
                    MakeDataBusInput(); //Set PORT D to Input: PLD can write DATA on that Port
                    messageindex = 0;
                    state_controler = get_message; //Listen to PLD for status report
                    break;
                }
                RisingEdge_RD = false;

    }

//The FIFO Write Cycle
    if ( state_controler == get_message){
    digitalWrite(rxf, HIGH );   //goes to high to tell PLD that no Data is availiable
    MakeDataBusInput(); //set Port D as Input
 	digitalWrite(txe, LOW );	//goes to low to tell PLD it can send Data
        if (risingEdge_WR){  //write Data from Byte into FiFo when WR goes from high to low and there was an Rising Edge before
            switch( messageindex ){
                        case 0:
                        message[messageindex] = databus;
                        messageindex++;
                        break;

                        case 1:
                        message[messageindex] = databus;
                        messageindex++;
                        break;

                        case 2:
                        message[messageindex] = databus;
                        messageindex = 0;
                        state_controler = pc_serial;
                        pc_com = transmit_PLD_answer;
                        break;
            }
            risingEdge_WR = false; //reset RisingEdge interrupt variable to false to detect incomming Edge
            digitalWrite(txe, HIGH); //reset TXE to tell PLD not not to write

        }
    }

  // delay(1000); //slow down process for test pourpose

// Communication with PC via USB
    if ( state_controler == pc_serial){

    // Outgoing Commands
    switch ( pc_com ){
        //report PLD response to PC
        case transmit_PLD_answer:
            Serial.println("PLD answered: ");
            for (messageindex = 0; messageindex < 3; messageindex++){
            Serial.print(message[messageindex], BIN);
            Serial.print( "\t" );
            }
            Serial.println(" in BIN ");
            for (messageindex = 0; messageindex < 3; messageindex++){
            Serial.print(message[messageindex], HEX);
            Serial.print( "\t" );
            }
            messageindex = 0;
            Serial.println(" in HEX ");
            Serial.println( "_______________________ " );
            pc_com = get_request;
            break;

    // Incoming Commands
        case get_request:
            while (Serial.available() < 3){};
            if (Serial.available() > 0){
                Serial.print("I received: ");
                messageindex = 0;
                while (messageindex < 3){
                message[messageindex] = Serial.read();
                Serial.print( "0x" );
                Serial.print(message[messageindex], HEX);
                messageindex++;
                Serial.print( " " );
                }
                messageindex = 0;
                //Serial.print(incommingByte, BYTE);
                Serial.println(",du Nase!");
                }
                state_controler = send_message;
            break;

    }

    }

}


/*
This are the funtions that can be used with the board:
*/
//Read Data from DATABUS
unsigned char ReadFromDataBus(){
	// The eight bits of the databus are distributed over
	// the two differen ports of the ATmega. Port C and D
	// actually they are connected like this:
	// data0...data3 are connected to Arduino Pins D4..D7
	// data4...data7 are connected to Arduino Pins C0..D3
	return  ( (PINC&0x0f) << 4 ) | ( (PIND&0xf0) >> 4)  ;
}

//Write Data from DATABUS
void PutOnDataBus ( unsigned char data) {
	// okay .. .again
	// data0..3 is connected to PD4..7
	// data4..7 is connected to PC0..3
	PORTC = (PORTC & 0xf0) | (data&0xf0)>>4 ;
	PORTD = (PORTD & 0x0f) | (data&0x0f)<<4 ;

}

void MakeDataBusOutput() {
  // making the databus out can be done like this:
	// data0..3 is connected to PD4..7
	// data4..7 is connected to PC0..3

  // also auf PORTC die bits 0 bis 3 setzen!
  DDRC |= 0x0f;
  // und auf PORTD die bits 4 bis 7 setzen!
  DDRD |= 0xf0;

  // Um die Pullups muss man sicht nicht kümmern, da nach dieser Funktion sowieso ein neuer Wert auf den Bus
  // geschrieben wird.
}

void MakeDataBusInput() {
// see the comments in MakeDataBusOutput()

//first we switch the outs to ins
// then we switch on the pullups

  DDRC &= 0xf0;
  DDRD &= 0x0f;
}

//System Reset
void systemReset(){
  for (messageindex = 0; messageindex < 2; messageindex++){
  message[messageindex] = 0x0; // set alle 3byte to 0
  }
}

//Read channel Status & Current
void channelStaus(unsigned char brd, unsigned char chan){
  message[0] = READSTATEBYTE | (brd << 1);    //put funtion, board in first byte
  message[0] = message[0] | (chan >> 4);      //add first channelbit to end of first byte
  message[1] = BLANKBYTE | (chan << 4);       //put rest of channelbit in byte2
  message[2] = BLANKBYTE;     // set byte 3 to 0
}

//Global set
void globalSet(unsigned int volt){
  message[0] = GLOBALSETBYTE;
  message[1] = BLANKBYTE | (volt >> 8);
  message[2] = BLANKBYTE | volt;
}

//Channel set to Voltage
void channelSet(unsigned char brd, unsigned char chan, unsigned int volt){
  message[0] = CHANNELSETBYTE | (brd << 1);
  message[0] = message[0] | (chan >> 4);
  message[1] = BLANKBYTE | (chan << 4);
  message[1] = message[1] | (volt >> 4);
  message[2] = BLANKBYTE | volt;
}

//functions for Interrupts on RD and WR
void RisingEdgeOnRD(){
    //digitalWrite(FREE1, HIGH);
    //digitalWrite(FREE1, LOW);

    PORTC ^= 1<<PC5;
    PORTC ^= 1<<PC5;
    PORTC ^= 1<<PC5;
    PORTC ^= 1<<PC5;
    PORTC ^= 1<<PC5;
    PORTC ^= 1<<PC5;

    RisingEdge_RD = true;
}

void RisingEdgeOnWR(){
    databus = ReadFromDataBus();
    risingEdge_WR = true;
    PORTC ^= 1<<PC5;
    PORTC ^= 1<<PC5;
    PORTC ^= 1<<PC5;
    PORTC ^= 1<<PC5;
    PORTC ^= 1<<PC5;
    PORTC ^= 1<<PC5;
    PORTC ^= 1<<PC5;
    PORTC ^= 1<<PC5;

}
/*
void CommandToPLD(unsigned int txe, unsigned int rxf ){
//txe auf D8, rxf auf D9
if (txe == HIGH){
	if (rxf == HIGH){
		PORTB = B00000011;
	}
	else if(rxf == LOW){
		PORTB = B00000001;
	}
}
else if (txe ==
PORTB = PORTB;
}
*/
