/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz    1/2002 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Wolfgang Wittek 1/2002 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MHAlphaEnergyTheta                                                       //
//                                                                          //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MHAlphaEnergyTheta.h"

#include <TCanvas.h>

#include "MMcEvt.hxx"
#include "MHillasSrc.h"
#include "MEnergyEst.h"

#include "MBinning.h"
#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHAlphaEnergyTheta);

// --------------------------------------------------------------------------
//
// Default Constructor. It sets name and title only. Typically you won't
// need to change this.
//
MHAlphaEnergyTheta::MHAlphaEnergyTheta(const char *name, const char *title)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHAlphaEnergyTheta";
    fTitle = title ? title : "3-D histogram in alpha, energy and theta";

    fHist.SetDirectory(NULL);

    fHist.GetXaxis()->SetTitle("\\alpha [\\circ]");
    fHist.GetYaxis()->SetTitle("E_{est} [GeV]");
    fHist.GetZaxis()->SetTitle("\\Theta [\\circ]");
}

Bool_t MHAlphaEnergyTheta::SetupFill(const MParList *plist)
{
   fEnergy = (MEnergyEst*)plist->FindObject("MEnergyEst");
   if (!fEnergy)
   {
       *fLog << err << dbginf << "MEnergyEst not found... aborting." << endl;
       return kFALSE;
   }

   fMcEvt = (MMcEvt*)plist->FindObject("MMcEvt");
   if (!fMcEvt)
   {
       *fLog << err << dbginf << "MMcEvt not found... aborting." << endl;
       return kFALSE;
   }

   MBinning* binsenergy = (MBinning*)plist->FindObject("BinningE");
   MBinning* binsalpha  = (MBinning*)plist->FindObject("BinningAlpha");
   MBinning* binstheta  = (MBinning*)plist->FindObject("BinningTheta");
   if (!binsenergy || !binsalpha || !binstheta)
   {
       *fLog << err << dbginf << "At least one MBinning not found... aborting." << endl;
       return kFALSE;      
   }

   SetBinning(&fHist, binsalpha, binsenergy, binstheta);

   return kTRUE;
}

Bool_t MHAlphaEnergyTheta::Fill(const MParContainer *par)
{
    MHillasSrc &hil = *(MHillasSrc*)par;

    fHist.Fill(hil.GetAlpha(), fEnergy->GetEnergy(), fMcEvt->GetTheta()*kRad2Deg);

    return kTRUE;
}

void MHAlphaEnergyTheta::Draw(Option_t *opt)
{
    if (!gPad)
        MakeDefCanvas("AlphaEnergyTheta", "Distrib of alpha, E, Theta");

    gPad->Divide(2,2);

    TH1 *h;

    gPad->cd(1);
    h = fHist.Project3D("x");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    gPad->cd(2);
    h = fHist.Project3D("y");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    gPad->cd(3);
    h = fHist.Project3D("z");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    gPad->cd(4);
    fHist.Draw(opt);

    gPad->Modified();
    gPad->Update();
}

TObject *MHAlphaEnergyTheta::DrawClone(Option_t *opt) const
{
    TCanvas *c = MakeDefCanvas("AlphaEnergyTheta", "Distrib of alpha, E, Theta");
    c->Divide(2, 2);

    gROOT->SetSelectedPad(NULL);

    //
    // FIXME: ProjectionX,Y is not const within root
    //
    TH1 *h;

    c->cd(1);
    h = ((TH3*)(&fHist))->Project3D("x");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    c->cd(2);
    h = ((TH3*)(&fHist))->Project3D("y");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    c->cd(3);
    h = ((TH3*)(&fHist))->Project3D("z");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    c->cd(4);
    ((TH3*)(&fHist))->DrawCopy(opt);

    c->Modified();
    c->Update();

    return c;
}

void MHAlphaEnergyTheta::Substract(const TH3D *h1, const TH3D *h2)
{
    MH::SetBinning(&fHist, (TH1*)h1);

    fHist.Sumw2();
    fHist.Add((TH1*)h1, (TH1*)h2, 1, -1); // Root: FIXME
}

void MHAlphaEnergyTheta::SetAlphaRange(Axis_t lo, Axis_t up)
{
    TAxis &axe = *fHist.GetXaxis();

    //
    // FIXME: ROOT Binning??? of projection
    // root 3.02: SetRangeUser
    axe.SetRange(axe.FindFixBin(lo), axe.FindFixBin(up));
}

TH2D *MHAlphaEnergyTheta::GetAlphaProjection(Axis_t lo, Axis_t up)
{
    SetAlphaRange(lo, up);
    return (TH2D*)fHist.Project3D("yz");
}
