/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz    1/2002 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Wolfgang Wittek 1/2002 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MHAlphaEnergyTime                                                       //
//                                                                          //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MHAlphaEnergyTime.h"

#include <TCanvas.h>

#include "MHillasSrc.h"
#include "MEnergyEst.h"
#include "MTime.h"

#include "MBinning.h"
#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHAlphaEnergyTime);


// --------------------------------------------------------------------------
//
// Default Constructor. It sets name and title only. Typically you won't
// need to change this.
//
#include <iostream.h>
MHAlphaEnergyTime::MHAlphaEnergyTime(const char *name, const char *title) 
  : fHist()
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHAlphaEnergyTime";
    fTitle = title ? title : "3-D histogram in alpha, energy and time";

    fHist.SetDirectory(NULL);

    fHist.GetXaxis()->SetTitle("\\alpha [\\circ]");
    fHist.GetYaxis()->SetTitle("E_{est} [GeV]");
    fHist.GetZaxis()->SetTitle("t [s]");
}

Bool_t MHAlphaEnergyTime::SetupFill(const MParList *plist)
{
   fEnergy = (MEnergyEst*)plist->FindObject("MEnergyEst");
   if (!fEnergy)
   {
       *fLog << err << dbginf << "MEnergyEst not found... aborting." << endl;
       return kFALSE;
   }

   fTime = (MTime*)plist->FindObject("MTime");
   if (!fTime)
   {
       *fLog << err << dbginf << "MTime not found... aborting." << endl;
       return kFALSE;
   }

   MBinning* binsenergy = (MBinning*)plist->FindObject("BinningE");
   MBinning* binsalpha  = (MBinning*)plist->FindObject("BinningAlpha");
   MBinning* binstime   = (MBinning*)plist->FindObject("BinningTime");
   if (!binsenergy || !binsalpha || !binstime)
   {
       *fLog << err << dbginf << "At least one MBinning not found... aborting." << endl;
       return kFALSE;      
   }

   SetBinning(&fHist, binsalpha, binsenergy, binstime);

   fHist.Sumw2();

   return kTRUE;
}

Bool_t MHAlphaEnergyTime::Fill(const MParContainer *par)
{
    MHillasSrc &hil = *(MHillasSrc*)par;

    fHist.Fill(hil.GetAlpha(), fEnergy->GetEnergy(), 0.0001*fTime->GetTimeLo());
    return kTRUE;
}

void MHAlphaEnergyTime::Draw(Option_t *opt)
{
    if (!gPad)
        MakeDefCanvas("AlphaEnergyTime", "Distrib of alpha, E, t");

    gPad->Divide(2,2);

    TH1 *h;

    gPad->cd(1);
    h = fHist.Project3D("x");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    gPad->cd(2);
    h = fHist.Project3D("y");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    gPad->cd(3);
    h = fHist.Project3D("z");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    gPad->cd(4);
    fHist.Draw(opt);

    gPad->Modified();
    gPad->Update();
}

TObject *MHAlphaEnergyTime::DrawClone(Option_t *opt) const
{
    TCanvas *c = MakeDefCanvas("AlphaEnergyTime", "Distrib of alpha, E, t");
    c->Divide(2, 2);

    gROOT->SetSelectedPad(NULL);

    //
    // FIXME: ProjectionX,Y is not const within root
    //
    TH1 *h;

    c->cd(1);
    h = ((TH3D*)(&fHist))->Project3D("x");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    c->cd(2);
    h = ((TH3D*)(&fHist))->Project3D("y");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    c->cd(3);
    h = ((TH3D*)(&fHist))->Project3D("z");
    h->Draw(opt);
    h->SetBit(kCanDelete);

    c->cd(4);
    ((TH3D*)(&fHist))->DrawCopy(opt);

    c->Modified();
    c->Update();

    return c;
}

void MHAlphaEnergyTime::Substract(const TH3D *h1, const TH3D *h2)
{
    MH::SetBinning(&fHist, (TH1*)h1);

    fHist.Sumw2();
    fHist.Add((TH1*)h1, (TH1*)h2, 1, -1); // ROOT: FIXME!
}

void MHAlphaEnergyTime::SetAlphaRange(Axis_t lo, Axis_t up)
{
    TAxis &axe = *fHist.GetXaxis();

    //
    // FIXME: ROOT Binning??? of projection
    // root 3.02: SetRangeUser
    axe.SetRange(axe.FindFixBin(lo), axe.FindFixBin(up));
}

TH2D *MHAlphaEnergyTime::GetAlphaProjection(Axis_t lo, Axis_t up)
{
    SetAlphaRange(lo, up);
    return (TH2D*)fHist.Project3D("yz");
}
