#include "DataCalib.h"

#include "FAD.h"
#include "FitsFile.h"
#include "DimDescriptionService.h"

#include "externals/fits.h"

using namespace std;

DrsCalibration DataCalib::fData;
bool DataCalib::fProcessing = false;
vector<float> DataCalib::fStats(1440*1024*6+3);

void DataCalib::Restart()
{
    fData.Clear();

    reinterpret_cast<uint32_t*>(fStats.data())[0] = 0;
    reinterpret_cast<uint32_t*>(fStats.data())[1] = 0;
    reinterpret_cast<uint32_t*>(fStats.data())[2] = 0;

    int i=0;
    while (i<1024*1440*2+3)  // Set mean and RMS to 0
        fStats[i++] = 0;
    while (i<1024*1440*3+3)
        fStats[i++] = 2000./4096; // Set mean to 0.5
    while (i<1024*1440*6+3)
        fStats[i++] = 0;   // Set everything else to 0

    fProcessing = false;
}

void DataCalib::Update(DimDescribedService &dim)
{
    dim.Update(fStats);
}

bool DataCalib::Open(RUN_HEAD* h)
{
    if (h->NPix != 1440)
    {
        fMsg.Error("Number of pixels in header not 1440.");
        return false;
    }

    if (h->Nroi != 1024)
    {
        fMsg.Error("Region of interest not 1024.");
        return false;
    }

    if (fProcessing)
    {
        fMsg.Warn("Previous DRS calibration run not yet finished!");
        return false;
    }

    if (fData.fStep==4)
    {
        fMsg.Warn("DRS Calibration already finished... please restart!");
        return false;
    }

    fProcessing = true;

    ostringstream name;
    name << "drs-calib-" << fData.fStep;
    fFileName = name.str();

    Reset();
    InitSize(1440, 1024);

    return DataWriteFits::Open(h);
}

bool DataCalib::WriteEvt(EVENT *e)
{
    // FIXME: SET StartPix to 0 if StartPix is -1

    if (fData.fStep==0)
    {
        AddRel(e->Adc_Data, e->StartPix);
    }
    if (fData.fStep==1)
    {
        AddRel(e->Adc_Data, e->StartPix, fData.fOffset.data(), fData.fNumOffset);
    }
    if (fData.fStep==2)
    {
        AddAbs(e->Adc_Data, e->StartPix, fData.fOffset.data(), fData.fNumOffset);
    }

    return DataWriteFits::WriteEvt(e);
}

bool DataCalib::ReadFits(const string &str, MessageImp &msg)
{
    if (fProcessing)
    {
        msg.Error("Reading "+str+" failed: DRS calibration in process.");
        return false;
    }

    try
    {
        const string txt = fData.ReadFitsImp(str, fStats);
        if (txt.empty())
            return true;

        msg.Error(txt);
        return false;
    }
    catch (const runtime_error &e)
    {
        msg.Error("Exception reading "+str+": "+e.what());
        return false;
    }
}

void DataCalib::WriteFits()
{
#ifdef HAVE_FITS
    FitsFile file(fMsg);

    file.AddColumn('I', "RunNumberBaseline");
    file.AddColumn('I', "RunNumberGain");
    file.AddColumn('I', "RunNumberTriggerOffset");
    file.AddColumn('F', "BaselineMean",      1024*1440, "mV");
    file.AddColumn('F', "BaselineRms",       1024*1440, "mV");
    file.AddColumn('F', "GainMean",          1024*1440, "mV");
    file.AddColumn('F', "GainRms",           1024*1440, "mV");
    file.AddColumn('F', "TriggerOffsetMean", 1024*1440, "mV");
    file.AddColumn('F', "TriggerOffsetRms",  1024*1440, "mV");

    fFileName = FormFileName("drs.fits");

    if (!file.OpenFile(fFileName))
        return;

    if (!file.OpenTable("DrsCalibration"))
        return;

    if (!file.WriteDefaultKeys("fadctrl"))
        return;

    if (!file.WriteKeyNT("STEP",     fData.fStep, "") ||
        !file.WriteKeyNT("ADCRANGE", 2000,             "Dynamic range of the ADC in mV") ||
        !file.WriteKeyNT("DACRANGE", 2500,             "Dynamic range of the DAC in mV") ||
        !file.WriteKeyNT("ADC",      12,               "Resolution of ADC in bits")      ||
        !file.WriteKeyNT("DAC",      16,               "Resolution of DAC in bits")      ||
        !file.WriteKeyNT("DACLEVEL", 50000,            "Applied DAC level in counts")    ||
        !file.WriteKeyNT("NBOFFSET", fData.fNumOffset,       "Number of entries for offset calibration") ||
        !file.WriteKeyNT("NBGAIN",   fData.fNumGain/1953125, "Number of entries for gain calibration")   ||
        !file.WriteKeyNT("NBTRGOFF", fData.fNumTrgOff,       "Number of entries for trigger offset calibration") ||
        !file.WriteKeyNT("NPIX",     1440,             "Number of channels in the camera") ||
        !file.WriteKeyNT("NROI",     1024,             "Region of interest")
       )
        return;

    vector<char> buf;
    buf.reserve(fStats.size()*sizeof(float));

    char *src  = reinterpret_cast<char*>(fStats.data());
    char *end  = reinterpret_cast<char*>(fStats.data()+1024*1440*6+3);
    char *dest = buf.data();

    while (src<end)
    {
        reverse_copy(src, src+sizeof(float), dest);
        src  += sizeof(float);
        dest += sizeof(float);
    }

    if (!file.AddRow())
        return;

    if (!file.WriteData(buf.data(), 1024*1440*sizeof(float)*6+3))
        return;

    ostringstream str;
    str << "Wrote DRS calibration data (step=" << fData.fStep << ") to '" << fFileName << "'";
    Info(str.str());
#endif
}

bool DataCalib::Close(RUN_TAIL *tail)
{
    if (fData.fStep==0)
    {
        fData.fOffset.assign(fSum.begin(), fSum.end());
        fData.fNumOffset = fNumEntries;

        for (int i=0; i<1024*1440; i++)
            fData.fGain[i] = 4096*fNumEntries;

        // Scale ADC data from 12bit to 2000mV
        GetSampleStats(fStats.data()+3, 2000./4096);
        reinterpret_cast<uint32_t*>(fStats.data())[0] = GetRunId();;
    }
    if (fData.fStep==1)
    {
        fData.fGain.assign(fSum.begin(), fSum.end());
        fData.fNumGain = fNumEntries;

        // DAC:  0..2.5V == 0..65535
        // V-mV: 1000
        //fNumGain *= 2500*50000;
        //for (int i=0; i<1024*1440; i++)
        //    fGain[i] *= 65536;
        fData.fNumGain *= 1953125;
        for (int i=0; i<1024*1440; i++)
            fData.fGain[i] *= 1024;

        // Scale ADC data from 12bit to 2000mV
        GetSampleStats(fStats.data()+1024*1440*2+3, 2000./4096/fData.fNumOffset);//0.5);
        reinterpret_cast<uint32_t*>(fStats.data())[1] = GetRunId();;
    }
    if (fData.fStep==2)
    {
        fData.fTrgOff.assign(fSum.begin(), fSum.end());
        fData.fNumTrgOff = fNumEntries;

        // Scale ADC data from 12bit to 2000mV
        GetSampleStats(fStats.data()+1024*1440*4+3, 2000./4096/fData.fNumOffset);//0.5);
        reinterpret_cast<uint32_t*>(fStats.data())[2] = GetRunId();;
    }

    if (fData.fStep<=2)
        WriteFits();

    fDim.Update(fStats);

    fData.fStep++;

    fProcessing = false;

    return DataWriteFits::Close(tail);
}
