// Opens FACT Fits Raw Data  -- for more Info refer to header file
size_t openDataFits(
	const char *datafilename,		// path to fits file containing FACT raw data
	fits * * datafile,				// ptr to pointer, where to return the fits object
	vector<int16_t> &Data,			// vector, which will be filled with raw data
	vector<int16_t> &StartCells,	// vector, which will be filled with DRS start positions
	unsigned int &EventID,			// index of the current event
	UInt_t &RegionOfInterest,		// Width of the Region, read out of the DRS
	UInt_t &NumberOfPixels,			// Total number of pixel, read out of the camera
	size_t &PXLxROI,				// Size of column "Data" = #Pixel x ROI
			// this can be used, to x-check RegionOfInterest and NumberOfPixels
	int VerbosityLevel				//
) {
	size_t NumberOfEvents;
	*datafile = new fits(datafilename);
	if (!(*(*datafile))) {
		if (VerbosityLevel > 0)
			cout << "Couldn't properly open the datafile: " << datafilename << endl;
		return 0;
	}

	NumberOfEvents = (*datafile)->GetNumRows();
	if (NumberOfEvents < 1){
		if (VerbosityLevel > 0){
			cout << "Warning in FOpenDataFile of file: " << datafilename << endl;
			cout << "the file contains no events." << endl;
		}
	}

	RegionOfInterest = (*datafile)->GetUInt("NROI");
	NumberOfPixels = (*datafile)->GetUInt("NPIX");

	// I am not really sure, why anybody should be interested in this number
	// It can be used for the sanity check, which follows only ... I guess.
	PXLxROI = (*datafile)->GetN("Data");

	if ( RegionOfInterest * NumberOfPixels != PXLxROI) // something in the file went wrong
	{
		if (VerbosityLevel > 0){
			cout << "Warning in FOpenDataFile of file: " << datafilename << endl;
			cout << "RegionOfInterest * NumberOfPixels != PXLxROI" << endl;
			cout << "--> " << RegionOfInterest;
			cout << " * " << NumberOfPixels;
			cout << " = " << RegionOfInterest * NumberOfPixels;
			cout << ", but PXLxROI =" << PXLxROI << endl;
		}
		return 0;
	}

	//Set the sizes of the data vectors
	Data.resize(PXLxROI, 0);
	StartCells.resize(NumberOfPixels, 0);

	//Link the data to variables
	(*datafile)->SetRefAddress("EventNum", EventID);
	(*datafile)->SetVecAddress("Data", Data);
	(*datafile)->SetVecAddress("StartCellData", StartCells);

	return NumberOfEvents;
}

size_t openCalibFits(
	const char *FileName,			// path to fits file
	vector<float> &Offset,
	vector<float> &Gain,
	vector<float> &TriggerOffset,
	size_t &TriggerOffsetROI,
	int VerbosityLevel	
)
{
	size_t NumberOfEvents = 0;

	// sizes of the data vectors
	size_t Offset_size;
	size_t Gain_size;
	size_t TriggerOffset_size;

	fits * datafile = new fits( FileName );
	if (!(*datafile)) {
		if (VerbosityLevel > 0)
			cout << "Couldn't properly open the file: " << FileName << endl;
		return 0;
	}

	NumberOfEvents = datafile->GetNumRows();
	if (NumberOfEvents < 1){
		if (VerbosityLevel > 0){
			cout << "Warning during open of file: " << FileName << endl;
			cout << "the file contains no events." << endl;
			return 0;
		}
	}

	// only if the user really wants, this stuff is output
	if (VerbosityLevel > 1){
 	 cout << "----------------- DRS Calib Header -----------------" << endl;
 	 datafile->PrintKeys();
 	 cout << "---------------- DRS Calib Columns -----------------" << endl;
 	 datafile->PrintColumns();
	}

	Offset_size = datafile->GetN("BaselineMean");
	Gain_size = datafile->GetN("GainMean");
	TriggerOffset_size = datafile->GetN("TriggerOffsetMean");

  Offset.resize(Offset_size, 0);
  Gain.resize(Gain_size, 0);
  TriggerOffset.resize(TriggerOffset_size, 0);

	TriggerOffsetROI = (size_t)datafile->GetUInt("NROI");
  datafile->SetVecAddress("BaselineMean", Offset);
  datafile->SetVecAddress("GainMean", Gain);
  datafile->SetVecAddress("TriggerOffsetMean", TriggerOffset);
  datafile->GetRow(0); //Read the calibration data

  return NumberOfEvents;
}
