/********************************************************************\

  Alarm handler of the Evidence Control System

  - Checks periodically if all required servers are up
    (later it should try to start them if not)
  - Listens to the 'Status' service of each server. The server state 
    is published as a DIM service.
    
  Oliver Grimm, September 2009

\********************************************************************/

#define SERVER_NAME "Alarm"
#include "../Evidence.h"

//
// Data handling class
//
class AlarmHandler : public DimClient, public EvidenceServer {
    
    DimStampedInfo **StatusService;
    void infoHandler();

  public:
    AlarmHandler();
    ~AlarmHandler();
    
    char **Server;
    unsigned int NumServers;
    char *StateString;
    char *ServerList;  
}; 

// Constructor
AlarmHandler::AlarmHandler(): EvidenceServer(SERVER_NAME) {

  char *ServerNames = GetConfig(SERVER_NAME " servers");

  // Copy original list of servers to observe
  ServerList = new char [strlen(ServerNames)+1];
  strcpy(ServerList, ServerNames);
  
  // Extract DIM servers to observe
  Server = new char* [strlen(ServerNames)];
  NumServers = 0;
  char *NextToken = strtok(ServerNames, " \t");
  while (NextToken != NULL) {
    Server[NumServers++] = NextToken; // Subscribe with handler
    NextToken = strtok(NULL, " \t");     
  }

  // Subscribe with handler to 'Status' service of all servers
  StatusService = new DimStampedInfo* [NumServers];
  
  for (int i=0; i<NumServers; i++) {
    char *Buffer = new char [strlen(Server[i])+10];
    strcpy(Buffer, Server[i]);
    strcat(Buffer, "/Status");
    StatusService[i] = new DimStampedInfo(Buffer, 0, this);
    printf("Subscribed to %s\n", Buffer);
    delete[] Buffer; 
  }
  
  StateString = new char [NumServers+1];
  for (int i=0; i<NumServers; i++) StateString[i] = '1';
  StateString[NumServers] = '\0';
}

// Destructor
AlarmHandler::~AlarmHandler() {

  for (int i=0; i<NumServers; i++) delete StatusService[i];
  delete[] StatusService;
  delete[] Server;
  delete[] ServerList;
}

// Print messages of status changes to screen
void AlarmHandler::infoHandler() {

  // Ignore empty messages
  if (strlen(getInfo()->getString()) == 0) return;
  
  // Print message
  time_t RawTime = getInfo()->getTimestamp();
  struct tm *TM = localtime(&RawTime);
  printf("%s (%02d:%02d:%02d): %s\n", getInfo()->getName(), TM->tm_hour,
	TM->tm_min, TM->tm_sec, getInfo()->getString());

  // Update status string
  for (int i=0; i<NumServers; i++) {
    if (strcmp(getInfo()->getName(),Server[i]) == 0) {
      StateString[i] = '2';
    }
  }  
}


//	    
// Main program
//
int main() {
    
  DimBrowser Browser;
  char *ServerName, *Node;
  bool Exists;
  
  // Static declaration ensures calling of destructor by exit()
  static AlarmHandler Alarm; 
  
  // Request configuration data
  unsigned int Period = atoi(Alarm.GetConfig(SERVER_NAME " period"));

  // Create DIM services
  static DimService ServerState(SERVER_NAME"/State", Alarm.StateString);
  static DimService ServerList(SERVER_NAME"/Servers", Alarm.ServerList);

  // Check periodically if servers are up
  while(!EvidenceServer::ExitRequest) {
    for (int i=0; i<Alarm.NumServers; i++) {
      Exists = false;
      Browser.getServers();
      while (Browser.getNextServer(ServerName, Node) == 1) {
        if (strcmp(ServerName, Alarm.Server[i]) == 0) Exists = true;
      }
      if (!Exists) {
        Alarm.Msg(Alarm.WARN, "Server %s unavailable", Alarm.Server[i]);
	Alarm.StateString[i] = '0';
      }
      else if (Alarm.StateString[i] == '0') Alarm.StateString[i] = '1';
    }
    
    ServerState.updateService(Alarm.StateString);
    sleep(Period);
  }
}
