/********************************************************************\

  General code to start a server of the Evidence Control System
  
  - The server is started with the given name.
  - DIM exit and error handlers are implemented.
  - The Status service is published.
    It can be updated with the Msg() method. The text will also be logged.
  - Configuration data can be requested by GetConfig(). 
  - Signal handlers to ignore common signals are installed.
    These signals will then cause pause() to return which can be used
	by the application to terminate gracefully.
  - The static method ToString() converts the contents of a
    DIMInfo service into text
	
  All memory allocated by the non-static methods will be freed by the
  class destructor.
  
  Oliver Grimm, December 2009
 
\********************************************************************/

#include "Evidence.h"

bool EvidenceServer::ExitRequest = false;

// Constructor starts server with given name
EvidenceServer::EvidenceServer(const char *Name) {

  // Initialize
  Status = NULL;
  MsgBuffer = NULL;
  ConfigList = NULL;
  ConfigNum = 0;
  
  // Catch some signals
  signal(SIGQUIT, &SignalHandler);  // CTRL-Backspace
  signal(SIGTERM, &SignalHandler);  // Termination signal
  signal(SIGINT, &SignalHandler);   // CTRL-C
  signal(SIGHUP, &SignalHandler);   // Terminal closed
  
  // Create server name
  if (asprintf(&ServerName, "%s/Status", Name) == -1) {
    ServerName = NULL;
    Msg(FATAL, "Could not generate service name, asprintf() failed");
  }
  
  // Start server
  static char InitMsg[] = "OK" "\0" "0";
  Status = new DimService(ServerName, (char *) "C", InitMsg, sizeof(InitMsg));
  DimServer::start(Name);
  DimServer::addExitHandler(this);
  
  Msg(INFO, "Server started");
}

// Destructor: Frees all buffers
EvidenceServer::~EvidenceServer() {

  free(ServerName);
  free(MsgBuffer);
  
  for (unsigned int i=0; i<ConfigNum; i++) {
    delete *((DimRpcInfo **) ConfigList + i);
  }
  free(ConfigList);
}

// DIM exit handler
void EvidenceServer::exitHandler(int Code) {
  Msg(INFO, "Server stopped (DIM exit code %d)", Code);
  exit(EXIT_SUCCESS);
}

// DIM error handler
void EvidenceServer::errorHandler(int Severity, int Code, char *Message) {   
  Msg(ERROR, "%s (DIM Error, code %d, severity %d)\n", Message, Code, Severity);
}

// Print message to screen, to log and to status
//
// The message format is special: after the string-terminating '\0'
// the Severity is given, terminated by another '\0' 
// The buffer must have the same lifetime as the DIM service.
// If Severity is FATAL, exit() will be invoked.
void EvidenceServer::Msg(MsgType Severity, const char *Format, ...) {

  char *TmpBuffer;
  int Size;
  
  static char DefaultMsg[]="vasprintf() or asprintf() failed, cannot process message" "\0" "2";   // Severity 'Error'
   
  // Assemble message from application
  va_list ArgumentPointer;
  va_start(ArgumentPointer, Format);
  if (vasprintf(&TmpBuffer, Format, ArgumentPointer) == -1) TmpBuffer = NULL;
  va_end(ArgumentPointer);

  // Free previous message buffer if allocated dynamically
  if (MsgBuffer != DefaultMsg) free(MsgBuffer);
  
  // Assemble final message including headers and Severity encoding
  if (TmpBuffer == NULL) Size = -1;
  else Size = asprintf(&MsgBuffer, "%s: %s%s*%d", ServerName,
		(Severity==FATAL) ? "(Fatal) ":"", TmpBuffer,
		(unsigned char) Severity);
  free(TmpBuffer);

  // If memory could not be allocated, show default message,
  // otherwise replace '*' by '0' for severity encoding		
  if (Size == -1) {
    MsgBuffer = DefaultMsg;
	Size = sizeof(DefaultMsg) - 1;
  }
  else *(strrchr(MsgBuffer, '*')) = '\0';

  // Send message to console, log file and update DIM status service
  printf("%s\n", MsgBuffer);
  DimClient::sendCommand("DColl/Log", MsgBuffer);
  if (Status != NULL) Status->updateService(MsgBuffer, Size+1);

  // Terminate if message type is fatal
  if (Severity == FATAL) exit(EXIT_FAILURE);
}

// Get configuration data (program terminates if data is missing)
//
// The memory allocated by all calls to this function will be freed by
// the destructor.
char* EvidenceServer::GetConfig(const char *Item) {

  // Enlarge memory to hold new pointer
  DimRpcInfo **Config = (DimRpcInfo **) realloc(ConfigList, sizeof(void *) * (++ConfigNum));
  if (Config == NULL) {
    Msg(WARN, "Could not realloc() memory for DimRpcInfo list (%s)", strerror(errno));
    ConfigNum--;  
  }
  else ConfigList = (void *) Config;
  
  // Advance array pointer to position for new DimRpcInfo pointer 
  Config += ConfigNum - 1;
  
  // Make configuration request
  *Config = new DimRpcInfo((char *) "ConfigRequest", (char *) "");
  (*Config)->setData((char *) Item);

  // Terminate if not successful
  if (strlen((*Config)->getString()) == 0) {
    Msg(FATAL, "Missing configuration data '%s'", Item);
  }

  return (*Config)->getString();
}


// ====== Static methods ======

// Signal handler (causes pause() to return)
void EvidenceServer::SignalHandler(int) {
  EvidenceServer::ExitRequest = true;
}

// Translates DIMInfo to string (memory has to be freed by caller)
char *EvidenceServer::ToString(DimInfo *Item) {

  char *Text;
  int R;
  
  // Cannot handle complex data formats
  if (strlen(Item->getFormat()) != 1) return NULL;
  
  // Interpret format
  switch (*(Item->getFormat())) {
    case 'I':  R = asprintf(&Text, "%d", Item->getInt());   break;
    case 'C':  R = asprintf(&Text, "%s", Item->getString());   break;
    case 'S':  R = asprintf(&Text, "%hd", Item->getShort());   break;
    case 'F':  R = asprintf(&Text, "%.3f", Item->getFloat());   break;
    case 'D':  R = asprintf(&Text, "%.3f", Item->getDouble());   break;
    case 'X':  R = asprintf(&Text, "%lld", Item->getLonglong());   break;
    default: return NULL;
  }

  // Check if aprintf() worked OK
  if (R != -1) return Text;
  else return NULL;
}
