/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  2002 <mailto:tbretz@astro.uni-wuerzburg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MH3
//
// With this histogram you can fill a histogram with up to three
// variables from Mars parameter containers in an eventloop.
//
// In the constructor you can give up to three variables which should be
// filled in the histogram. Dependend on the number of given variables
// (data members) a TH1F, TH2F or TH3F is created.
// Specify the data mamber like the following:
//   "MHillas.fLength"
// Where MHillas is the name of the parameter container in the parameter
// list and fLength is the name of the data member which should be filled
// in the histogram. Assuming that your MHillas container has a different
// name (MyHillas) the name to give would be:
//   "MyHillas.fLength"
//
// The axis binning is retrieved from the parameter list, too. Create a
// MBinning with the name "Binning" plus the name of your MH3 container
// plus the axis name ("X", "Y" or "Z") and add it to the parameter list.
//
// If you want to use a different unit for histogramming use SetScaleX,
// SetScaleY and SetScaleZ.
//
// For example:
//   MH3 myhist("MHillas.fLength");
//   myhist.SetName("MyHist");
//   myhist.SetScaleX(geomcam.GetConvMm2Deg()); //convert length to degree
//   MBinning bins("BinningMyHistX");
//   bins.SetEdges(10, 0, 150);
//   plist.AddToList(&myhist);
//   plist.AddToList(&bins);
//
/////////////////////////////////////////////////////////////////////////////

#include "MH3.h"

#include <TH2.h>
#include <TH3.h>
#include <TPad.h>
#include <TCanvas.h>
#include <TDataMember.h>
#include <TMethodCall.h>

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"

ClassImp(MH3);

// --------------------------------------------------------------------------
//
// Creates an TH1F. memberx is filled into the X-bins. For a more detailed
// description see the class description above.
//
MH3::MH3(const char *memberx) : fDimension(1)
{
    fHist = new TH1F;

    fDataMember[0] = memberx;

    fName  = "MH3";
    fTitle = "Container for a 1D Mars Histogram";

    fHist->SetDirectory(NULL);
    fHist->SetYTitle("Counts");

    fScale[0] = 1;
    fScale[1] = 1;
    fScale[2] = 1;
}

// --------------------------------------------------------------------------
//
// Creates an TH2F. memberx is filled into the X-bins. membery is filled
// into the Y-bins. For a more detailed description see the class
// description above.
//
MH3::MH3(const char *memberx, const char *membery) : fDimension(2)
{
    fHist = new TH2F;

    fDataMember[0] = memberx;
    fDataMember[1] = membery;

    fName  = "MH3";
    fTitle = "Container for a 2D Mars Histogram";

    fHist->SetDirectory(NULL);
    fHist->SetZTitle("Counts");

    fScale[0] = 1;
    fScale[1] = 1;
    fScale[2] = 1;
}

// --------------------------------------------------------------------------
//
// Creates an TH3F. memberx is filled into the X-bins. membery is filled
// into the Y-bins. membery is filled into the Z-bins. For a more detailed
// description see the class description above.
//
MH3::MH3(const char *memberx, const char *membery, const char *memberz) : fDimension(3)
{
    fHist = new TH3F;

    fDataMember[0] = memberx;
    fDataMember[1] = membery;
    fDataMember[2] = memberz;

    fName  = "MH3";
    fTitle = "Container for a 3D Mars Histogram";

    fScale[0] = 1;
    fScale[1] = 1;
    fScale[2] = 1;
}

// --------------------------------------------------------------------------
//
// Deletes the histogram
//
MH3::~MH3()
{
    delete fHist;
}

// --------------------------------------------------------------------------
//
// Trys to determin the TMethodCall corresponding to the num-axis
// corresponding data member.
//
Bool_t MH3::GetMethodCall(const MParList *plist, Int_t num)
{
    TString cname(fDataMember[num]);
    TString mname(fDataMember[num]);

    const char *dot = strrchr(cname, '.');

    if (dot)
    {
        const int pos = dot-cname;

        cname.Remove(pos);
        mname.Remove(0, pos+1);
    }

    fObject[num] = (MParContainer*)plist->FindObject(cname);
    if (!fObject[num])
    {
        *fLog << err << "Object '" << cname << "' not in parameter list... aborting." << endl;
        return kFALSE;
    }

    fMethodCall[num] = fObject[num]->GetterMethod(mname);

    return fMethodCall[num] ? kTRUE : kFALSE;
}


// --------------------------------------------------------------------------
//
// Setup the Binning for the histograms automatically if the correct
// instances of MBinning are found in the parameter list
// For a more detailed description see class description above.
//
Bool_t MH3::SetupFill(const MParList *plist)
{
    TString bname("Binning");
    bname += fName;

    MBinning *binsx = NULL;
    MBinning *binsy = NULL;
    MBinning *binsz = NULL;
    switch (fDimension)
    {
    case 3:
        binsz = (MBinning*)plist->FindObject(bname+"Z");
        if (!binsz)
        {
            *fLog << err << dbginf << "MBinning '" << bname << "X' not found... aborting." << endl;
            return kFALSE;
        }
        fHist->SetZTitle(fName+"Z");
        if (!GetMethodCall(plist, 2))
            return kFALSE;
    case 2:
        binsy = (MBinning*)plist->FindObject(bname+"Y");
        if (!binsy)
        {
            *fLog << err << dbginf << "MBinning '" << bname << "Y' not found... aborting." << endl;
            return kFALSE;
        }
        fHist->SetYTitle(fName+"Y");
        if (!GetMethodCall(plist, 1))
            return kFALSE;
    case 1:
        binsx = (MBinning*)plist->FindObject(bname+"X");
        if (!binsx)
        {
            *fLog << err << dbginf << "MBinning '" << bname << "X' not found... aborting." << endl;
            return kFALSE;
        }
        fHist->SetXTitle(fName+"X");
        if (!GetMethodCall(plist, 0))
            return kFALSE;
    }

    fHist->SetName(fName);

    TString title("Histogram for ");
    title += fName;

    switch (fDimension)
    {
    case 1:
        fHist->SetTitle(title+" (1D)");
        SetBinning(fHist, binsx);
        return kTRUE;
    case 2:
        fHist->SetTitle(title+" (2D)");
        SetBinning(fHist, binsx, binsy);
        return kTRUE;
    case 3:
        fHist->SetTitle(title+" (3D)");
        SetBinning(fHist, binsx, binsy, binsz);
        return kTRUE;
    }

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Set the name of the histogram ant the MH3 container
//
void MH3::SetName(const char *name)
{
    fHist->SetName(name);
    MParContainer::SetName(name);
}

// --------------------------------------------------------------------------
//
// Set the title of the histogram ant the MH3 container
//
void MH3::SetTitle(const char *title)
{
    fHist->SetTitle(title);
    MParContainer::SetTitle(title);
}

// --------------------------------------------------------------------------
//
// Returns the value corresponding to the data member of the given object
//
Bool_t MH3::GetValue(Int_t num, Double_t &v)
{
    switch (fMethodCall[num]->ReturnType())
    {
    case TMethodCall::kLong:
        Long_t l;
        fMethodCall[num]->Execute(fObject[num], l); // FIXME: const, root
        v = fScale[num]*l;
        return kTRUE;

    case TMethodCall::kDouble:
        fMethodCall[num]->Execute(fObject[num], v); // FIXME: const, root
        v *= fScale[num];
        return kTRUE;

    default:
        *fLog << err << "DataMember "  << fDataMember[num] << " of ";
        *fLog << fObject[num]->GetName() << " neither int nor float... abort." << endl;
        return kFALSE;
    }
}

// --------------------------------------------------------------------------
//
// Fills the one, two or three data members into our histogram
//
Bool_t MH3::Fill(const MParContainer *par)
{
    Double_t x, y, z;

    switch (fDimension)
    {
    case 3:
        if (!GetValue(2, z))
            return kFALSE;
    case 2:
        if (!GetValue(1, y))
            return kFALSE;
    case 1:
        if (!GetValue(0, x))
            return kFALSE;
    }

    switch (fDimension)
    {
    case 3:
        ((TH3*)fHist)->Fill(x, y, z);
        return kTRUE;
    case 2:
        ((TH2*)fHist)->Fill(x, y);
        return kTRUE;
    case 1:
        fHist->Fill(x);
        return kTRUE;
    }

    return kFALSE;
}

// --------------------------------------------------------------------------
//
// Draw clone of histogram. So that the object can be deleted
// and the histogram is still visible in the canvas.
// The cloned object are deleted together with the canvas if the canvas is
// destroyed. If you want to handle destroying the canvas you can get a
// pointer to it from this function
//
TObject *MH3::DrawClone(Option_t *opt) const
{
    TCanvas &c = *MH::MakeDefCanvas(fHist);

    *fLog << all << "Created Canvas: '" << fHist->GetName() <<"' '" << fHist->GetTitle() << "'" << endl;

    //
    // This is necessary to get the expected bahviour of DrawClone
    //
    gROOT->SetSelectedPad(NULL);

    fHist->DrawCopy(opt);

    c.Modified();
    c.Update();

    return &c;
}

// --------------------------------------------------------------------------
//
// Creates a new canvas and draws the histogram into it.
// Be careful: The histogram belongs to this object and won't get deleted
// together with the canvas.
//
void MH3::Draw(Option_t *opt)
{
    if (!gPad)
        MH::MakeDefCanvas(fHist);

    fHist->Draw(opt);

    gPad->Modified();
    gPad->Update();
}
