#!/usr/bin/python
#
# Werner Lustermann
# ETH Zurich
#
import numpy as np

from pyfact_rename import RawData
from myhisto import *
from hist    import *

class  baseline_ana( object):
    """
    """
    def __init__(self, rfiles={}, step=500, bsl_corr=True, max_events=-1):
        """
        rfiles   : dictionary of strings : run file names
        step     : integer : number of events to be used for
                   the calculation of a baseline values
        bsl_corr : bool : apply baseline correction
        max_events : maximum number of events to be processed
        
        self:
            blsMeanValues : np.array() to store computed baseline values
        """

        if rfiles == {}:
            print "no file names given for analysis!!!"
            return (-1) 
        else:
            print "baseline analysing data: "
            #rfiles.info()
            for key in rfiles.names.keys():
                print key, rfiles.names[key]

        self.rfiles   = rfiles
        self.step     = step
        self.bsl_corr = bsl_corr

        if bsl_corr:
            print 'apply baseline correction'
        else:
            print 'no baseline correction'

        # create the object providing the rawdata
        self.rd = RawData(rfiles.names['data'], rfiles.names['drscal'])
        self.npix = self.rd.npix
        
        if max_events == -1:
            self.nevents = self.rd.nevents
        else:
            self.nevents = max_events
        
        self.steps() # compute the number of steps

        self.bslMeanValues = np.zeros( ( self.nsteps, self.npix ) )
            
        self.pixhist = hist_array( self.npix, 400, -99.5, 100.5,
                              'bsl_pix', 'Pixel Baseline',
                              'sample value in mV', 'Entries / 0.5 mV' )  
        self.bslSummaryHistos = bslHistograms( 'bsl_sum' )

        self.run()  # compute baseline
        self.save() # save results
        
    def steps( self ):
        """ compute the ranges of events
        
            self:
                nsteps   : integer : number of ranges of events
                evranges : list of range boundaries of length nsteps + 1
        """
            
        nsteps   = self.nevents / self.step + 1
        evranges = range(0, nsteps * self.step, self.step)
            
        # add the events of the last incomplete step to the previous once
        evranges[-1] = int( self.nevents )

        self.nsteps   = nsteps
        self.evranges = evranges

        print 'nevents: ', self.nevents
        print 'ranges:  ', self.evranges


    def loopEvents( self, first, last, hist):
        """ LOOP over all events: Do what you want, here!
        first : integer : first event to process
        last  : integer : last event to process
        hist  : hist_array : fill all pixel data
        """
        for ev in range( first, last ):
            if ev%100 == 0: print 'Event', ev
            self.rd.next_event()
            # rd.CorrectBaseline()
            hist.fillall( self.rd.acal_data )
                
                
    def loopPixel( self, npix, phist ):
        """ loop over pixel and extract most probable values 
        """
        vmaxprob = np.zeros( npix )
        v_rms    = np.zeros( npix )

        for pix in range( npix ):
    
            hpix = phist.list[pix]
    
            vmaxprob[pix] = hpix.GetXaxis().GetBinCenter( hpix.GetMaximumBin() )
            self.bslSummaryHistos.dict['hbsl_mean'].Fill( vmaxprob[pix] )
            self.bslSummaryHistos.dict['hplt_mean'].SetBinContent(pix+1,
                                                             vmaxprob[pix] )
                
            v_rms[pix] = hpix.GetRMS()
            self.bslSummaryHistos.dict['hbsl_rms'].Fill( v_rms[pix] )
            self.bslSummaryHistos.dict['hplt_rms'].SetBinContent( pix+1,
                                                             v_rms[pix] );
        
        self.vmaxprob = vmaxprob


    def run( self ):
        """ run the analysis
        """
        
        for step in range( self.nsteps - 1 ):
                
            print step, 'processing events: ', self.evranges[ step ], 'to ',
            print self.evranges[ step+1 ]
            
            pixstephist = hist_array( self.npix, 400, -99.5, 100.5,
                                      'bsl_pix_step', 'Pixel Baseline',
                                      'sample value in mV', 'Entries / 0.5 mV' )
                
            self.loopEvents( self.evranges[step], self.evranges[step+1],
                             pixstephist )

            self.pixhist.y[:,:] += pixstephist.y[:,:] 
            pixstephist.SetBinContent()
            
            bslSummaryHistos = bslHistograms( 'bsl_sum_step' )
            self.bslMeanValues[step,:] = self.loopPixel( self.npix, pixstephist )
            # print self.bslMeanValues

            fname = 'h'+ str( self.evranges[step] ) + '_to_'+ str( self.evranges[step+1] ) + '.root'
            SaveHistograms( [pixstephist, bslSummaryHistos], fname )
    
            del pixstephist
            del bslSummaryHistos

        self.pixhist.SetBinContent()

        self.loopPixel( self.rd.npix, self.pixhist )


    def save( self ):
        """ save baseline values and histograms
        """
        SaveHistograms( [ self.pixhist, self.bslSummaryHistos],
                               self.rfiles.names['results'] + '_bsl.root' )
        
        np.savetxt( self.rfiles.names['results'] + '_bls.txt', self.bslMeanValues,
                    fmt = '%8.3f' )
        np.save( self.rfiles.names['results'] + '_bsl.npy', self.bslMeanValues )
        

if __name__ == '__main__':
    """ create an instance for test purposes

    """
    runfiles = pyfact.fnames(zipped = False)

    runfiles.info()

    bcal = baseline_ana( rfiles = runfiles, step = 14,
                         bsl_corr = False) 
