/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Harald Kornmayer 1/2001
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

///////////////////////////////////////////////////////////////////////
//
// MHFadcPix
//
// This container stores a hostogram to display an Fadc Spekrtum.
// The spektrum of all the values measured by the Fadcs.
//
///////////////////////////////////////////////////////////////////////

#include "MHFadcPix.h"

#include <TH1.h>
#include <TPad.h>

#include "MH.h"

ClassImp(MHFadcPix);

// --------------------------------------------------------------------------
//
// Creates the histograms for lo and hi gain of one pixel
//
MHFadcPix::MHFadcPix(UInt_t pixid) : fPixId(pixid)
{
    fHistHi =  new TH1F(pixid ? Form("HiGain%03d", pixid) : "HiGain",
                        pixid ? Form("Hi Gain Pixel #%d", pixid) : "Hi Gain Samples",
                        256, -.5, 255.5);

    fHistHi->SetDirectory(NULL);
    fHistHi->SetXTitle("Signal/FADC Units");
    fHistHi->SetYTitle("Count");

    fHistLo = new TH1F(pixid ? Form("LoGain%03d", pixid) : "LoGain",
                       pixid ? Form("Lo Gain Pixel #%d", pixid) : "Lo Gain Samples",
                       256, -.5, 255.5);

    fHistLo->SetDirectory(NULL);
    fHistLo->SetXTitle("Signal/FADC Units");
    fHistLo->SetYTitle("Count");
}

// --------------------------------------------------------------------------
MHFadcPix::~MHFadcPix()
{
    delete fHistHi;
    delete fHistLo;
}

// --------------------------------------------------------------------------
void MHFadcPix::FillHi(Byte_t i)
{
    fHistHi->Fill(i);
}

// --------------------------------------------------------------------------
void MHFadcPix::FillLo(Byte_t i)
{
    fHistLo->Fill(i);
}

// --------------------------------------------------------------------------
void MHFadcPix::DrawHi()
{
    fHistHi->Draw();
}

// --------------------------------------------------------------------------
void MHFadcPix::DrawLo()
{
    fHistLo->Draw();
}

// --------------------------------------------------------------------------
//
// We need our own clone function to get rid of the histogram in any
// directory
//
TObject *MHFadcPix::Clone(const char *) const
{
    MHFadcPix &pix = *(MHFadcPix*)TObject::Clone();

    pix.fHistHi->SetDirectory(NULL);
    pix.fHistLo->SetDirectory(NULL);

    return &pix;
}

// --------------------------------------------------------------------------
void MHFadcPix::Draw(Option_t *)
{
    if (!gPad)
    {
        const char *name  = StrDup(fPixId ? Form("Pixel #%d", fPixId) : "Pixel");
        MH::MakeDefCanvas(name, fPixId ? Form("%s FADC Samples", name) : "FADC Samples");
        delete [] name;
    }

    gPad->Divide(1, 2);

    gPad->cd(1);
    fHistHi->Draw();

    gPad->cd(2);
    fHistLo->Draw();
}
