// Reading data from FSC over Ethernet

#include <stdio.h>
#include <string.h>
#include <sys/socket.h>
#include <netdb.h>
#include <math.h>

#include "Evidence.h"
 
#define SERVER_NAME "FSCctrl"	// Name to use in DIM

const unsigned int MAX_SIZE = 10000;

const unsigned int TIME_OFF = 3;
const unsigned int TEMP_NUM = 64;
const unsigned int TEMP_OFF = 134;
const unsigned int VOLT_NUM = 40;
const unsigned int VOLT_OFF = 30;
const unsigned int CURR_NUM = 40;
const unsigned int CURR_OFF = 70;
const unsigned int HUMI_NUM = 4;
const unsigned int HUMI_OFF = 110;

const unsigned int MAX_VAL = TEMP_OFF + TEMP_NUM +100;

// Parameters for PT1000
const float R0 = 1000, A = 3.9083e-3, B = -5.775e-7;

using namespace std;

//
// Main program
//
int main(int argc, char *argv[]) {

  int Result;
  char Buffer[1000];
  float Val[MAX_VAL]={0};
  string Data;
  int Socket;
  struct sockaddr_in SocketAddress;
  vector<string> Items;
  time_t Time;

  // Start Evidence server
  EvidenceServer S(SERVER_NAME);
  
  // Get configuration data
  string Address = S.GetConfig("Address", "10.0.128.127");
  int Port = atoi(S.GetConfig("Port", "5000").c_str());

  if (argc > 1) Address = argv[1];
  if (argc > 2) Port = atoi(argv[2]);
   
  // Resolve hostname
  struct hostent *Host = gethostbyname(Address.c_str());
  if (Host == 0) S.Message(S.FATAL, "Could not resolve host name '%s'", Address.c_str());

  SocketAddress.sin_family = PF_INET;
  SocketAddress.sin_port = htons(Port);
  SocketAddress.sin_addr = *(struct in_addr*) Host->h_addr;

  // Open socket descriptor
  if ((Socket = socket(PF_INET, SOCK_STREAM, 0)) == -1) {
    S.Message(S.FATAL, "Could not open socket (%s)\n", strerror(errno));
  }
    
  // Connect to server
  if (connect(Socket, (struct sockaddr *) &SocketAddress, sizeof(SocketAddress)) == -1) {
    S.Message(S.FATAL, "Could not connect to %s:%hu (%s)", Address.c_str(), Port, strerror(errno));
  }

  // Create DIM services
  DimService Update(SERVER_NAME"/Time", "F", Val+TIME_OFF, sizeof(float));
  DimService Temperature(SERVER_NAME"/Temperature", "F", Val+TEMP_OFF, sizeof(float)*TEMP_NUM);
  DimService Humidity(SERVER_NAME"/Humindity", "F", Val+HUMI_OFF, sizeof(float)*HUMI_NUM);
  DimService Voltage(SERVER_NAME"/Voltage", "F",Val+VOLT_OFF, sizeof(float)*VOLT_NUM);
  DimService Current(SERVER_NAME"/Current", "F", Val+CURR_OFF, sizeof(float)*CURR_NUM);

  // Send start command to FSC board
  snprintf(Buffer, sizeof(Buffer), "m\n");
  Result = write(Socket, Buffer, 1);
  if (Result != 1) S.Message(S.ERROR, "Could not write to socket (%s)", strerror(errno));
	  
  // Loop receiving data from FSC board
  while (!S.ExitRequest) {

    Result = read(Socket, Buffer, sizeof(Buffer));

	if (Result == -1) {
	  S.Message(S.ERROR, "Could not read from socket, exiting (%s)", strerror(errno));
	  break;
	}
	else if (Result == 0) {
	  S.Message(S.INFO, "Server not existing anymore, exiting");
	  break;
	}

	// Append received data and check for maxumum length and end marker
	Data.append(Buffer, Result);

	if (Data.size() > MAX_SIZE) {
	  S.Message(S.ERROR, "Received more than %u characters without end marker, deleting", MAX_SIZE);
	  Data.clear();
	}

	if (Data.find("end.\n")==string::npos || Data.find("status:")==string::npos) continue;
	
	Data = Data.substr(Data.rfind("status:"), Data.rfind("end.\n")-Data.rfind("status:"));

	//
	// Full event received
	//
	Time = time(NULL);
	printf("%s%s\n", ctime(&Time), Data.c_str());

	// Extract numbers
	Items = S.Tokenize(Data, " \n\t:");
	
	for (unsigned int i=0; i<MAX_VAL && i<Items.size(); i++) {
	  Val[i] = atof(Items[i].c_str());
	  //printf("Value %i: %s / %.2f  \n", i, Items[i].c_str(), Val[i]);
	}

	// Translate resistance to temperatures
	for (unsigned int i=0; i<TEMP_NUM; i++) {	
	  Val[i+TEMP_OFF] = -A/(2*B)-sqrt(pow(A/(2*B),2)-(R0-Val[i+TEMP_OFF])/(B*R0));
	}
	
	// Update DIM service
	Temperature.updateService();
	Humidity.updateService();
	Voltage.updateService();
	Current.updateService();
	Update.updateService();
	
	Data.clear();
  } // while()
  
  if (close(Socket) == -1) {
	S.Message(S.ERROR, "Could not close socket descriptor (%s)", strerror(errno));  
  }

}
