/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Wolfgang Wittek 4/2002 <mailto:wittek@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

//////////////////////////////////////////////////////////////////////////////
//                                                                          //
//  MHMcEnergyMigration                                                     //
//                                                                          //
//  calculates the migration matrix E-est vs. E-true                        //
//  for different bins in Theta                                             //
//                                                                          //
//////////////////////////////////////////////////////////////////////////////

#include "MHMcEnergyMigration.h"

#include <TCanvas.h>

#include "MMcEvt.hxx"

#include "MEnergyEst.h"
#include "MBinning.h"
#include "MHillasSrc.h"
#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

ClassImp(MHMcEnergyMigration);


// --------------------------------------------------------------------------
//
// Default Constructor. It sets name and title of the histogram.
//
MHMcEnergyMigration::MHMcEnergyMigration(const char *name, const char *title)
    : fHist()
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MHMcEnergyMigration";
    fTitle = title ? title : "3-D histogram   E-true E-est Theta";

    fHist.SetDirectory(NULL);

    fHist.SetTitle("3D-plot   E-true E-est Theta");
    fHist.SetXTitle("E-true [GeV]            ");
    fHist.SetYTitle("E-est [GeV]            ");
    fHist.SetZTitle("\\Theta [\\circ]");
}

// --------------------------------------------------------------------------
//
// Set the binnings and prepare the filling of the histograms
//
Bool_t MHMcEnergyMigration::SetupFill(const MParList *plist)
{
    fEnergy = (MEnergyEst*)plist->FindObject("MEnergyEst");
    if (!fEnergy)
    {
        *fLog << err << dbginf << "MEnergyEst not found... aborting." << endl;
        return kFALSE;
    }

    fMcEvt = (MMcEvt*)plist->FindObject("MMcEvt");
    if (!fMcEvt)
    {
        *fLog << err << dbginf << "MMcEvt not found... aborting." << endl;
        return kFALSE;
    }

    const MBinning* binsenergy = (MBinning*)plist->FindObject("BinningE");
    const MBinning* binstheta  = (MBinning*)plist->FindObject("BinningTheta");
    if (!binsenergy || !binstheta)
    {
        *fLog << err << dbginf << "At least one MBinning not found... aborting." << endl;
        return kFALSE;
    }

    SetBinning(&fHist, binsenergy, binsenergy, binstheta);

    fHist.Sumw2();

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Draw a copy of the histogram
//
TObject *MHMcEnergyMigration::DrawClone(Option_t *opt) const
{
    TCanvas &c = *MakeDefCanvas("EnergyMigration", "E-est vs. E-true");

    c.Divide(2, 2);

    gROOT->SetSelectedPad(NULL);

    TH1 *h;

    c.cd(1);
    h = ((TH3*)(&fHist))->Project3D("expro");

    h->SetTitle("Distribution of E-true");
    h->SetXTitle("E-true [GeV]            ");
    h->SetYTitle("Counts");

    h->Draw(opt);
    h->SetBit(kCanDelete);
    gPad->SetLogx();


    c.cd(2);
    h = ((TH3*)(&fHist))->Project3D("eypro");

    h->SetTitle("Distribution of E-est");
    h->SetXTitle("E-est [GeV]            ");
    h->SetYTitle("Counts");

    h->Draw(opt);
    h->SetBit(kCanDelete);
    gPad->SetLogx();

    c.cd(3);
    h = ((TH3*)(&fHist))->Project3D("ezpro");

    h->SetTitle("Distribution of Theta");
    h->SetXTitle("\\Theta [\\circ]");
    h->SetYTitle("Counts");

    h->Draw(opt);
    h->SetBit(kCanDelete);


    c.cd(4);
    ((TH3*)(&fHist))->DrawCopy(opt);

    c.Modified();
    c.Update();

    return &c;
}

// --------------------------------------------------------------------------
//
// Draw the histogram
//
void MHMcEnergyMigration::Draw(Option_t *opt)
{
    if (!gPad)
        MakeDefCanvas("EnergyMigration", "E-est vs. E-true");

    gPad->Divide(2,2);
    
    TH1 *h;

    gPad->cd(1);
    h = fHist.Project3D("ex_pro");

    h->SetTitle("Distribution of E-true");
    h->SetXTitle("E-true [GeV]            ");

    h->Draw(opt);
    h->SetBit(kCanDelete);
    gPad->SetLogx();


    gPad->cd(2);
    h = fHist.Project3D("ey_pro");

    h->SetTitle("Distribution of E-est");
    h->SetXTitle("E-est [GeV]            ");
    h->Draw(opt);
    h->SetBit(kCanDelete);
    gPad->SetLogx();

    gPad->cd(3);
    h = fHist.Project3D("ez_pro");

    h->SetTitle("Distribution of Theta");
    h->SetXTitle("\\Theta [\\circ]");
    h->Draw(opt);
    h->SetBit(kCanDelete);


    gPad->cd(4);
    fHist.DrawCopy(opt);

    gPad->Modified();
    gPad->Update();
}

// --------------------------------------------------------------------------
//
// Fill the histogram
//
Bool_t MHMcEnergyMigration::Fill(const MParContainer *par)
{
    //    MHillasSrc &hil = *(MHillasSrc*)par;
    //    fHist.Fill(fMcEvt->GetEnergy(), hil.GetSize());

    // get E-true from fMcEvt and E-est from fEnergy
    fHist.Fill(fMcEvt->GetEnergy(), fEnergy->GetEnergy(), fMcEvt->GetTheta()*kRad2Deg);

    return kTRUE;
}
