/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Harald Kornmayer 1/2001
!
!   Copyright: MAGIC Software Development, 2000-2001
!
!
\* ======================================================================== */

#include "MMcCollectionAreaCalc.h"

#include "MParList.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MMcEvt.hxx"
#include "MMcTrig.hxx"
#include "MMcRunHeader.hxx"

#include "MHMcCollectionArea.h"

ClassImp(MMcCollectionAreaCalc);

MMcCollectionAreaCalc::MMcCollectionAreaCalc(const char *input,
                                             const char *name, const char *title)
{
    fName  = name  ? name  : "MMcCollectionAreaCalc";
    fTitle = title ? title : "Task to calculate the collection area";

    fObjName = input ? input : "MMcTrig";
    AddToBranchList(Form("%s.fNumFirstLevel", input));

    AddToBranchList("MMcEvt.fEnergy");
    AddToBranchList("MMcEvt.fImpact");
} 

Bool_t MMcCollectionAreaCalc::PreProcess (MParList *pList)
{
    // connect the raw data with this task

    fMcEvt = (MMcEvt*)pList->FindObject("MMcEvt");
    if (!fMcEvt)
    {
        *fLog << err << dbginf << "MMcEvt not found... exit." << endl;
        return kFALSE;
    }

    fMcTrig = (MMcTrig*)pList->FindObject(fObjName);
    if (!fMcTrig)
    {
        *fLog << err << dbginf << fObjName << " [MMcTrig] not found... exit." << endl;
        return kFALSE;
    }

    fCollArea = (MHMcCollectionArea*)pList->FindCreateObj("MHMcCollectionArea");
    if (!fCollArea)
        return kFALSE;


    fTotalNumSimulatedShowers = 0;
    fCorsikaVersion           = 0;
    fAllEvtsTriggered         = kFALSE;

    return kTRUE;
}

Bool_t MMcCollectionAreaCalc::ReInit(MParList *plist)
{
    MMcRunHeader *runheader = (MMcRunHeader*)plist->FindObject("MMcRunHeader");
    if (!runheader)
    {
        *fLog << err << dbginf << "Error - MMcRunHeader not found... exit." << endl;
        return kFALSE;
    }

    fTotalNumSimulatedShowers += runheader->GetNumSimulatedShowers();

    *fLog << inf << "Total Number of Simulated showers: " << fTotalNumSimulatedShowers << endl;

    if (fTheta>=0 && fTheta!=runheader->GetTelesTheta())
        *fLog << warn << dbginf << "Warning - Read files have different TelesTheta... exit." << endl;

    fTheta = runheader->GetTelesTheta();

    if (fCorsikaVersion!=0 && fCorsikaVersion!=runheader->GetCorsikaVersion())
        *fLog << warn << dbginf << "Warning - Read files have different Corsika versions... exit." << endl;

    fCorsikaVersion = runheader->GetCorsikaVersion();

    fAllEvtsTriggered |= runheader->GetAllEvtsTriggered();

    *fLog << inf << "Only triggered events avail: " << (fAllEvtsTriggered?"yes":"no") << endl;

    return kTRUE;
}

Bool_t MMcCollectionAreaCalc::Process()
{ 
    const Float_t energy = fMcEvt->GetEnergy();
    const Float_t impact = fMcEvt->GetImpact()/100.;

    if (!fAllEvtsTriggered)
        fCollArea->FillAll(energy, impact);

    if (fMcTrig->GetFirstLevel() <= 0)
        return kTRUE;

    fCollArea->FillSel(energy, impact);

    return kTRUE;
}

Bool_t MMcCollectionAreaCalc::PostProcess()
{ 
    //
    //   do the calculation of the effectiv area
    //
    *fLog << inf << "Calculation Collection Area..." << endl;

    if (!fAllEvtsTriggered)
        fCollArea->CalcEfficiency();
    else
    {
        *fLog << inf << "Total number of showers: " << fTotalNumSimulatedShowers << endl;
        fCollArea->CalcEfficiency(fTotalNumSimulatedShowers,
                                  fCorsikaVersion == 5200 ? fTheta : 0);
    }

    return kTRUE;
}

