/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz  12/2000 <mailto:tbretz@uni-sw.gwdg.de>
!   Author(s): Harald Kornmayer 1/2001
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

///////////////////////////////////////////////////////////////////////
//
// MGeomCam
//
// This is the base class of different camera geometries. It creates
// a pixel object for a given number of pixels and defines the
// interface of how to acccess the geometry information.
//
///////////////////////////////////////////////////////////////////////

#include "MGeomCam.h"

#include "MLog.h"
#include "MLogManip.h"

#include "MGeomPix.h"
#include "MHexagon.h"

ClassImp(MGeomCam);

// --------------------------------------------------------------------------
//
// Initializes a Camera Geometry with npix pixels. All pixels
// are deleted when the corresponding array is deleted.
//
MGeomCam::MGeomCam(UInt_t npix, Float_t dist, const char *name, const char *title)
    : fNumPixels(npix), fCamDist(dist), fMm2Deg(kRad2Deg/(dist*1000))
{
    fName  = name  ? name  : "MGeomCam";
    fTitle = title ? title : "Storage container for  a camera geometry";

    fPixels = new TObjArray(npix);

    //
    // make sure that the destructor delete all contained objects
    //
    fPixels->SetOwner();

    for (UInt_t i=0; i<npix; i++)
        (*fPixels)[i] = new MGeomPix;

    SetReadyToSave();
}

// --------------------------------------------------------------------------
//
//  Destructor delete the pixel array with all pixels
//
MGeomCam::~MGeomCam()
{
    delete fPixels;
}

// --------------------------------------------------------------------------
//
// Calculate the maximum radius of the camera. This is ment for GUI layout.
//
void MGeomCam::CalcMaxRadius()
{
    fMaxRadius = 0;

    for (UInt_t i=0; i<fNumPixels; i++)
    {
        const MGeomPix &pix = (*this)[i];

        const Float_t x = pix.GetX();
        const Float_t y = pix.GetY();
        const Float_t r = pix.GetR();

        const Float_t maxr = sqrt(x*x+y*y) + r;

        if (maxr>fMaxRadius)
            fMaxRadius = maxr;
    }
}

// --------------------------------------------------------------------------
//
//  Prints the Geometry information of all pixels in the camera
//
void MGeomCam::Print(Option_t *) const
{
    //
    //   Print Information about the Geometry of the camera
    //
    *fLog << all << " Number of Pixels (" << GetTitle() << "): " << fNumPixels << endl;

    for (UInt_t i=0; i<fNumPixels; i++)
    {
        *fLog << " Pixel: " << i << "  ";
        (*this)[i].Print();
    }
} 

