/* ======================================================================== *\
!
! *
! * This file is part of MARS, the MAGIC Analysis and Reconstruction
! * Software. It is distributed to you in the hope that it can be a useful
! * and timesaving tool in analysing Data of imaging Cerenkov telescopes.
! * It is distributed WITHOUT ANY WARRANTY.
! *
! * Permission to use, copy, modify and distribute this software and its
! * documentation for any purpose is hereby granted without fee,
! * provided that the above copyright notice appear in all copies and
! * that both that copyright notice and this permission notice appear
! * in supporting documentation. It is provided "as is" without express
! * or implied warranty.
! *
!
!
!   Author(s): Thomas Bretz, 5/2002 <mailto:tbretz@astro.uni-wuerzburg.de>
!   Author(s): Rudy Bock, 5/2002 <mailto:rkb@mppmu.mpg.de>
!
!   Copyright: MAGIC Software Development, 2000-2002
!
!
\* ======================================================================== */

/////////////////////////////////////////////////////////////////////////////
//
// MMultiDimDistCalc
//
// Calculated a multidimensional distance. It calculates the distance to
// all vectors in a given matrix describing Gammas and another one
// describing Hadrons (non gammas). The shortest distances are avaraged.
// How many distances are used for avaraging can be specified in the
// constructor.
//
////////////////////////////////////////////////////////////////////////////
#include "MMultiDimDistCalc.h"

#include "MHMatrix.h" // must be before MLogManip.h

#include "MLog.h"
#include "MLogManip.h"

#include "MParList.h"
#include "MDataChain.h"

#include "MHadroness.h"

ClassImp(MMultiDimDistCalc);

// --------------------------------------------------------------------------
//
// Setup histograms and the number of distances which are used for
// avaraging in CalcDist
//
MMultiDimDistCalc::MMultiDimDistCalc(Int_t num, const char *name, const char *title)
    : fNum(num)
{
    //
    //   set the name and title of this object
    //
    fName  = name  ? name  : "MMultiDimDistCalc";
    fTitle = title ? title : "Composite Probabilities Loop 1/2";

    fData = new TList;
    fData->SetOwner();
}

// --------------------------------------------------------------------------
//
// Delete the data chains
//
MMultiDimDistCalc::~MMultiDimDistCalc()
{
    delete fData;
}

// --------------------------------------------------------------------------
//
// Needs:
//  - MatrixGammas  [MHMatrix]
//  - MatrixHadrons {MHMatrix]
//  - MHadroness
//  - all data containers used to build the matrixes
//
// The matrix object can be filles using MFillH. And must be of the same
// number of columns (with the same meaning).
//
Bool_t MMultiDimDistCalc::PreProcess(MParList *plist)
{
    fMGammas = (MHMatrix*)plist->FindObject("MatrixGammas", "MHMatrix");
    if (!fMGammas)
    {
        *fLog << err << dbginf << "MatrixGammas [MHMatrix] not found... aborting." << endl;
        return kFALSE;
    }

    fMHadrons = (MHMatrix*)plist->FindObject("MatrixHadrons", "MHMatrix");
    if (!fMHadrons)
    {
        *fLog << err << dbginf << "MatrixHadrons [MHMatrix] not found... aborting." << endl;
        return kFALSE;
    }

    if (fMGammas->GetM().GetNcols() != fMHadrons->GetM().GetNcols())
    {
        *fLog << err << dbginf << "Error matrices have different numbers of columns... aborting." << endl;
        return kFALSE;
    }

    TIter Next(fMGammas->GetRules());
    TObject *data=NULL;
    while ((data=Next()))
    {
        MDataChain *chain = new MDataChain(data->GetName());
        if (!chain->PreProcess(plist))
        {
            delete chain;
            return kFALSE;
        }
        fData->Add(chain);
    }

    fHadroness = (MHadroness*)plist->FindCreateObj("MHadroness");
    if (!fHadroness)
        return kFALSE;

    return kTRUE;
}

// --------------------------------------------------------------------------
//
// Evaluates the avarage over the fNum shortest distances in a
// multidimensional space to a matrix (set of vectors) describing
// gammas and hadrons.
// The hadroness of the event is defines as the avarage distance to the
// set of gammas (dg) divided by the avarage distance to the set of
// hadrons (dh). Because this value is in teh range [0, inf] it is
// transformed into [0,1] by:
//  H = exp(-dh/dg);
//
Bool_t MMultiDimDistCalc::Process()
{
    const Double_t ncols = fMGammas->GetM().GetNcols();

    TVector event(ncols);

    Int_t n=0;
    TIter Next(fData);
    MData *data = NULL;
    while ((data=(MData*)Next()))
        event(n++) = data->GetValue();

    Double_t dg = fMGammas->CalcDist(event, fNum);
    Double_t dh = fMHadrons->CalcDist(event, fNum);

    fHadroness->SetHadroness(exp(-dh/dg));

    return kTRUE;
}

